package com.humuson.tas.sender.client;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.humuson.tas.sender.model.push.PushReport;

import lombok.extern.slf4j.Slf4j;

/**
 * 
 * @author hyogun
 *
 */
@Slf4j
public class PushResultConsumer implements Closeable, Runnable {
	private static final String APP_PUSH_REPORT_TOPIC = "prod-report-push";
	private static final String DEFAULT_GROUP_ID = "tas-sender-client-01";
	private Consumer<String, String> consumer;
	private final ObjectMapper mapper;
	
	private PushReportListener listener;
	private final String topic;
	private final String groupId;
	
	
	public PushResultConsumer() {
		this(1, APP_PUSH_REPORT_TOPIC, DEFAULT_GROUP_ID);
	}
	
	public PushResultConsumer(int threadNumber, String topic, String groupId) {
		this.topic = topic;
		this.groupId = groupId;
		mapper = new ObjectMapper();
		log.info("Push result consumer init [topic:{}, groupId:{}", topic, groupId);
	}
	
	public void setReportListener(PushReportListener listener) {
		this.listener = listener;
	}
	
	public void run() {
		
		consumer = KafkaConnectionFactory.createConsumer(this.topic, this.groupId);
		
		while (true) {
			final ConsumerRecords<String, String> records = consumer.poll(1000);
		
			if (records.count() == 0) {
				continue;
			}
			
			try {
				process(records);
			} finally {
				consumer.commitAsync();
			}
		}
	}
	
	public void process(ConsumerRecords<String, String> list) {
		
		List<PushReport> reportList = new ArrayList<PushReport>();
		
		for (ConsumerRecord<String, String> record : list) {
			if (log.isDebugEnabled()) {
				log.debug("partition:{}, offset:{}, value:{}",
						record.partition(), record.offset(), record.value());
			}
			
			try {
				final PushReport report = mapper.readValue(record.value(), PushReport.class);
				reportList.add(report);
			} catch (Exception e) {
				log.error("consumer parse error :[{}]", record.value(), e);
			}
		}
		
		if (this.listener != null) {
			this.listener.result(reportList);
		}
	}

	@Override
	public void close() throws IOException {
		this.consumer.close();
	}
	
	
	public void shutdown() {
	    consumer.wakeup();
	}
}
