package com.humuson.tas.sender.client;

import java.util.concurrent.TimeUnit;

import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.humuson.tas.sender.model.push.AppPushPayload;
import com.humuson.tas.sender.model.push.WebPushPayload;

import lombok.extern.slf4j.Slf4j;

@Slf4j
public class AppPushSendService implements KafkaService<AppPushPayload> {
	
	private static final String DEV_APPPUSH_TOPIC = "dev-app-push-request";
	private static final String PROD_APPPUSH_TOPIC = "prod-app-push-request";
	private static final long DEFAULT_TIMEOUT_MILLISECONDS = 5000;
	private Producer<String, String> pushProducer;
	private ObjectMapper mapper;
	
	@Override
	public void init() {
		pushProducer = KafkaConnectionFactory.getProducer();
		mapper = new ObjectMapper();
	}
	
	@Override
	public void send(final AppPushPayload payload) throws JsonProcessingException {
		this.send(payload, true);
	}
	
	@Override
	public void send(final AppPushPayload payload, boolean isProduct) throws JsonProcessingException {
		final String data = this.mapper.writeValueAsString(payload);
		final String topic = isProduct ? PROD_APPPUSH_TOPIC: DEV_APPPUSH_TOPIC;
		this.send(topic, data);
	}
	
	@Override
	public void send(final String topic, final AppPushPayload payload) throws JsonProcessingException {
		final String data = this.mapper.writeValueAsString(payload);
		this.send(topic, data);
	}

	@Override
	public void send(final String topic, final String data) {
		pushProducer.send(new ProducerRecord<String, String>(topic, data), new Callback() {
			@Override
			public void onCompletion(RecordMetadata metadata, Exception exception) {
				if (exception != null) {
					log.error("send error : {}", data);
		        }
			}
		});
	}
	
	@Override
	public boolean sendSync(AppPushPayload payload) throws Exception {
		return this.sendSync(payload, true);
	}

	@Override
	public boolean sendSync(AppPushPayload payload, boolean isProduct) throws Exception{
		return this.sendSync(payload, isProduct, DEFAULT_TIMEOUT_MILLISECONDS);
	}
	
	@Override
	public boolean sendSync(final String topic , final AppPushPayload payload) throws Exception{
		return this.sendSync(topic, payload, DEFAULT_TIMEOUT_MILLISECONDS);
	}

	@Override
	public boolean sendSync(AppPushPayload payload, boolean isProduct, long timeout) throws Exception{
		final String topic = isProduct? PROD_APPPUSH_TOPIC : DEV_APPPUSH_TOPIC;
		return this.sendSync(topic, payload, timeout);
	}

	@Override
	public boolean sendSync(String topic, AppPushPayload payload, long timeout)
			throws Exception {
		try {
			final String data = this.mapper.writeValueAsString(payload);
			RecordMetadata recodeMetadata = pushProducer.send(new ProducerRecord<String, String>(topic, data)).get(DEFAULT_TIMEOUT_MILLISECONDS, TimeUnit.MILLISECONDS);
			if (log.isDebugEnabled()) {
				log.debug("topic:{}, partition:{}, checksum:{}", recodeMetadata.topic(), recodeMetadata.partition(), recodeMetadata.checksum());
			}
			return true;
		} catch (Exception e) {
			log.error("send kafka error", e);
			throw e;
		}
	}
	
	@Override
	public void close() {
		if (pushProducer != null) {
			try {
				pushProducer.close();
			} catch (Exception e) {
				log.error("kafka producer close error", e);
			}
		}
	}
	
	public static void main(String[] args) {
		String json = "{\"grpId\":0,\"CUST_ID\":\"test\",\"DEVICE\":{\"ID\":2849289,\"APP_ID\":302,\"TOKEN\":\"dthN8XlAWLI:APA91bFqhDyfrqrzoSMUe0kxD3DMof8ojYVDY8FnNWm6SI_X68Hau30V6RO4_p8JE_9w0nF99pPcVahmfqUsMf327yNPtlYbqCET9VRy0_5fPNzoTlhyKv833BcLzThZiHUIQHM6Ty5J\",\"NOTI_FLAG\":\"Y\",\"PLATFORM\":\"ANDROID\"},\"APP_GRP_KEY\":\"483f102fe98e462ebd1e\",\"TITLE\":\"TITLE11112\",\"MESSAGE\":\"MESSAGE111112\",\"PUSH_KEY\":\"l\",\"PUSH_VALUE\":\"\",\"TTL\":0,\"REQUEST_UID\":\"TEST1113123132131\",\"REPORT_QUEUE\":\"dev-app-push-report\",\"MSG_ID\":\"117481\",\"MSG_TYPE\":\"T\"}";
		
		AppPushSendService s = new AppPushSendService();
		s.init();
		ObjectMapper mapper = new ObjectMapper();
		AppPushPayload payload;
		
		try {
			payload = mapper.readValue(json, AppPushPayload.class);
			payload.setPushKey("l");
			System.out.println("payload :"+payload.getCustId());
			s.sendSync("dev-app-push-request", payload);
			
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			if (s != null) s.close();
		}
	}

}
