package com.humuson.tas.sender.client;

import java.io.IOException;

import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.humuson.tas.sender.model.push.WebPushPayload;

import lombok.extern.slf4j.Slf4j;

@Slf4j
public class WebPushSendService implements KafkaService<WebPushPayload>{
	private static final String DEV_WEBPUSH_TOPIC = "dev-web-push-request";
	private static final String PROD_WEBPUSH_TOPIC = "prod-web-push-request";
	private Producer<String, String> pushProducer;
	private ObjectMapper mapper;
	
	@Override
	public void init() {
		pushProducer = KafkaProducerFactory.getProducer();
		mapper = new ObjectMapper();
	}
	
	@Override
	public void send(final WebPushPayload payload) throws JsonProcessingException {
		this.send(payload, true);
	}
	
	@Override
	public void send(final WebPushPayload payload, boolean isProduct) throws JsonProcessingException {
		final String data = this.mapper.writeValueAsString(payload);
		final String topic = isProduct? PROD_WEBPUSH_TOPIC : DEV_WEBPUSH_TOPIC;
		this.send(topic, data);
	}
	
	@Override
	public void send(final String topic, final WebPushPayload payload) throws JsonProcessingException {
		final String data = this.mapper.writeValueAsString(payload);
		this.send(topic, data);
	}

	@Override
	public void send(final String topic, final String data) {
		pushProducer.send(new ProducerRecord<String, String>(topic, data), new Callback() {

			@Override
			public void onCompletion(RecordMetadata metadata, Exception exception) {
				if (exception != null) {
					log.error("send error :{}", data);
		        }
				log.info("send complete:[topic:{}, data:{}]", topic, data); 
			}
		});
	}

	@Override
	public void close() {
		if (pushProducer != null) {
			try {
				pushProducer.close();
			} catch (Exception e) {
				log.error("kafka producer close error", e);
			}
		}
	}
	
	public static void main(String[] args) {
		WebPushSendService s = new WebPushSendService();
		s.init();
		String json = "{\"APP_GRP_KEY\":\"ed3c1123-2e7f-4616-b9af-f1509962aefb\",\"UUID\":\"8eac8f3c-527f-4443-b249-dc78d02aa041\",\"TITLE\":\"TAS 웹푸시 이벤트\",\"MESSAGE\":\"차효석 고객님! 모바일 앱 없이 푸시를 발송할 수 있는 TAS 웹푸시를 체험하셨습니다. 이벤트 상품 확인하세요!\",\"PUSH_IMG_URL\":\"https://img-mkt.tason.com/postman2011/event/seminar_180313/gift_new.png\",\"TTL\":0,\"REQUEST_UID\":\"140524|195|166|953|8eac8f3c-527f-4443-b249-dc78d02aa041|8eac8f3c-527f-4443-b249-dc78d02aa041\",\"REPORT_QUEUE\":\"dev-web-push-report\",\"LANDING_URL\":\"https://dev.tason.com/event/webpush/thanks\"}";
		String json2 = "{\"APP_GRP_KEY\": \"8772750e-176a-4d81-af2a-364195aea294\","
//				+ " \"CUST_ID\":\"8a399a63-4e39-42b1-b29f-4e198ea99374\"," // -- chrome mobile
				+ " \"CUST_ID\":\"1061c5c3-00c5-41fd-b10d-fb0d0209d1fd\"," //-- PC
//				+ " \"CUST_ID\":\"89356465-2385-4238-b391-a57406b93417\"," // samsung internet browser 
				+ " \"ICON_URL\":\"https://img-amc.tason.com/resources/renewal/images/tas-favicon.ico\","
				+ " \"TITLE\":\"push test\","
				+ " \"PUSH_IMG_URL\":\"https://media.shilladfs.com/medias/sys_master/images/h4d/hff/11758417739806.jpg\","
				+ " \"MESSAGE\":\"push msg test\", "
				+ " \"LANDING_URL\": \"https://www.tason.com\","
				+ " \"REQUEST_UID\":\"1234567890\"}";
		
//		0117ad10-f7ec-4fdb-8bd0-12dc2c97af36
//		String json = "{\"APP_GRP_KEY\": \"8772750e-176a-4d81-af2a-364195aea294\", \"CUST_ID\":\"0117ad10-f7ec-4fdb-8bd0-12dc2c97af36\", \"ICON_URL\":\"https://img-amc.tason.com/resources/renewal/images/tas-favicon.ico\", \"TITLE\":\"push test\", \"MESSAGE\":\"push msg test\", \"LANDING_URL\": \"https://www.tason.com\", \"REQUEST_UID\":\"1234567890\"}"; 
		ObjectMapper mapper = new ObjectMapper();
		
		WebPushPayload payload;
		try {
			payload = mapper.readValue(json, WebPushPayload.class);
			payload.setPushKey("l");
			payload.setPushValue("https://jsonlint.com/");
			payload.setReportQueue("dev-web-push-report");
			System.out.println("payload :"+payload.getUuid());
			s.send(payload, false);
			
			try {
				Thread.sleep(10000);
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		} catch (JsonParseException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (JsonMappingException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}
}
