package com.humuson.tas.sender.client;

import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.humuson.tas.sender.model.push.AppPushPayload;

import lombok.extern.slf4j.Slf4j;

@Slf4j
public class AppPushSendService implements KafkaService<AppPushPayload> {
	
	private static final String DEV_APPPUSH_TOPIC = "dev-app-push-request";
	private static final String PROD_APPPUSH_TOPIC = "prod-app-push-request";
	
	private Producer<String, String> pushProducer;
	private ObjectMapper mapper;
	
	@Override
	public void init() {
		pushProducer = KafkaProducerFactory.getProducer();
		mapper = new ObjectMapper();
	}
	
	@Override
	public void send(final AppPushPayload payload) throws JsonProcessingException {
		this.send(payload, true);
	}
	
	@Override
	public void send(final AppPushPayload payload, boolean isProduct) throws JsonProcessingException {
		final String data = this.mapper.writeValueAsString(payload);
		final String topic = isProduct ? PROD_APPPUSH_TOPIC: DEV_APPPUSH_TOPIC;
		this.send(topic, data);
	}
	
	@Override
	public void send(final String topic, final AppPushPayload payload) throws JsonProcessingException {
		final String data = this.mapper.writeValueAsString(payload);
		this.send(topic, data);
	}

	@Override
	public void send(final String topic, final String data) {
		pushProducer.send(new ProducerRecord<String, String>(topic, data), new Callback() {
			@Override
			public void onCompletion(RecordMetadata metadata, Exception exception) {
				if (exception != null) {
					log.error("send error : {}", data);
		        }
			}
		});
	}
	
	@Override
	public void close() {
		if (pushProducer != null) {
			try {
				pushProducer.close();
			} catch (Exception e) {
				log.error("kafka producer close error", e);
			}
		}
		
	}

}
