package com.humuson.tas.sender.client;

import java.util.concurrent.TimeUnit;

import org.apache.kafka.clients.producer.Callback;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.humuson.tas.sender.model.push.PushReport;

import lombok.extern.slf4j.Slf4j;

@Slf4j
public class AppPushReportService implements KafkaService<PushReport> {
	
	private static final String DEV_APPPUSH_REPORT_TOPIC = "dev-report-push";
	private static final String PROD_APPPUSH_REPORT_TOPIC = "prod-report-push";
	private static final long DEFAULT_TIMEOUT_MILLISECONDS = 5000;
	private Producer<String, String> pushProducer;
	private ObjectMapper mapper;
	
	@Override
	public void init() {
		pushProducer = KafkaConnectionFactory.getProducer();
		mapper = new ObjectMapper();
	}
	
	@Override
	public void send(final PushReport payload) throws JsonProcessingException {
		this.send(payload, true);
	}
	
	@Override
	public void send(final PushReport payload, boolean isProduct) throws JsonProcessingException {
		final String data = this.mapper.writeValueAsString(payload);
		final String topic = isProduct ? PROD_APPPUSH_REPORT_TOPIC : DEV_APPPUSH_REPORT_TOPIC ;
		this.send(topic, data);
	}
	
	@Override
	public void send(final String topic, final PushReport payload) throws JsonProcessingException {
		final String data = this.mapper.writeValueAsString(payload);
		this.send(topic, data);
	}

	@Override
	public void send(final String topic, final String data) {
		pushProducer.send(new ProducerRecord<String, String>(topic, data), new Callback() {
			@Override
			public void onCompletion(RecordMetadata metadata, Exception exception) {
				if (exception != null) {
					log.error("send error : {}", data);
		        }
			}
		});
	}
	
	@Override
	public boolean sendSync(PushReport payload) throws Exception {
		return this.sendSync(payload, true);
	}

	@Override
	public boolean sendSync(PushReport payload, boolean isProduct) throws Exception{
		return this.sendSync(payload, isProduct, DEFAULT_TIMEOUT_MILLISECONDS);
	}

	@Override
	public boolean sendSync(PushReport payload, boolean isProduct, long timeout) throws Exception{
		final String topic = isProduct? PROD_APPPUSH_REPORT_TOPIC : DEV_APPPUSH_REPORT_TOPIC;
		return this.sendSync(topic, payload, isProduct, timeout);
	}

	@Override
	public boolean sendSync(String topic, PushReport payload, boolean isProduct, long timeout)
			throws Exception {
		try {
			final String data = this.mapper.writeValueAsString(payload);
			RecordMetadata recodeMetadata = pushProducer.send(new ProducerRecord<String, String>(topic, data)).get(DEFAULT_TIMEOUT_MILLISECONDS, TimeUnit.MILLISECONDS);
			if (log.isDebugEnabled()) {
				log.debug("topic:{}, partition:{}, checksum:{}", recodeMetadata.topic(), recodeMetadata.partition(), recodeMetadata.checksum());
			}
			return true;
		} catch (Exception e) {
			log.error("send kafka error", e);
			throw e;
		}
	}
	
	@Override
	public void close() {
		if (pushProducer != null) {
			try {
				pushProducer.close();
			} catch (Exception e) {
				log.error("kafka producer close error", e);
			}
		}
	}

}
