package com.humuson.tms.common.security;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.security.KeyStore;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Enumeration;

import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

public class SSLChecker {

	public static final String PRODUCTION = "Apple Push Services|Apple Production IOS Push Services";
	public static final String DEVELOPMENT = "Apple Development IOS Push Services";
	
	private static final Logger logger = LoggerFactory.getLogger(SSLChecker.class);
	
	
	public static boolean isValidPassword(String keystore, String password) {
		InputStream certStream = null;
		try {
			certStream = new FileInputStream(keystore);
			return isValidPassword(certStream, password);
		} catch (Exception e1) {
			// TODO Auto-generated catch block
			logger.error("isValidPassword error", e1);
		}
		return false;
	}
	
	public static boolean isValidPassword(InputStream certStream, String password) {
		try {
			KeyStore p12 = KeyStore.getInstance("pkcs12");
			logger.info("password :{}", password);
			p12.load(certStream, password.toCharArray());
			return true;
		} catch (CertificateException e) {
			logger.error("password check failed", e);
		} catch (Exception e) {
			logger.error("isValidPassword error", e);
		}
		
		return false;
	}

	public static boolean isProduction(InputStream certStream, String password) {
		return isProduction(certStream, password, PRODUCTION, DEVELOPMENT, true);
	}
	
	public static boolean isProduction(InputStream certStream, String password, boolean isWeb) {
		return isProduction(certStream, password, PRODUCTION, DEVELOPMENT, isWeb);
	}
	
	public static boolean isProduction(InputStream certStream, String password,
										String productionCN, String developmentCN, boolean isWeb) {
		
		boolean isProductionCert = false;
		KeyStore keystore = null;
		try {
			keystore = KeyStore.getInstance("PKCS12");

			keystore.load(certStream, password.toCharArray());
			Enumeration<String> aliases = keystore.aliases();

			while (aliases.hasMoreElements()) {
				X509Certificate certificate = (X509Certificate) keystore.getCertificate((String) aliases.nextElement());
				String cnName = findCnName(certificate);
				
				for(String cn : productionCN.split("\\|")){
					if(cnName.indexOf(cn) > -1){
						isProductionCert = true;
					}
				}
				for(String cn : developmentCN.split("\\|")){
					if(cnName.indexOf(cn) > -1){
						isProductionCert = false;
					}
				}
				break;
			}
		} catch (Exception e) {
			logger.error("p12ProductionCheck error", e);
		} finally {
			try {
				if (certStream != null && !isWeb ) {
					certStream.close();
				}
			} catch (IOException e) {
				logger.error("p12ProductionCheck error", e);
			}
		}
		return isProductionCert;
		
		
	}
	
	public static boolean isProduction(String keystore, String password) {
		InputStream certStream = null;
		try {
			certStream = new FileInputStream(keystore);
			return isProduction(certStream, password);
		} catch (Exception e1) {
			// TODO Auto-generated catch block
			logger.error("isProduction error", e1);
		}
		return false;
	}
	
	public static String getExpireDate(String keystore, String password) {
		InputStream certStream = null;
		try {
			certStream = new FileInputStream(keystore);
			return getExpireDate(certStream, password);
		} catch (Exception e1) {
			// TODO Auto-generated catch block
			logger.error("getExpireDate error", e1);
		}
		return null;
	}
	
	public static String getExpireDate(InputStream certStream, String password) {
		String expireDate = null;
		
		KeyStore p12;
		try {
			p12 = KeyStore.getInstance("pkcs12");
			logger.info("password :{}", password);
			p12.load(certStream, password.toCharArray());
	        Enumeration<String> e = p12.aliases();
	        
	        while (e.hasMoreElements()) {
	            String alias = (String) e.nextElement();
	            X509Certificate c = (X509Certificate) p12.getCertificate(alias);
	            expireDate = new SimpleDateFormat("yyyyMMddHHmmss").format(c.getNotAfter());
	            break;
	        }
		} catch (Exception e1) {
			// TODO Auto-generated catch block
			logger.error("getExpireDate error", e1);
		} 
		logger.info("expireDate  :{}", expireDate);
		return expireDate;
	}
	
	public static boolean checkValidity(InputStream certStream, String password) {
		boolean isExpired = true;
		
		KeyStore p12;
		try {
			p12 = KeyStore.getInstance("pkcs12");
			
			p12.load(certStream, password.toCharArray());
	        Enumeration<String> e = p12.aliases();
	        while (e.hasMoreElements()) {
	            String alias = (String) e.nextElement();
	            X509Certificate c = (X509Certificate) p12.getCertificate(alias);
	            c.checkValidity(new Date());
	        }
		} catch (Exception e1) {
			// TODO Auto-generated catch block
			logger.error("checkValidity error", e1);
			isExpired = false;
		}
		
		return isExpired;
		
	}
	
	public static boolean checkValidity(String keystore, String password) {
		InputStream certStream = null;
		try {
			certStream = new FileInputStream(keystore);
			return checkValidity(certStream, password);
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return false;
	}
	
	protected static String findCnName(X509Certificate certificate) throws InvalidNameException{
		LdapName ldapDN = new LdapName(certificate.getSubjectX500Principal().getName());
		String cnName = null;
		for (Rdn rdn : ldapDN.getRdns()) {
			if(rdn.getType().equals("CN")){
				cnName = String.valueOf(rdn.getValue());
				break;
			}
		}
		return cnName;
	}
	
	public static void main(String[] args) {
		String cert = "/Users/hyogun/workspace/pms-work/pms-commons-trunk/apns_apns-gsshop-prd.p12";
		if (SSLChecker.isValidPassword(cert, "gs13253")) {
			String expireDate = SSLChecker.getExpireDate(cert, "gs13253");
			logger.info(expireDate);
			
			boolean isValid = SSLChecker.checkValidity(cert, "gs13253");
			logger.info("isValid : {}", isValid);
			
			boolean isProduction = SSLChecker.isProduction(cert, "gs13253");
			logger.info("production p12 {}",isProduction);
		} else {
			logger.info("password invalid");
		}
		
	}
}
