/*
 * Created on 2005-12-13
 */
package venus.spool.common.parser;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

import lombok.extern.slf4j.Slf4j;
import venus.spool.auto.task.AutoBaseSpoolTask;

/**
 * @author 이상근
 */
@Slf4j
public class RcptInfoHashList {

	/**
	 * The default initial capacity - MUST be a power of two.
	 */
	static final int				DEFAULT_INITIAL_CAPACITY	= 100;

	/**
	 * The maximum capacity, used if a higher value is implicitly specified by either of the constructors with arguments. <br>
	 * MUST be a power of two <= 1 < <30.
	 */
	static final int				MAXIMUM_CAPACITY			= 1 << 30;

	/**
	 * The load factor used when none specified in constructor.
	 */
	static final float				DEFAULT_LOAD_FACTOR			= 0.75f;

	/**
	 * 도메인별 그룹핑 갯수 설정
	 */
	private static Hashtable		__INNER_SINGLE_INFORM__		= null;

	private static Hashtable		__INNER_MULTI_INFORM__		= null;

	private static int				__SINGLE_DEFAULT_SIZE__		= 10;

	private static int				__MULTI_DEFAULT_SIZE__		= 20;

	public static final DomainEntry	EMPTY_DOMAIN_ENTRY			= new DomainEntry();

	/**
	 * 그루핑 카운트 매핑 테이블이 저장된 Property파일의 경로를 받아 <br>
	 * 내부 Property 파라미터를 초기화한다. <br>
	 * 초기화 설정에서 <B>target.file </B>: 프로퍼티 파일 경로 <br>
	 * <B>contain.limit </B>: 동시에 저장하는 도메인의 수 <br>
	 * 에 지정된 파일을 읽어 초기화 한다.
	 * 
	 * @param prop
	 *        Property파일의 경로
	 * @throws Exception
	 *         입력파라미터 오류나 경로가 틀렸을 경우
	 */
	public synchronized static void init(Object tmp) throws Exception {
		Properties prop = (Properties) tmp;

		__INNER_SINGLE_INFORM__ = new Hashtable();
		__INNER_MULTI_INFORM__ = new Hashtable();

		/**
		 * 숫자인지 검증을 해봐야 한다.
		 */
		for (Enumeration a = prop.propertyNames(); a.hasMoreElements();) {
			String key = a.nextElement().toString();
			String value = prop.getProperty(key);

			try {
				if( key.startsWith("s.") ) {

					__INNER_SINGLE_INFORM__.put(key.substring(2), Integer.valueOf(value));
				}
				else if( key.startsWith("m.") ) {
					__INNER_MULTI_INFORM__.put(key.substring(2), Integer.valueOf(value));
				}
			}
			catch(Exception e) {
				log.error("DOMAIN KEY SETTING ERROR", e);
				continue;
			}
		}

		__SINGLE_DEFAULT_SIZE__ = Integer.parseInt(prop.getProperty("default.single", "10"));
		__MULTI_DEFAULT_SIZE__ = Integer.parseInt(prop.getProperty("default.multi", "10"));
		log.error("SINGLE SETTING"+ __INNER_SINGLE_INFORM__.toString());
		log.error("MULTI SETTING"+ __INNER_MULTI_INFORM__.toString());
	}

	/**
	 * 등록된 설정의 도메인별 갯수를 반환한다. 등록된 도메인이 아닐 경우 기본값을 반환한다.
	 * 
	 * @param Domain
	 *        타겟도메인
	 * @throws Exception
	 *         에러가 나면
	 * @return 동보/원투원의 해당도메인에 대한 그루핑 카운트
	 */
	static final int getSingleSize(String domain) {
		if( domain == null ) {
			return -1;
		}

		Integer ReturnValue = (Integer) __INNER_SINGLE_INFORM__.get(domain);

		if( ReturnValue != null ) {
			return ReturnValue.intValue();
		}

		return __SINGLE_DEFAULT_SIZE__;
	}

	/**
	 * 등록된 설정의 도메인별 갯수를 반환한다. 등록된 도메인이 아닐 경우 기본값을 반환한다.
	 * 
	 * @param Domain
	 *        타겟도메인
	 * @throws Exception
	 *         에러가 나면
	 * @return 동보/원투원의 해당도메인에 대한 그루핑 카운트
	 */
	static final int getMultipleSize(String domain) {
		if( domain == null ) {
			return -1;
		}

		Integer ReturnValue = (Integer) __INNER_MULTI_INFORM__.get(domain);

		if( ReturnValue != null ) {
			return ReturnValue.intValue();
		}

		return __MULTI_DEFAULT_SIZE__;
	}

	/**
	 * The table, resized as necessary. Length MUST Always be a power of two.
	 */
	transient DomainEntry[]	table;

	/**
	 * The number of key-value mappings contained in this identity hash map.
	 */
	transient int			size;

	/**
	 * The next size value at which to resize (capacity * load factor).
	 * 
	 * @serial
	 */
	int						threshold;

	/**
	 * The load factor for the hash table.
	 * 
	 * @serial
	 */
	final float				loadFactor;

	public RcptInfoHashList(int initialCapacity, float loadFactor) {
		// -- 입력 값 점검 -- //
		if( initialCapacity < 0 ) {
			throw new IllegalArgumentException("Illegal initial capacity: " + initialCapacity);
		}

		if( loadFactor <= 0 || Float.isNaN(loadFactor) ) {
			throw new IllegalArgumentException("Illegal load factor: " + loadFactor);
		}

		if( initialCapacity > MAXIMUM_CAPACITY ) {
			initialCapacity = MAXIMUM_CAPACITY;
		}

		// Find a power of 2 >= initialCapacity
		int capacity = 1;
		while (capacity < initialCapacity) {
			capacity <<= 1;
		}

		this.loadFactor = loadFactor;
		threshold = (int) (capacity * loadFactor);
		table = new DomainEntry[capacity];
	}

	/**
	 * Constructs an empty <tt>HashMap</tt> with the specified initial capacity and the default load factor (0.75).
	 * 
	 * @param initialCapacity
	 *        the initial capacity.
	 * @throws IllegalArgumentException
	 *         if the initial capacity is negative.
	 */
	public RcptInfoHashList(int initialCapacity) {
		this(initialCapacity, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Constructs an empty <tt>HashMap</tt> with the default initial capacity (16) and the default load factor (0.75).
	 */
	public RcptInfoHashList() {
		this.loadFactor = DEFAULT_LOAD_FACTOR;
		threshold = (int) (DEFAULT_INITIAL_CAPACITY * DEFAULT_LOAD_FACTOR);
		table = new DomainEntry[DEFAULT_INITIAL_CAPACITY];
	}

	/**
	 * Returns a hash value for the specified object. In addition to the object's own hashCode, this method applies a "supplemental hash function," which defends against poor quality hash functions.
	 * This is critical because HashMap uses power-of two length hash tables.
	 * <p>
	 * 
	 * The shift distances in this function were chosen as the result of an automated search over the entire four-dimensional search space.
	 */
	static int hash(Object x) {
		int h = x.hashCode();

		h += ~(h << 9);
		h ^= (h >>> 14);
		h += (h << 4);
		h ^= (h >>> 10);
		return h;
	}

	/**
	 * Check for equality of non-null reference x and possibly-null y.
	 */
	static boolean eq(Object x, Object y) {
		return x == y || x.equals(y);
	}

	/**
	 * Returns index for hash code h.
	 */
	static int indexFor(int h, int length) {
		return h & (length - 1);
	}

	/**
	 * Returns the number of key-value mappings in this map.
	 * 
	 * @return the number of key-value mappings in this map.
	 */
	public int size() {
		return size;
	}

	/**
	 * Returns <tt>true</tt> if this map contains no key-value mappings.
	 * 
	 * @return <tt>true</tt> if this map contains no key-value mappings.
	 */
	public boolean isEmpty() {
		return size == 0;
	}

	void minusSize() {
		this.size--;
	}

	// -- 외부 모듈 호출 Method -- //

	/**
	 * 해당 도메인 엔트리에 대상 리스트를 추가한다. <br>
	 * 도메인이나 다른 입력 값에대한 검증은 행하지 않으며 asynch로 동작하기 때문에 ThreadSafe가 보장되지 않으므로 sync 를 호출 측에서 보장해야한다. <br>
	 * 동보와 멀티의 구분은 mapping에 값이 null or Not 으로 구분된다. <br>
	 * 정상적으로 엔트리에 추가가 되면 빈 리스트가 담긴 DomainEntry를 반환하고 <br>
	 * 해당 도메인에 제한을 초과하는 리스트가 들어온다면 해당 리스트를 가지는 DomainEntry를 반환한다.
	 * 
	 * @param domain
	 * @param step
	 * @param mid
	 * @param email
	 * @param mapping
	 * @param etc
	 * @return
	 */
	public DomainEntry put(String domain, int step, String mid, String email, String mapping, String etc) {
		if( domain == null ) {
			if (log.isDebugEnabled()) {
				log.debug("Domain Is Null");
				Exception e = new Exception("DEBUG");				
				log.error("Exception", e);
				//				Hashtable;
			}
			return null;
		}
		int hash = hash(domain);
		int i = indexFor(hash, table.length);
		DomainEntry checkEntry = table[i];
		DomainEntry prevEntry = checkEntry;
		// -- 엔트리 점검하여 해당 엔트리가 존재하는지 점검 -- //
		while (checkEntry != null) {
			if( checkEntry.hash == hash && eq(domain, checkEntry.domain) ) {
				// Bingo ~~~
				if( checkEntry.addListEntry(step, mid, email, mapping, etc) ) {
					// limit를 초과하였으므로 해당 리스트를 정리하고 이녀석을 반환해야한다.
					if( prevEntry == checkEntry ) {
						// 첫방에 걸리면.
						table[i] = checkEntry.next;
					}
					else {
						// 아니면 뒤에 엔트리와 앞 엔트리를 붙여주고
						prevEntry.next = checkEntry.next;
					}

					// 해당 엔트리를 반환한다.
					return checkEntry;
				}
				else {
					// 정상적으로 attach 되었기 때문에 null을 반환한다.
					return EMPTY_DOMAIN_ENTRY;
				}
			}
			prevEntry = checkEntry;
			checkEntry = checkEntry.next;
		}

		// -- 일치하는 엔트리가 없기 때문에 새로운 엔트리를 추가해야 한다. -- //
		// 신규로 생성되는 도메인에 해당하는 리미트 값을 가져온다.
		int newLimit = (mapping == null) ? getMultipleSize(domain) : getSingleSize(domain);

		// 신규 엔트리를 생성한다.
		DomainEntry newEntry = new DomainEntry(hash, domain, table[i], newLimit);
		table[i] = newEntry;

		// 생성된 엔트리에 지금 들어온 리스트를 추가한다.
		newEntry.addListEntry(step, mid, email, mapping, etc);
		this.size++;
		// -- 저장 공간을 초과하였다면? -- //
		if( this.size > this.threshold ) {
			// 저장공간 초과를 알린다.
			return null;
		}
		else {
			// 하나 추가한거니까 그냥 돌아간다. 제한이 하나라면? ㅠㅠ
			return EMPTY_DOMAIN_ENTRY;
		}
	}

	public HashListIterator getIterator() {
		return new HashListIterator(this);
	}

	public static class HashListIterator {
		private int			current_check_index	= -1;

		RcptInfoHashList	owner				= null;

		HashListIterator(RcptInfoHashList list) {
			this.owner = list;
			current_check_index = 0;
		}

		public DomainEntry nextEntry() {
			DomainEntry[] table = this.owner.table;

			DomainEntry target = null;

			while (current_check_index < table.length) {
				target = table[current_check_index];
				if( target == null ) {
					current_check_index++;
				}
				else {
					break;
				}
			}

			if( target == null ) {
				return null;
			}

			table[current_check_index] = target.next;
			this.owner.minusSize();
			return target;
		}
	}

	public static class DomainEntry {
		// -- Interval Member Variable -- //

		public String	POST_ID		= null;

		public String	SEND_KIND	= null;

		public String	MAIL_FROM	= null;

		public String	LIST_TABLE	= null;

		final String	domain;

		final int		hash;

		ListEntry		headOfList;

		DomainEntry		next;

		int				size;

		int				limit;

		DomainEntry() {
			this.domain = null;
			this.hash = -1;
		}

		/**
		 * Create new DomainEntry.
		 */
		DomainEntry(int hashValue, String d, DomainEntry n, int l) {
			this.domain = d;
			this.next = n;
			this.hash = hashValue;
			this.headOfList = new ListEntry();
			this.size = 0;
			this.limit = l;
		}

		// -- 기본정보 세팅 -- //
		/**
		 * 
		 * @param pid
		 */
		public void setPOST_ID(String pid) {
			this.POST_ID = pid;
		}

		public void setSEND_KIND(String kind) {
			this.SEND_KIND = kind;
		}

		public void setMAIL_FROM(String mailfrom) {
			this.MAIL_FROM = mailfrom;
		}

		public void setLIST_TABLE(String listTable) {
			this.LIST_TABLE = listTable;
		}

		public String getDomain() {
			return this.domain;
		}

		public ListEntry getListEntry() {
			return this.headOfList;
		}

		/**
		 * 
		 * @param step
		 * @param mid
		 * @param email
		 * @param mapping
		 * @param addInfo
		 */
		public synchronized boolean addListEntry(int step, String mid, String email, String mapping, String etc) {
			ListEntry newEntry = new ListEntry(step, mid, email, mapping, etc, headOfList, headOfList.previous);

			headOfList.previous.next = newEntry;
			headOfList.previous = newEntry;

			return this.size++ > this.limit;
		}

		@Override
		public int hashCode() {
			return (domain.hashCode()) ^ (headOfList.hashCode());
		}

		public String toString() {
			return domain + " entry";
		}
	}

	public static class ListEntry {
		public ListEntry	next;

		public ListEntry	previous;

		public String		TMS_M_ID;

		public String		TMS_M_TOKEN;

		public String		MAPPING;

		public String		ETC_INFOS;

		public int			step;

		public short		status;

		ListEntry(int s, String mid, String token, String mapping, String etc, ListEntry next, ListEntry previous) {
			this.TMS_M_ID = mid;
			this.TMS_M_TOKEN = token;
			this.MAPPING = mapping;
			this.ETC_INFOS = etc;
			this.step = s;
			this.next = next;
			this.previous = previous;
			this.status = -1;
		}

		ListEntry() {
			this.next = this;
			this.previous = this;
		}
	}
}
