/*
 * @(#)StandardSpooler.java            2004. 11. 30.
 *
 * Copyright (c) 1998-2004 Amail, Inc.
 * 708-8 Global Building 10th floor, YeokSamdong, Gangnamgu, Seoul, 
 * Korea republic of. All rights reserved.
 * 
 * This software is the confidential and proprietary information of Amail,
 * Inc. ("Confidential Information"). You shall not disclose such 
 * Confidential Information and shall use it only in accordance with
 * the terms of the license agreement you entered into Amail.
 * 
 */

package venus.spool.common.handler;

import java.io.File;
import java.io.FilenameFilter;
import java.util.Properties;

import lombok.extern.slf4j.Slf4j;
import pluto.io.HeadTailFilenameFilter;
import pluto.io.eMsFileWriter;
import pluto.util.Cal;

/**
 * 재발송을 위한 스풀 파일을 생성하는 작업을 지원한다.
 * 
 * @version
 * @author dragon
 *  
 */
@Slf4j
public class StandardSpooler implements Spooler {

	/** Class 이름 */
	public static final String		name						= "StandardSpooler";

	/** 30 분동안 저장을 하고 지나면 재발송을 시작한다. */
	public static final long		DEFAULT_STORE_TIME			= 30 * 60 * 1000L;

	/** 10000 개 이상의 스풀이 생성되면 시간에 관계없이 교체 */
	public static final int			DEFAULT_MAX_STORE_COUNT		= 10000;

	/** 저장시간 */
	public static long				STORE_TIME					= DEFAULT_STORE_TIME;

	/** 스풀 교체 한계 개수 */
	public static int				MAX_STORE_COUNT				= DEFAULT_MAX_STORE_COUNT;

	/** 스풀 저장 디렉토리 */
	private static String			STORE_BASE_DIR				= null;

	/** 스풀 파일 이름 header */
	private static String			STORE_WORK_FILE_HEADER		= null;

	/** 스풀 파일 확장자 */
	private static String			STORE_WORK_FILE_EXT			= null;

	/** 스풀 발송 대기 파일 이름 header */
	private static String			STORE_WAIT_FILE_HEADER		= null;

	/** 스풀 발송 대기 파일 확장자 */
	private static String			STORE_WAIT_FILE_EXT			= null;

	/** 발송되고 있는 스풀 파일 이름 header */
	private static String			STORE_WORKING_FILE_HEADER	= null;

	/** 발송되고 있는 스풀 파일 확장자 */
	private static String			STORE_WORKING_FILE_EXT		= null;

	/** eMsFileWriter */
	private eMsFileWriter			writer						= null;

	/** 현재 작업 스풀 파일 */
	private String					current_spool_file			= null;

	private int						spool_count					= 0;

	private long					create_time					= -1;

	/** Creates new StandardSpooler */
	public StandardSpooler() {
		this.create_time = System.currentTimeMillis();
	}

	/**
	 * <br>
	 * Spooler 타겟에서 spooler.class에 지정된 경우 실행된다. <br>
	 * var의 name,value 셋을 Properties에 넣어서 실행한다. <br>
	 * <br>
	 * >>>>> 설정 파일 예제 <br>
	 * &lt;TARGET name="SPOOLER"> <br>
	 * &lt;class name="venus.spool.basic.SpoolingManager"/> <br>
	 * &lt;!-- 내부 모니터링 쓰레드의 체크주기 단위는 (분) --> <br>
	 * &lt;var name="next.check.recycle" value="30"/> <br>
	 * &lt;var name="spooler.class" value="venus.spool.module.StandardSpooler"/> <br>
	 * &lt;var name="resend.class" value="pluto.common.task.SimpleQueRecycleMailSendTask"/> <br>
	 * &lt;!-- 큐 스풀링을 하는 디렉토리 지정 --> <br>
	 * &lt;var name="work.dir" value="${base}/mqueue"/> <br>
	 * &lt;var name="work.file.header" value="ResendQue"/> <br>
	 * &lt;var name="work.file.ext" value=".spooling"/> <br>
	 * &lt;var name="wait.file.header" value="Resending"/> <br>
	 * &lt;var name="wait.file.ext" value=".work"/> <br>
	 * &lt;var name="send.file.header" value="QueProcess"/> <br>
	 * &lt;var name="send.file.ext" value=".sending"/> <br>
	 * &lt;!-- 발송을 마친 파일을 저장하는 디렉토리 --> <br>
	 * &lt;var name="end.dir" value="${base}/mqueue"/> <br>
	 * &lt;var name="end.file.header" value="ResendDone"/> <br>
	 * &lt;var name="end.file.ext" value=".done"/> <br>
	 * &lt;var name="end.dir.timestamp" value="yyyyMMdd"/> <br>
	 * &lt;!-- <br>
	 * 최장 시간에 도달 하지 않더라도 이 숫자의 큐발송건이 발생하면 큐발송을 시작한다. <br># [default] 10000 <br>
	 * --> <br>
	 * &lt;var name="store.limit" value="100000"/> <br>
	 * &lt;!-- <br># 큐에 저장되는 최장 시간 limit에 도달 하지 않더라도 이간이 지나면 큐 발송을 시작한다. 단위( sec:초 )<br># 단위 ( 분 )<br>
	 * --> <br>
	 * &lt;var name="max.age" value="30"/> <br>
	 * &lt;/TARGET>
	 * 
	 * @param prop
	 * @throws Exception
	 */
	public static void init(Object prop) throws Exception {

		Properties tmp = (Properties) prop;

		STORE_BASE_DIR = tmp.getProperty("work.dir");

		if( STORE_BASE_DIR == null )
			throw new RuntimeException("store.dir param is not set... ");

		try {
			STORE_TIME = Long.parseLong(tmp.getProperty("max.age")) * 60 * 1000;
		}
		catch(Exception e) {
		}

		if(log.isDebugEnabled()){
			log.debug("MAX AGE:" + STORE_TIME);
		}

		STORE_WORK_FILE_HEADER = tmp.getProperty("work.file.header", "ResendQue");

		STORE_WORK_FILE_EXT = tmp.getProperty("work.file.ext", ".spooling");

		STORE_WAIT_FILE_HEADER = tmp.getProperty("wait.file.header", "Resending");

		STORE_WAIT_FILE_EXT = tmp.getProperty("wait.file.ext", ".work");

		STORE_WORKING_FILE_HEADER = tmp.getProperty("send.file.header", "QueProcess");

		STORE_WORKING_FILE_EXT = tmp.getProperty("send.file.ext", ".sending");

		try {
			MAX_STORE_COUNT = Integer.parseInt(tmp.getProperty("store.limit"));
		}
		catch(Exception e) {
		}

		if(log.isDebugEnabled()){
			log.debug("MAX_STORE_COUNT:" + MAX_STORE_COUNT);
		}

		/*
		 * 발송 중이었던 파일이 있을 경우 삭제 해버린다.
		 */
		log.debug( "DELETE BEFORE WORKING FILES");

		FilenameFilter workingFileFilter = new HeadTailFilenameFilter(STORE_WORKING_FILE_HEADER, STORE_WORKING_FILE_EXT);

		File base = new File(STORE_BASE_DIR);

		String working_files[] = base.list(workingFileFilter);

		for (int i = 0; i < working_files.length; i++) {

			String tmpFilename = STORE_BASE_DIR + File.separator + working_files[i];

			if (log.isDebugEnabled()) {
				log.debug("StandardSpooler before working file delete=>" + tmpFilename);
			}

			File source = new File(tmpFilename);

			boolean fileDel = source.delete();
			if(!fileDel) log.error("File deletion failed");
		}

		/*
		 * 비정상적으로 종료 되었을 경우에 작업중이던 파일을 작업 완료로 전환한다.
		 */
		log.debug( "RENAME BEFORE SPOOLING FILES");

		FilenameFilter tmpFilter = new HeadTailFilenameFilter(STORE_WORK_FILE_HEADER, STORE_WORK_FILE_EXT);

		File dir = new File(STORE_BASE_DIR);

		String files[] = dir.list(tmpFilter);

		for (int i = 0; i < files.length; i++) {

			String tmpFilename = STORE_BASE_DIR + File.separator + files[i];
			String TargetFileName = STORE_BASE_DIR + File.separator + STORE_WAIT_FILE_HEADER + Cal.getSerialDate() + STORE_WAIT_FILE_EXT;

			File source = new File(tmpFilename);
			File target = new File(TargetFileName);

			boolean fileRename = source.renameTo(target);
			if(!fileRename) log.error("File name rename failed");

			try {
				Thread.currentThread().sleep(500L);
			}
			catch(Exception e) {
			}
		}
	}

	public static String getName() {
		return name;
	}

	/**
	 * 기존의 저장되어 있는 스풀을 처리하고 모든 관련 자원을 반환한다.
	 */
	public void clean() {

		try {
			writer.flush();
		}
		catch(Exception e) {
		}

		try {
			writer.close();
		}
		catch(Exception e) {
		}
	}

	/**
	 * 발송중에 재발송이 필요한 스풀을 append한다.
	 * 
	 * @param spool
	 *        스풀 스트링
	 * @throws Exception
	 *         에러입니다.
	 */
	public synchronized void addSpool(Object spool) throws Exception {

		if( spool == null )
			return;

		if( writer == null ) {

			this.current_spool_file = STORE_BASE_DIR + File.separator + STORE_WORK_FILE_HEADER + Cal.getSerialDate() + STORE_WORK_FILE_EXT;

			this.writer = new eMsFileWriter(this.current_spool_file);
			this.create_time = System.currentTimeMillis();
			this.spool_count = 0;
		}

		if( writer != null ) {
			try {
				this.writer.println(spool.toString());
				this.writer.flush();
				this.spool_count++;
			}
			catch(Exception e) {
				log.error("SPOOLER", e);

				try {
					this.writer.close();
				}
				catch(Exception _e) {
				}

				this.writer = new eMsFileWriter(this.current_spool_file, true);
				this.writer.println(spool.toString());
				this.writer.flush();
				this.spool_count++;
			}
		}

		if( this.spool_count > MAX_STORE_COUNT )
			switchSpool();
	}

	/**
	 * 스풀을 정리할때가 되었는지를 시간을 체크하여 그 결과를 반환한다.
	 * 
	 * @return true: 스풀을 처리하고 다시 생성해야함( Manager는 switchSpool 을 호출한다. )<br>
	 *         false: 아직은 그냥 더 있어도 된다.
	 */
	public boolean storeNext() {
		if (log.isDebugEnabled()) {
			log.debug("[Spooler] check next current : " + this.spool_count + " create time: " + this.create_time + " now : " + System.currentTimeMillis());
			log.debug("result:" + ((System.currentTimeMillis() - this.create_time) > STORE_TIME));
		}
		log.info("[Spooler] check next current : " + this.spool_count + " create time: " + this.create_time + " now : " + System.currentTimeMillis());

		if( this.spool_count == 0 ) {
			this.create_time = System.currentTimeMillis();
		}

		return (System.currentTimeMillis() - this.create_time) > STORE_TIME;
	}

	/**
	 * 호출전까지 저장된 스풀을 처리하고 새로운 스풀저장공간을 설정한다.
	 * 
	 * @throws Exception
	 *         에러발생
	 */
	public synchronized void switchSpool() throws Exception {

		try {
			writer.flush();
		}
		catch(Exception e) {
		}

		try {
			writer.close();
		}
		catch(Exception e) {
		}

		try {			
			wait(1000);			
		}
		catch(Exception e) {
		}

		try {
			File tmpFile = new File(this.current_spool_file);

			String swap_filename = STORE_BASE_DIR + File.separator + STORE_WAIT_FILE_HEADER + Cal.getSerialDate() + STORE_WAIT_FILE_EXT;

			if( !tmpFile.renameTo(new File(swap_filename)) ) {
				log.info("SWAPING FILE FAIL:" + this.current_spool_file + "=>" + swap_filename);
			}

			tmpFile = null;

			this.current_spool_file = STORE_BASE_DIR + File.separator + STORE_WORK_FILE_HEADER + Cal.getSerialDate() + STORE_WORK_FILE_EXT;

			this.writer = new eMsFileWriter(this.current_spool_file);
			this.create_time = System.currentTimeMillis();
			this.spool_count = 0;
		}
		catch(Exception e) {
			log.error("SPOOLER", e);
			writer = null;
		}
	}
}
