/*
 * @(#)SpoolInfoManager.java            2004. 12. 1.
 *
 * Copyright (c) 1998-2004 Amail, Inc.
 * 708-8 Global Building 10th floor, YeokSamdong, Gangnamgu, Seoul, 
 * Korea republic of. All rights reserved.
 * 
 * This software is the confidential and proprietary information of Amail,
 * Inc. ("Confidential Information"). You shall not disclose such 
 * Confidential Information and shall use it only in accordance with
 * the terms of the license agreement you entered into Amail.
 * 
 */

package venus.spool.common.basic;

import java.io.File;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

import pluto.config.eMsSystem;
import pluto.io.FileElement;
import lombok.extern.slf4j.Slf4j;
import pluto.schedule.ScheduledMonitor;
import venus.spool.auto.task.AutoBaseSpoolTask;

/**
 * SpoolInfo를 저장하고 관리한다.
 * 
 * @version
 * @author lena
 *  
 */
@Slf4j
public class SpoolInfoManager extends ScheduledMonitor {

	/* 10분 마다 한번씩 컨텐트의 유효기간 검사를 하게 된다. */
	public static final long		CHECK_INTERVAL			= 60 * 1000L;

	private static String			INFO_SAVE_BASE_DIR		= null;

	private static String			INFO_SAVE_BASE_URL		= null;

	private static String			INFO_SAVE_FILE_EXT		= null;

	private static String			INFO_DTD_LOCATION		= null;

	private static String			INFO_XML_CHARSET		= null;

	private static String			INFO_XML_OUT_CHARSET	= null;

	public static Object			lock					= new Object();

	// private static Thread inner_monitor = null;
	// private static boolean inner_monitor_alive = false;

	private static Hashtable		SPOOL_INFO_HASH			= null;

	public static String			name					= "SpoolInfoManager";

	private static SpoolInfoManager	INNER_MANAGER			= null;

	/* 병렬발송시 서버별 최종 spool xml 파일 명 */
	private static Hashtable		PARALLEL_SVR_SPOOLINFO	= null;

	static {
		INNER_MANAGER = new SpoolInfoManager();
		INNER_MANAGER.start();
	}

	public synchronized static void unload() throws Exception {
		INNER_MANAGER.close();
	}

	public synchronized static void init(Object prop) throws Exception {

		Properties tmp = (Properties) prop;

		INFO_SAVE_BASE_DIR = tmp.getProperty("base.dir");
		INFO_SAVE_BASE_URL = tmp.getProperty("base.url");
		INFO_SAVE_FILE_EXT = tmp.getProperty("file.ext", ".xml");
		INFO_DTD_LOCATION = tmp.getProperty("dtd.location");
		INFO_XML_CHARSET = tmp.getProperty("info.xml.charset", "euc-kr");
		INFO_XML_OUT_CHARSET = tmp.getProperty("info.out.charset", "KSC5601");

		if( INFO_SAVE_BASE_URL != null && !INFO_SAVE_BASE_URL.trim().equals("") ) {
			// DTD 파일을 INFO_SAVE_BASE_DIR에 복사하고 이쪽을 DTD Location으로 잡는다.
			String xmlDtd = FileElement.getFileBody(INFO_DTD_LOCATION);
			int i = INFO_DTD_LOCATION.lastIndexOf("/");
			int j = INFO_DTD_LOCATION.lastIndexOf("\\");
			String xmlDtdFileOnlyName = INFO_DTD_LOCATION.substring(Math.max(i, j) + 1);
			FileElement.toFile(INFO_SAVE_BASE_DIR + File.separator + xmlDtdFileOnlyName, xmlDtd);

			SpoolInfo.setDTDLocation(INFO_SAVE_BASE_URL + "/" + xmlDtdFileOnlyName);
		}
		else {
			SpoolInfo.setDTDLocation(INFO_DTD_LOCATION);
		}

		SpoolInfo.setEncoding(INFO_XML_CHARSET);
		SpoolInfo.setFileEncoding(INFO_XML_OUT_CHARSET);

		if( INFO_SAVE_BASE_DIR == null ) {
			throw new RuntimeException("SpoolBaseDirectory  parameter is not set!!!");
		}
	}

	/**
	 * 모니터링 쓰레드를종료하고 자원을 반환한다.
	 */
	public synchronized static void destroy() throws Exception {

		SPOOL_INFO_HASH.clear();
		PARALLEL_SVR_SPOOLINFO.clear();
		//inner_monitor_alive=false;
	}

	/**
	 * Creates new SpoolInfoManager <br>
	 * 내부적으로 등록된 SpoolInfo를 모니터링하는 내부 Thread를 생성한다.
	 */
	private SpoolInfoManager() {

		super(CHECK_INTERVAL);
		SPOOL_INFO_HASH = new Hashtable();
		PARALLEL_SVR_SPOOLINFO = new Hashtable();
	}

	protected void check() throws Exception {

		Enumeration ENUM = SPOOL_INFO_HASH.elements();

		SpoolInfo tmpInfo = null;
		if (log.isDebugEnabled()) 
			log.debug("SpoolInfoManager Scan start...");
		while (ENUM.hasMoreElements()) {
			tmpInfo = (SpoolInfo) ENUM.nextElement();

			if( tmpInfo.expire() ) {
				synchronized (lock) {
					log.info(tmpInfo.getID() + " is removed from SpoolManager");
					SPOOL_INFO_HASH.remove(tmpInfo.getID());
				}
			}
		}
		if (log.isDebugEnabled()) 
			log.debug("SpoolInfoManager Scan end...");
	}

	/**
	 * 외부에서 호출되며 SpoolInfo를 등록한다. 기본 스풀 파일 형식을 따른다. ( id_sendState.xml )
	 * 
	 * @param spool
	 * @throws Exception
	 */
	public synchronized static void registSpoolInfo(SpoolInfo spool) throws Exception {
		StringBuilder sb = new StringBuilder();
		sb.append(spool.getID()).append("_").append(spool.getSendState());
		sb.append("_").append(spool.getSpoolDivideScheduleSeq());
		registSpoolInfo(spool, sb.toString());
	}

	public synchronized static void registSpoolInfo(SpoolInfo spool, String file_name) throws Exception {

		String store_file_name = FileElement.CheckSubDirectory(INFO_SAVE_BASE_DIR, spool.getID().substring(0, 8)) + "/";
		file_name += INFO_SAVE_FILE_EXT;

		File __OUT_FILE__ = new File(store_file_name + file_name);

		if( __OUT_FILE__.exists() ) {
			boolean fileDel = __OUT_FILE__.delete();
			if(!fileDel) log.error("File deletion failed");
		}

		SpoolInfo.toXmlFile(store_file_name + file_name, spool);
		SpoolInfo returnValue = SpoolInfo.XmlToSpoolInfo(store_file_name + file_name);

		//xml 파일 위치를 spool info 객체에 담는다.
		String path = store_file_name + file_name;
		if( INFO_SAVE_BASE_URL != null ) {
			int i = INFO_SAVE_BASE_DIR.length();
			path = INFO_SAVE_BASE_URL + store_file_name.substring(i) + file_name;
		}
		returnValue.setSpoolInfoPath(path);

		SpoolInfo.toXmlFile(store_file_name + "copy_" + file_name, returnValue);

		putSpoolInfo(returnValue, false);

		log.info(spool.getID() + " is registered in SpoolManager");
	}

	/**
	 * 외부에서 호출되며 SpoolInfo를 등록한다. 스풀 파일 이름을 정할 수 있다. 단 확장자는 등록된 것을 사용하기 때문에 확장자를 빼고 넣어줘야 한다.
	 * 
	 * @param spool
	 *        등록할 SpoolInfo
	 * @throws Exception
	 *         등록에러
	 */
	public synchronized static void registParallelSpoolInfo(SpoolInfo spool, String curSvrName) throws Exception {

		String file_name = spool.getID() + "_" + spool.getSendState() + "_" + curSvrName;

		String store_file_name = FileElement.CheckSubDirectory(INFO_SAVE_BASE_DIR, spool.getID().substring(0, 8)) + "/";
		file_name += INFO_SAVE_FILE_EXT;

		File __OUT_FILE__ = new File(store_file_name + file_name);

		if( __OUT_FILE__.exists() ) {
			boolean fileDel = __OUT_FILE__.delete();
			if(!fileDel) log.error("File deletion failed");
		}

		SpoolInfo.toXmlParallelFile(store_file_name + file_name, spool, curSvrName);
		SpoolInfo returnValue = SpoolInfo.XmlToSpoolInfo(store_file_name + file_name);

		//xml 파일 위치를 spool info 객체에 담는다.
		String path = store_file_name + file_name;
		if( INFO_SAVE_BASE_URL != null ) {
			int i = INFO_SAVE_BASE_DIR.length();
			path = INFO_SAVE_BASE_URL + store_file_name.substring(i) + file_name;
		}
		returnValue.setSpoolInfoPath(path); //	이 변수에는 사실상 병렬발송 스풀에서는 필요없으나 그냥 둬도 무관해서 그냥 둔다(혹시 쓰일수도 있으므로)..
		PARALLEL_SVR_SPOOLINFO.put(spool.getID() + "|" + curSvrName, path); //	병렬발송 서버별로 스풀 이름 셋팅..

		SpoolInfo.toXmlFile(store_file_name + "copy_" + file_name, returnValue);

		putSpoolInfo(returnValue, false);

		log.info(spool.getID() + " is registered in SpoolManager");
	}

	/**
	 * 외부에서 호출되며 SpoolInfo를 등록한다.
	 * 
	 * @param spool
	 *        등록할 SpoolInfo
	 * @throws Exception
	 *         등록에러
	 * @deprecated use putSpoolInfo(spool,false) instead
	 */
	public synchronized static void registTestSpoolInfo(SpoolInfo spool) throws Exception {

		if (log.isDebugEnabled()) 
			log.debug("REGISTRATION ID => " + spool.getID());
		putSpoolInfo(spool, false);
		log.info(spool.getID() + " is registered in SpoolManager");
	}

	public synchronized static void putSpoolInfo(SpoolInfo spool, boolean check) throws Exception {

		spool.execute();
		synchronized (lock) {
			if( check ) {
				if( SPOOL_INFO_HASH.containsKey(spool.getID()) ) {
					return;
				}
			}
			SPOOL_INFO_HASH.put(spool.getID(), spool);
		}
	}

	/**
	 * 지정된 아이디에 해당하는 SpoolInfo를 반환한다. <br>
	 * 만일 내부 Hash에 보관하지 않고 있다면 저장 디렉토리에서 찾아서 반환한다. <br>
	 * 그래도 없다면 null을 반환한다.
	 * 
	 * @param key
	 *        SpoolInfo의 ID를 지정한다.
	 * @throws Exception
	 *         변환에러 발생
	 * @return 지정된 SpoolInfo <br>
	 *         만일 저당되어 있지 않는 SpoolInfo라면 null을 반환한다.
	 */
	public static SpoolInfo getSpoolInfo(String key) throws Exception {

		if( key == null || key.length() < 2 ) {
			log.error(INNER_MANAGER.getName(), "INVALID KEY[".concat(key).concat("]"));
			return null;
		}

		if (log.isDebugEnabled()) 
			log.debug(INNER_MANAGER.getName(), "REQUEST ID => " + key);

		if (log.isDebugEnabled()) 
			log.debug(INNER_MANAGER.getName(), "CHECK ID => " + key);
		if( SPOOL_INFO_HASH.containsKey(key) ) {
			if (log.isDebugEnabled()) 
				log.debug(INNER_MANAGER.getName(), "ALREADY CONTAIN => " + key);
			SpoolInfo returnValue = (SpoolInfo) SPOOL_INFO_HASH.get(key);
			returnValue.execute();
			return returnValue;
		}

		/*
		 * < < < < File Scan 의 Rule이 다음과 같이 변화했음 2004.12.22 >>>> base dir은 메모리에 (eMsSystem) "spool.info.save.dir" 값으로 셋팅되어 있고 그 값 디렉토리를 뒤져서 key로 시작하는 파일을 가져온다.
		 */

		String scan_file_name = null;

		try {
			File f = new File(FileElement.CheckSubDirectory(eMsSystem.getProperty("spool.info.save.dir"), key.substring(0, 8)));
			String[] fNames = f.list();

			for (int u = 0; u < fNames.length; u++) {
				if( (fNames[u].startsWith(key+"_") || fNames[u].startsWith(key+".")) && fNames[u].endsWith(".xml") ) {
					//bingo
					scan_file_name = FileElement.CheckSubDirectory(eMsSystem.getProperty("spool.info.save.dir"), key.substring(0, 8)) + File.separator + fNames[u];
					break;
				}
			}
		}
		catch(Exception ex) {
			// 없는 셈 치자.
			return null;
		}

		// 못 찾은 경우는 무시하고 넘어간다.
		if( scan_file_name == null )
			return null;

		File tmpFile = new File(scan_file_name);

		/**
		 * 파일이 존재할 경우 파일을 읽어서 반환하고 파일도 없으면 널을 반환한다.
		 */
		if( tmpFile.exists() ) {
			SpoolInfo returnValue = SpoolInfo.XmlToSpoolInfo(scan_file_name);
			synchronized (lock) {
				SPOOL_INFO_HASH.put(key, returnValue);
			}

			returnValue.execute();
			return returnValue;
		}

		log.info(INNER_MANAGER.getName(), "SCAN FILE NOT EXISTS SO RETURN NULL...");
		return null;
	}

	/*
	 * add 2002.10.18 MakeSendInfoTask의 사용의 범위가 작아졌기 때문에 삭제를 하고 남은 기능을 MailSendTask로 이관을 하면서 컨텐트의 생성유무를 체그하기 위하여 추가
	 */
	/**
	 * 지정된 아이디의 SpoolInfo가 존재 하는지를 체크한다.
	 * 
	 * @param key
	 *        검사할 ID
	 * @return true : 존재함 <br>
	 *         false : 존재하지 않음
	 */
	public static boolean checkSpoolInfo(String key) {
		synchronized (lock) {
			if( SPOOL_INFO_HASH.containsKey(key) ) {
				return true;
			}
		}
		try {
			String scan_file_name = null;

			boolean exist = false;

			try {
				File f = new File(FileElement.CheckSubDirectory(eMsSystem.getProperty("spool.info.save.dir"), key.substring(0, 8)));
				String[] fNames = f.list();

				for (int u = 0; u < fNames.length; u++) {
					if( ( fNames[u].startsWith(key+"_") || fNames[u].startsWith(key+".")) && fNames[u].endsWith(".xml") ) {
						//bingo
						scan_file_name = FileElement.CheckSubDirectory(eMsSystem.getProperty("spool.info.save.dir"), key.substring(0, 8)) + File.separator + fNames[u];
						exist = true;
						break;
					}
				}
			}
			catch(Exception ex) {
				// 없는 셈 치자.
				return false;
			}

			/**
			 * 파일이 존재할 경우 파일을 읽어서 반환하고 파일도 없으면 널을 반환한다.
			 */
			if( exist ) {
				SpoolInfo returnValue = SpoolInfo.XmlToSpoolInfo(scan_file_name);
				synchronized (lock) {
					SPOOL_INFO_HASH.put(key, returnValue);
				}

				returnValue.execute();

				return true;
			}
		}
		catch(Exception e) {
			log.error("SpoolInfoManager", e);
		}

		return false;
	}

	/**
	 * @return Returns the pARALLEL_SVR_SPOOLINFO.
	 */
	public static String getParallelSvrSpoolInfo(String key) {
		return (String) PARALLEL_SVR_SPOOLINFO.get(key);
	}

	/**
	 * @param parallel_svr_spoolinfo
	 *        The pARALLEL_SVR_SPOOLINFO to set.
	 */
	public static void setParallelSvrSpoolInfo(String key, String path) {
		PARALLEL_SVR_SPOOLINFO.put(key, path);
	}

	
	public static void main(String[] args) throws Exception  {
		
		SpoolInfoManager sm = new SpoolInfoManager();
		Properties prop = new Properties();
		
		prop.setProperty("base.dir", "C://asbc/spool_infos");
		prop.setProperty("file.ext", ".xml");
		prop.setProperty("dtd.location", "../../../../conf/venus_spoolinfo.dtd");
		prop.setProperty("info.xml.charset", "UTF-8");
		prop.setProperty("info.out.charset", "UTF-8");
		
		sm.init(prop);
	}
	
}
