/*
 * BufferedObjectPool.java
 *
 * Created on 2003년 11월 20일 (목), 오전 10:21
 */

package pluto.util.recycle;

import java.util.LinkedList;

import lombok.extern.slf4j.Slf4j;
import pluto.log.LogChannel;
import pluto.schedule.ScheduledMonitor;
import pluto.util.FIFOBuffer;

/**
 * 
 * @author t??
 */
@Slf4j
public abstract class BufferedObjectPool extends ScheduledMonitor implements BufferBin {
	

	protected static final boolean	POOL_DEBUG						= false;

	protected static final boolean	INOUT_DEBUG						= false;

	protected volatile boolean		FLUSH_LOCKING_FLAG				= false;

	/**
	 * 집어넣다가 안넣어지면 붙잡을 녀석
	 */
	protected Object				LOCK_OF_PRIORITY_INPUT					= new Object();
	
	/**
	 * 우선순위 발송을 위해 붙잡을 녀석 
	 */
	protected Object				LOCK_OF_NORMAL_INPUT					= new Object();

	/**
	 * 빼다가 안빠지면 붙잡을 녀석
	 */
	protected Object				LOCK_OF_OUTPUT					= new Object();

	protected LogChannel			LOG_CHANNEL_INSTANCE			= null;

	protected FIFOBuffer			STACK_OF_BUFFERED_OBJECT		= null;
	
	//PRIORITY TMS3.2 by Taez
	protected FIFOBuffer			STACK_PRIORITY_OF_BUFFERED_OBJECT		= null;

	protected LinkedList			WORKER_LIST						= null;

	/**
	 * Bufferable의 갯수에 관한 파라미터들
	 */
	protected int					CAPACITY						= 0;
	//resetsize
	protected int					RESET_SIZE						= 0;

	protected int					LEVEL_OF_BUFFERABLE				= 0;

	protected int					INDEX_OF_BUFFERABLE				= 0;

	protected int					NAME_INDEX_OF_CREATE_BUFFERABLE	= 0;

	protected long					LAST_TOUCH_TIME					= -1;

	protected long					LAST_FLUSH_TIME					= -1;

	/**
	 * flushAll() 할때 이녀석을 true로 하고 호출을 한다.
	 */
	protected boolean				NEXT_FLUSH_FLAG					= false;

	/**
	 * flush() 도중에 또 들어오는 것이 있으면 flush를 멈추어야 하기 때문에 push() 할때 이녀석을 true로 하고 flush()에서는 notifyAll() 하고 난다음에 이녀석을 다시 false로 해준다.
	 */
	protected boolean				BUFFER_IS_MODIFIED				= false;

	public BufferedObjectPool() {
		throw new RuntimeException("NOT IMPLEMENT CONSTRUCTOR");
	}

	/** Creates a new instance of BufferedObjectPool */
	protected BufferedObjectPool(long flush_check_interval, String name) {
		super(flush_check_interval, name);
		log.info("BUFFEROBJECTPOOL NAME : {}",name);
		this.CAPACITY = 0;
		this.LEVEL_OF_BUFFERABLE = 0;
		this.STACK_OF_BUFFERED_OBJECT = new FIFOBuffer(10);
		this.STACK_PRIORITY_OF_BUFFERED_OBJECT = new FIFOBuffer(10);
		this.WORKER_LIST = new LinkedList();
	}

	/**
	 * 상속받은 Class에서 실제적으로 모니터링하는 로직
	 */
	protected void check() throws Exception {
		if( NEXT_FLUSH_FLAG || System.currentTimeMillis() > LAST_FLUSH_TIME + 5 * 60 * 1000L ) {
			flush();
			LAST_FLUSH_TIME = System.currentTimeMillis();
			NEXT_FLUSH_FLAG = false;
		}
	}

	public synchronized void registFlush() {
		log("CALL registFlush() method");
		NEXT_FLUSH_FLAG = true;
		interruptInnerThread();
	}

	/**
	 * 재활용하지않고 그냥 버려버린다.
	 */
	public void destroy(Bufferable src) {
		log.info("{} is remove",src.getName());
		synchronized (this.WORKER_LIST) {
			this.WORKER_LIST.remove(src);
		}
		try {
			resetWorkerSize(this.CAPACITY);
		}
		catch(Exception e) {
			log.debug("destroy reSize Error: {}" , e.toString());
		}
	}

	/**
	 * 통 사이즈를 반환한다.
	 */
	public synchronized int getWorkerSize() {
		return this.CAPACITY;
	}

	/**
	 * 통 사이즈를 반환한다.
	 */
	public int getBufferSize() {
		log.debug("CALL getBufferSize()");
		int totalSize = 0;
		totalSize += this.STACK_PRIORITY_OF_BUFFERED_OBJECT.size();
		totalSize += this.STACK_OF_BUFFERED_OBJECT.size();
		return totalSize;
	}

	/**
	 * 버퍼 사이즈 조정
	 */
	public synchronized void resetBufferSize(int size) {
		this.STACK_OF_BUFFERED_OBJECT.reSize(size);
		this.STACK_PRIORITY_OF_BUFFERED_OBJECT.reSize(size);
	}

	protected synchronized void setWorkerSize(int size) throws Exception {

		log("CALL setWorkerSize()");

		this.CAPACITY = size;  //용량지정 <TARGET name="SMTP AGENT POOL" -> zone>size.value -->

		Bufferable target = null;

		while (this.CAPACITY > this.WORKER_LIST.size()) {
			
			//BufferedAgentPool.create(int seq, int level)
			target = create(LEVEL_OF_BUFFERABLE++, INDEX_OF_BUFFERABLE++);
			this.WORKER_LIST.addLast(target);
			target.execute();
		}
		
		log.info("this.WORKER_LIST toString     : {}", this.WORKER_LIST.toString());
		log.info("this.WORKER_LIST addressValue : {}", this.WORKER_LIST);
		
		return;
	}
	//resetsize
	protected synchronized void setWorkerSize(int size, int rsize) throws Exception {
		this.RESET_SIZE = rsize;
		setWorkerSize(size);
	}
	
	/*
	protected synchronized void setWorkerSize(int size, int rsetsize) throws Exception {

		log("CALL setWorkerSize()");

		this.CAPACITY = rsetsize;

		Bufferable target = null;

		while (size > this.WORKER_LIST.size()) {

			target = create(LEVEL_OF_BUFFERABLE++, INDEX_OF_BUFFERABLE++);
			this.WORKER_LIST.addLast(target);
			target.execute();
		}

		return;
	}
*/
	/**
	 * 통 사이즈를 재조정한다.
	 */
	public synchronized void resetWorkerSize(int size) throws Exception {
		log.info("CALL resetWorkerSize()");

		this.CAPACITY = size;

		if( this.CAPACITY == this.WORKER_LIST.size() )
			return;

		flush();

		if( this.CAPACITY < this.WORKER_LIST.size() ) {
			Bufferable target = null;
			while (this.CAPACITY < this.WORKER_LIST.size()) {
				target = (Bufferable) this.WORKER_LIST.removeLast();
				target.setEnd();
			}

			this.notifyAll();

			return;
		}

		if( this.CAPACITY > this.WORKER_LIST.size() ) {
			Bufferable target = null;
			while (this.CAPACITY > this.WORKER_LIST.size()) {
				target = create(LEVEL_OF_BUFFERABLE++, INDEX_OF_BUFFERABLE++);

				this.WORKER_LIST.addLast(target);

				target.execute();
			}

			return;
		}
	}

	/**
	 * 버퍼링된것을 끄집어 낸다. 발송하는 Getter에서만 호출이 된다.
	 */
	public Object popup() {
		Object returnValue = null;
		int wait_count = 0;
		while (true) {
			if( POOL_DEBUG )
				log("popup Check IN");

			/**
			 * 비어 있을 경우 계속해서 기다리는지 아니면 그냥 null을 보내는지 결정한다.
			 */
			if( (returnValue = this.STACK_PRIORITY_OF_BUFFERED_OBJECT.pop()) == null ) {
				/* 우선순위 발송 체크를 위해서 추가예정 / 속도에 영향이 있어 주석처리
				 * try {
					Thread.sleep(2000);
				} catch (InterruptedException e) {
					e.printStackTrace();
				}*/
				returnValue = this.STACK_OF_BUFFERED_OBJECT.pop();
			}
			
			synchronized (this.LOCK_OF_PRIORITY_INPUT) {
				if( POOL_DEBUG )
					log("call LOCK_OF_PRIORITY_INPUT.notify()");
				this.LOCK_OF_PRIORITY_INPUT.notifyAll();
			}
			synchronized (this.LOCK_OF_NORMAL_INPUT) {
				if( POOL_DEBUG )
					log("call LOCK_OF_NORMAL_INPUT.notify()");
				this.LOCK_OF_NORMAL_INPUT.notifyAll();
			}

			if( returnValue != null ) {
				/**
				 * Flush 할때를 잡아야 하니까..
				 */
				LAST_TOUCH_TIME = System.currentTimeMillis();

				if( INOUT_DEBUG )
					log("popup success level=>".concat(String.valueOf(this.STACK_OF_BUFFERED_OBJECT.size())));
				return returnValue;
			}

			if( POOL_DEBUG )
				log("INTO popup WAIT.");

			if( INOUT_DEBUG )
				log("popup empty wait level=>".concat(String.valueOf(this.STACK_OF_BUFFERED_OBJECT.size())));
			if( POOL_DEBUG )
				log("call LOCK_OF_OUTPUT.wait()");
			synchronized (this.LOCK_OF_OUTPUT) {
				if( wait_count < 10 ) {
					wait_count++;
					try {
						this.LOCK_OF_OUTPUT.wait(5000);
					}
					catch(Exception e) {
					}
				}
				else {
					try {
						this.LOCK_OF_OUTPUT.wait();
					}
					catch(Exception e) {
					}
				}
			}
			if( POOL_DEBUG )
				log("OUT popup WAIT.");
		}
	}

	/**
	 * 현재 버퍼링 된것이 있는지를 점검한다. 없으면 바로 NULL을 반환한다. 널을 반환 받으면 그냥 popup()을 호출하여 그 안에서 기다리도록 한다.
	 */
	public Object popupNoWait() {
		Object returnValue = null;
		if( POOL_DEBUG )
			log("popupNoWait Check IN");
		/**
		 * 비어 있을 경우 계속해서 기다리는지 아니면 그냥 null을 보내는지 결정한다.
		 */
		if( (returnValue = this.STACK_PRIORITY_OF_BUFFERED_OBJECT.pop()) != null ) {
			LAST_TOUCH_TIME = System.currentTimeMillis();
			if( INOUT_DEBUG )
				log("popupNoWait  success level=>".concat(String.valueOf(this.STACK_PRIORITY_OF_BUFFERED_OBJECT.size())));
		}else if( (returnValue = this.STACK_OF_BUFFERED_OBJECT.pop()) != null ) {
			/**
			 * Flush 할때를 잡아야 하니까..
			 */
			LAST_TOUCH_TIME = System.currentTimeMillis();

			if( INOUT_DEBUG )
				log("popupNoWait  success level=>".concat(String.valueOf(this.STACK_OF_BUFFERED_OBJECT.size())));
		}
		else {
			if( POOL_DEBUG )
				log("NO BUFFERED_OBJECT");
		}
		synchronized (this.LOCK_OF_PRIORITY_INPUT) {
			if( POOL_DEBUG )
				log("call LOCK_OF_INPUT.notify()");
			this.LOCK_OF_PRIORITY_INPUT.notifyAll();
		}
		synchronized (this.LOCK_OF_NORMAL_INPUT) {
			if( POOL_DEBUG )
				log("call LOCK_OF_NORMAL_INPUT.notify()");
			this.LOCK_OF_NORMAL_INPUT.notifyAll();
		}
		return returnValue;
	}
	
	/**
	 * 일반 발송 버퍼링 할것을 넣는다.
	 */
	public void normalPush(Object src) {
		if( src == null ){
			return;
		}
		int wait_count = 0;
		boolean success_flag = false;
		while (true) {
			if( POOL_DEBUG )
				log("push Check IN");
			
			/**
			 * Flush 할때를 잡아야 하니까..
			 */
			LAST_TOUCH_TIME = System.currentTimeMillis();

			success_flag = this.STACK_OF_BUFFERED_OBJECT.push(src);
			synchronized (this.LOCK_OF_OUTPUT) {
				if( POOL_DEBUG )
					log("call LOCK_OF_OUTPUT.notify()");
				try {
					this.LOCK_OF_OUTPUT.notifyAll();
				}
				catch(Exception e) {
				}
			}

			if( success_flag ) {
				/**
				 * 버퍼가 변경 되었음을 마킹
				 */
				this.BUFFER_IS_MODIFIED = true;

				if( INOUT_DEBUG )
					log.info("push success level=> {}",String.valueOf(this.STACK_OF_BUFFERED_OBJECT.size()));
				return;
			}

			if( INOUT_DEBUG )
				log.info("push maximum wait level=> {}",String.valueOf(this.STACK_OF_BUFFERED_OBJECT.size()));
			if( POOL_DEBUG )
				log.info("call LOCK_OF_NORMAL_INPUT.wait()");
			synchronized (this.LOCK_OF_NORMAL_INPUT) {
				if( wait_count < 5 ) {
					wait_count++;
					try {
						this.LOCK_OF_NORMAL_INPUT.wait(5000);
					}
					catch(Exception e) {
					}
				}
				else {
					try {
						this.LOCK_OF_NORMAL_INPUT.wait();
					}
					catch(Exception e) {
					}
				}
			}
			if(log.isDebugEnabled()){
				log.debug("OUT WAIT");
			}
		}
	}
	/**
	 * 사용하지 않을 경우 로직을 타지 않도록 설정하기 위해서 서로 분리
	 * @param src
	 */
	public void fastPush(Object src){
		log.info("FAST SEND USE : Y");
		int wait_count = 0;
		boolean success_flag = false;
		while (true) {
			if( POOL_DEBUG )
				log("push Check IN");
			
			/**
			 * Flush 할때를 잡아야 하니까..
			 */
			LAST_TOUCH_TIME = System.currentTimeMillis();

			if(src.toString().indexOf("FASTSEND")>0){
				log.info("PRIORITY : STACK_PRIORITY_OF_BUFFERED_OBJECT");
				success_flag = this.STACK_PRIORITY_OF_BUFFERED_OBJECT.push(src);
			}
			synchronized (this.LOCK_OF_OUTPUT) {
				if( POOL_DEBUG )
					log("call LOCK_OF_OUTPUT.notify()");
				try {
					this.LOCK_OF_OUTPUT.notifyAll();
				}
				catch(Exception e) {
				}
			}

			if( success_flag ) {
				/**
				 * 버퍼가 변경 되었음을 마킹
				 */
				this.BUFFER_IS_MODIFIED = true;

				if( INOUT_DEBUG )
					log.info("push success level=> {}",String.valueOf(this.STACK_PRIORITY_OF_BUFFERED_OBJECT.size()));
				return;
			}

			if( INOUT_DEBUG )
				log.info("push maximum wait level=> {}",String.valueOf(this.STACK_PRIORITY_OF_BUFFERED_OBJECT.size()));
			if( POOL_DEBUG )
				log.info("call LOCK_OF_PRIORITY_INPUT.wait()");
			synchronized (this.LOCK_OF_PRIORITY_INPUT) {
				if( wait_count < 5 ) {
					wait_count++;
					try {
						this.LOCK_OF_PRIORITY_INPUT.wait(5000);
					}
					catch(Exception e) {
					}
				}
				else {
					try {
						this.LOCK_OF_PRIORITY_INPUT.wait();
					}
					catch(Exception e) {
					}
				}
			}
			if(log.isDebugEnabled()){
				log.debug("OUT WAIT");
			}
		}
	}
	public void push(Object src) {
		if( src == null )
			return;
		if(src.toString().indexOf("FASTSEND")>0){
			fastPush(src);
		}else{
			normalPush(src);
		}
	}
	
	@Override
	public void log(String logStr) {
		if (log.isDebugEnabled()) {
			log.debug(logStr);
		}
	}

	protected abstract void flush() throws Exception;

	protected abstract Bufferable create(int seq, int level) throws Exception;
}
