/*
 * AmailHashtable.java
 *
 * Created on 2004년 5월 12일 (수), 오후 12:56
 */

package pluto.util;

import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
/**
 *
 * @author  이상근
 */
public class AmailHashtable implements java.util.Iterator {
	/**
	 * The hash table data.
	 */
	private Entry table[];
	
	/**
	 * The total number of entries in the hash table.
	 */
	private int count;
	
	/**
	 * The table is rehashed when its size exceeds this threshold.  (The
	 * value of this field is (int)(capacity * loadFactor).)
	 *
	 * @serial
	 */
	private int threshold;
	
	/**
	 * The load factor for the hashtable.
	 *
	 * @serial
	 */
	private float loadFactor;
	
	/**
	 * The number of times this Hashtable has been structurally modified
	 * Structural modifications are those that change the number of entries in
	 * the Hashtable or otherwise modify its internal structure (e.g.,
	 * rehash).  This field is used to make iterators on Collection-views of
	 * the Hashtable fail-fast.  (See ConcurrentModificationException).
	 */
	private int modCount = 0;
	
	/**
	 * Constructs a new, empty hashtable with the specified initial
	 * capacity and the specified load factor.
	 *
	 * @param      initialCapacity   the initial capacity of the hashtable.
	 * @param      loadFactor        the load factor of the hashtable.
	 * @exception  IllegalArgumentException  if the initial capacity is less
	 *             than zero, or if the load factor is nonpositive.
	 */
	public AmailHashtable(int initialCapacity, float loadFactor) {
		if (initialCapacity < 0)
			throw new IllegalArgumentException("Illegal Capacity: "+ initialCapacity);
		if (loadFactor <= 0 || Float.isNaN(loadFactor))
			throw new IllegalArgumentException("Illegal Load: "+loadFactor);
		
		if (initialCapacity==0){
			initialCapacity = 1;
		}
		
		this.loadFactor = loadFactor;
		table = new Entry[initialCapacity];
		threshold = (int)(initialCapacity * loadFactor);
	}
	
	/**
	 * Constructs a new, empty hashtable with the specified initial capacity
	 * and default load factor, which is <tt>0.75</tt>.
	 *
	 * @param     initialCapacity   the initial capacity of the hashtable.
	 * @exception IllegalArgumentException if the initial capacity is less
	 *              than zero.
	 */
	public AmailHashtable(int initialCapacity) {
		this(initialCapacity, 0.75f);
	}
	
	/**
	 * Constructs a new, empty hashtable with a default initial capacity (11)
	 * and load factor, which is <tt>0.75</tt>.
	 */
	public AmailHashtable() {
		this(11, 0.75f);
	}
	
	/**
	 * Returns the number of keys in this hashtable.
	 *
	 * @return  the number of keys in this hashtable.
	 */
	public synchronized int size() {
		return count;
	}
	
	/**
	 * Tests if this hashtable maps no keys to values.
	 *
	 * @return  <code>true</code> if this hashtable maps no keys to values;
	 *          <code>false</code> otherwise.
	 */
	public synchronized boolean isEmpty() {
		return count == 0;
	}
	
	/**
	 * Tests if some key maps into the specified value in this hashtable.
	 * This operation is more expensive than the <code>containsKey</code>
	 * method.<p>
	 *
	 * Note that this method is identical in functionality to containsValue,
	 * (which is part of the Map interface in the collections framework).
	 *
	 * @param      value   a value to search for.
	 * @return     <code>true</code> if and only if some key maps to the
	 *             <code>value</code> argument in this hashtable as
	 *             determined by the <tt>equals</tt> method;
	 *             <code>false</code> otherwise.
	 * @exception  NullPointerException  if the value is <code>null</code>.
	 * @see        #containsKey(Object)
	 * @see        #containsValue(Object)
	 * @see	   Map
	 */
	public synchronized boolean containsValue(Object value) {
		if (value == null) {
			throw new NullPointerException();
		}
		
		Entry tab[] = table;
		for (int i = tab.length ; i-- > 0 ;) {
			for (Entry e = tab[i] ; e != null ; e = e.next) {
				if (e.value.equals(value)) {
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * Tests if the specified object is a key in this hashtable.
	 *
	 * @param   key   possible key.
	 * @return  <code>true</code> if and only if the specified object
	 *          is a key in this hashtable, as determined by the
	 *          <tt>equals</tt> method; <code>false</code> otherwise.
	 * @throws  NullPointerException  if the key is <code>null</code>.
	 * @see     #contains(Object)
	 */
	public synchronized boolean containsKey(Object key) {
		Entry tab[] = table;
		int hash = key.hashCode();
		int index = (hash & 0x7FFFFFFF) % tab.length;
		for (Entry e = tab[index] ; e != null ; e = e.next) {
			if ((e.hash == hash) && e.key.equals(key)) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Returns the value to which the specified key is mapped in this hashtable.
	 *
	 * @param   key   a key in the hashtable.
	 * @return  the value to which the key is mapped in this hashtable;
	 *          <code>null</code> if the key is not mapped to any value in
	 *          this hashtable.
	 * @throws  NullPointerException  if the key is <code>null</code>.
	 * @see     #put(Object, Object)
	 */
	public synchronized Object get(Object key) {
		Entry tab[] = table;
		int hash = key.hashCode();
		int index = (hash & 0x7FFFFFFF) % tab.length;
		for (Entry e = tab[index] ; e != null ; e = e.next) {
			if ((e.hash == hash) && e.key.equals(key)) {
				return e.value;
			}
		}
		return null;
	}
	
	/**
	 * Increases the capacity of and internally reorganizes this
	 * hashtable, in order to accommodate and access its entries more
	 * efficiently.  This method is called automatically when the
	 * number of keys in the hashtable exceeds this hashtable's capacity
	 * and load factor.
	 */
	protected void rehash() {
		int oldCapacity = table.length;
		Entry oldMap[] = table;
		
		int newCapacity = oldCapacity * 2 + 1;
		Entry newMap[] = new Entry[newCapacity];
		
		modCount++;
		threshold = (int)(newCapacity * loadFactor);
		table = newMap;
		
		for (int i = oldCapacity ; i-- > 0 ;) {
			for (Entry old = oldMap[i] ; old != null ; ) {
				Entry e = old;
				old = old.next;
				
				int index = (e.hash & 0x7FFFFFFF) % newCapacity;
				e.next = newMap[index];
				newMap[index] = e;
			}
		}
	}
	
	/**
	 * Maps the specified <code>key</code> to the specified
	 * <code>value</code> in this hashtable. Neither the key nor the
	 * value can be <code>null</code>. <p>
	 *
	 * The value can be retrieved by calling the <code>get</code> method
	 * with a key that is equal to the original key.
	 *
	 * @param      key     the hashtable key.
	 * @param      value   the value.
	 * @return     the previous value of the specified key in this hashtable,
	 *             or <code>null</code> if it did not have one.
	 * @exception  NullPointerException  if the key or value is
	 *               <code>null</code>.
	 * @see     Object#equals(Object)
	 * @see     #get(Object)
	 */
	public synchronized Object put(Object key, Object value) {
		// Make sure the value is not null
		if (value == null) {
			throw new NullPointerException();
		}
		
		// Makes sure the key is not already in the hashtable.
		Entry tab[] = table;
		int hash = key.hashCode();
		int index = (hash & 0x7FFFFFFF) % tab.length;
		for (Entry e = tab[index] ; e != null ; e = e.next) {
			if ((e.hash == hash) && e.key.equals(key)) {
				Object old = e.value;
				e.value = value;
				return old;
			}
		}
		
		modCount++;
		if (count >= threshold) {
			// Rehash the table if the threshold is exceeded
			rehash();
			
			tab = table;
			index = (hash & 0x7FFFFFFF) % tab.length;
		}
		
		// Creates the new entry.
		Entry e = new Entry( hash, key, value, tab[index] );
		tab[index] = e;
		count++;
		return null;
	}
	
	/**
	 * Removes the key (and its corresponding value) from this
	 * hashtable. This method does nothing if the key is not in the hashtable.
	 *
	 * @param   key   the key that needs to be removed.
	 * @return  the value to which the key had been mapped in this hashtable,
	 *          or <code>null</code> if the key did not have a mapping.
	 * @throws  NullPointerException  if the key is <code>null</code>.
	 */
	public synchronized Object remove(Object key) {
		int hash = key.hashCode();
		int index = (hash & 0x7FFFFFFF) % table.length;
		for (Entry e = table[index], prev = null ; e != null ; prev = e, e = e.next) {
			if ((e.hash == hash) && e.key.equals(key)) {
				modCount++;
				if (prev != null) {
					prev.next = e.next;
				} else {
					table[index] = e.next;
				}
				count--;
				Object oldValue = e.value;
				e.value = null;
				return oldValue;
			}
		}
		return null;
	}
	
	/**
	 * Clears this hashtable so that it contains no keys.
	 */
	public synchronized void clear() {
		Entry tab[] = table;
		modCount++;
		for (int index = tab.length; --index >= 0; )
			tab[index] = null;
		count = 0;
	}
	
	/**
	 * Returns a string representation of this <tt>Hashtable</tt> object
	 * in the form of a set of entries, enclosed in braces and separated
	 * by the ASCII characters "<tt>,&nbsp;</tt>" (comma and space). Each
	 * entry is rendered as the key, an equals sign <tt>=</tt>, and the
	 * associated element, where the <tt>toString</tt> method is used to
	 * convert the key and element to strings. <p>Overrides to
	 * <tt>toString</tt> method of <tt>Object</tt>.
	 *
	 * @return  a string representation of this hashtable.
	 */
	public synchronized String toString() {
		int max = size() - 1;
		StringBuffer buf = null;
		try{
			buf = new StringBuffer(1024);
			Iterator it = iterator();
			
			buf.append("{");
			for (int i = 0; i <= max; i++) {
				Map.Entry e = (Map.Entry) (it.next());
				Object key = e.getKey();
				Object value = e.getValue();
				buf.append( key   == this ? "(this Map)" : key.toString() );
				buf.append( "=" );
				buf.append( value == this ? "(this Map)" : value.toString() );
				
				if (i < max)
					buf.append(", ");
			}
			buf.append("}");
			return buf.toString();
		}
		catch( RuntimeException e ){
			throw e;
		}
		finally{
			buf = null;
		}
	}
	
	
	// Views
	
	// Comparison and hashing
	
	/**
	 * Returns the hash code value for this Map as per the definition in the
	 * Map interface.
	 *
	 * @see Map#hashCode()
	 * @since 1.2
	 */
	@Override
	public synchronized int hashCode() {
		/*
		 * This code detects the recursion caused by computing the hash code
		 * of a self-referential hash table and prevents the stack overflow
		 * that would otherwise result.  This allows certain 1.1-era
		 * applets with self-referential hash tables to work.  This code
		 * abuses the loadFactor field to do double-duty as a hashCode
		 * in progress flag, so as not to worsen the space performance.
		 * A negative load factor indicates that hash code computation is
		 * in progress.
		 */
		int h = 0;
		if (count == 0 || loadFactor < 0)
			return h;  // Returns zero
		
		loadFactor = -loadFactor;  // Mark hashCode computation in progress
		Entry tab[] = table;
		for (int i = 0; i < tab.length; i++)
			for (Entry e = tab[i]; e != null; e = e.next)
				h += e.key.hashCode() ^ e.value.hashCode();
		loadFactor = -loadFactor;  // Mark hashCode computation complete
		
		return h;
	}
	
	private int current_check_index = -1;
	
	private Entry IteratorNextEntry = null;
	
	public Iterator iterator(){
		current_check_index = 0;
		IteratorNextEntry = null;
		return this;
	}
	
	// Iterator Implements Section
	public synchronized boolean hasNext() {
		if( IteratorNextEntry == null ){
			IteratorNextEntry = table[ current_check_index ];
		}
		else if( IteratorNextEntry.next != null ){
			IteratorNextEntry = IteratorNextEntry.next;
		}
		else{
			current_check_index++;
			IteratorNextEntry = table[ current_check_index ];
		}
		
		while( true ){
			if( IteratorNextEntry != null ){
				break;
			}
			current_check_index++;
			if( current_check_index < table.length )
				IteratorNextEntry = table[ current_check_index ];
			else
				break;
		}
		
		return IteratorNextEntry != null;
	}
	
	public Object next() {
		if(!hasNext()) throw new NoSuchElementException();
		return IteratorNextEntry;
	}
	
	public void remove() {
		for (Entry e = IteratorNextEntry, prev = null ; e != null ; prev = e, e = e.next) {
			if ((e.hash == IteratorNextEntry.getKey().hashCode() ) && e.key.equals(IteratorNextEntry.getKey())) {
				modCount++;
				if (prev != null) {
					prev.next = e.next;
				} else {
					table[current_check_index] = e.next;
				}
				count--;
				// Object oldValue = e.value;
				e.value = null;
				return;
			}
		}
	}
	// Iterator implements Section End
	/**
	 * Hashtable collision list.
	 */
	private static class Entry implements KeyValueEntry{
		int hash;
		Object key;
		Object value;
		Entry next;
		
		protected Entry(int hash, Object key, Object value , Entry next) {
			this.hash = hash;
			this.key = key;
			this.value = value;
			this.next = next;
		}
		
		protected Object clone() {
			return new Entry(hash, key, value, (next==null ? null : (Entry)next.clone()));
		}
		
		// Map.Entry Ops
		
		public Object getKey() {
			return key;
		}
		
		public Object getValue() {
			return value;
		}
		
		public Object setValue(Object value) {
			if (value == null)
				throw new NullPointerException();
			
			Object oldValue = this.value;
			this.value = value;
			return oldValue;
		}
		
		public boolean equals(Object o) {
			if (!(o instanceof Map.Entry))
				return false;
			Map.Entry e = (Map.Entry)o;
			
			return (key==null ? e.getKey()==null : key.equals(e.getKey())) && (value==null ? e.getValue()==null : value.equals(e.getValue()));
		}
		
		public int hashCode() {
			return hash ^ (value==null ? 0 : value.hashCode());
		}
		
		public String toString() {
			return key.toString()+"="+value.toString();
		}
	}
}
