/*
 * TaskManager.java
 *
 * Created on 2003년 3월 5일 수, 오후 9:31
 */

package pluto.schedule;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;

import lombok.extern.slf4j.Slf4j;
import pluto.lang.eMsLocale;

/**
 * 수행되는 각각의 Task를 관리하고 실행이 필요한 Task에 대해서는 <br>
 * TaskWorker를 생성하여 Task를 실행한다.
 * 
 * @author Administrator
 * @version
 */
@Slf4j
public class TaskManager extends ScheduledMonitor {

	public static ThreadGroup					taskThreadGroup					= new ThreadGroup(eMsLocale.EMS_ROOT_THREAD_GROUP, "TaskThreadGroup");

	public static final String					KEY_WORKER_SIZE					= "worker.size";

	public static final String					KEY_CHECK_INTERVAL				= "check.interval";
	
	public static int							TASK_THREAD_NUM					= 0;

	public static final int						DEFAULT_WORKER_SIZE				= 10;

	public static final long					DEFAULT_CHECK_INTERVAL			= 30 * 1000L;

	private static Object						__INNER_LOCK__					= new Object();

	private static boolean						init_flag						= false;

	private static NoIDTaskRegistException		__NO_ID_EXCEPTION__				= new NoIDTaskRegistException();

	/**
	 * 초기화 파라미터
	 * 
	 * <PRE>
	 * 
	 * &lt;TARGET name="TASK MANAGER"&gt; &nbsp;&nbsp;&nbsp;&nbsp;&lt;class
	 * name="Lib2002.schedule.TaskManager"/&gt; &nbsp;&nbsp;&nbsp;&nbsp;&lt;var
	 * name="worker.size" value="10"/&gt; &nbsp;&nbsp;&nbsp;&nbsp;&lt;var
	 * name="check.interval" value="60000"/&gt; &lt;/TARGET&gt;
	 * 
	 * </PRE>
	 * 
	 * <PRE>
	 * 
	 * worker.size : worker recycling size check.interval : 등록 Task 관리 주기 ( 단위 :
	 * ms )
	 * 
	 * </PRE>
	 * 
	 * @param prop
	 *            초기화 파라미터
	 * @throws Exception
	 *             초기화 에러
	 */
	public synchronized static void init(Object prop) throws Exception {

		Properties tmp = (Properties) prop;

		int size = DEFAULT_WORKER_SIZE;
		long inter = DEFAULT_CHECK_INTERVAL;

		try {
			size = Integer.parseInt(tmp.getProperty(KEY_WORKER_SIZE));
		}
		catch(Exception ignore) {
			log.error("error", ignore);
		}

		try {
			inter = Integer.parseInt(tmp.getProperty(KEY_CHECK_INTERVAL));
		}
		catch(Exception ignore) {
			log.error("error", ignore);
		}
		
		TASK_THREAD_NUM++;

		singleTaskManager = new TaskManager(size, inter);
		singleTaskManager.start();
		init_flag = true;
	}

	/**
	 * 종료될때 호출해줘야 모니터 쓰레드가 죽는다.
	 * 
	 * @param tmp
	 * @throws Exception
	 */
	public static void unload() throws Exception {
		if( singleTaskManager != null ) {
			singleTaskManager.close();
		}
	}

	/**
	 * 매니져에 등록된 Task를 저장
	 */
	private List<Task>			taskList	= new LinkedList<Task>();

	/**
	 * singleton 자신의 레퍼런스를 저장
	 */
	private static TaskManager	singleTaskManager;

	/**
	 * 내부 생성자
	 * 
	 * @param size
	 *            Worker의 recycling size
	 * @param check_interval
	 *            내부 Task 점검 주기
	 */
	private TaskManager(int size, long check_interval) {
		super(check_interval, "TaskManager"+TASK_THREAD_NUM);
		setName("TaskManager"+TASK_THREAD_NUM);
	}

	/**
	 * Task를 등록하고 실행을 한다.
	 * 
	 * @param work
	 *            등록되어 실행될 Task
	 */
	public static final void executeTask(Task work) throws NoIDTaskRegistException, AlreadyRegistTaskException {
		
		if( !init_flag ) {
			throw new RuntimeException("TaskManager is not Initialized... First init TaskManager");
		}

		synchronized (__INNER_LOCK__) {
			String __TARGET_TASK_ID__ = work.getTaskID();			
			log.debug("[CURRENT TASK]="+ __TARGET_TASK_ID__);
			
			if( __TARGET_TASK_ID__ == null )
				throw __NO_ID_EXCEPTION__;
			int i = 0;
			
			for (Iterator iter = singleTaskManager.taskList.iterator(); iter.hasNext();) {
				Task __CHECK_TASK__ = (Task) iter.next();
				i++;
				
				if( __TARGET_TASK_ID__.equals(__CHECK_TASK__.getTaskID()) ) {
					log.info("CURRENT TASK="+__TARGET_TASK_ID__+" ["+i+"]TASK="+__CHECK_TASK__.getTaskID());
					throw new AlreadyRegistTaskException(__TARGET_TASK_ID__ + " is already regist");
				}
			}
			
			singleTaskManager.getTaskList().add(work);
			
			/*// 선행 작업이 있는지를 검사해보고 시작시켜 준다.
			if( singleTaskManager.timeToStart(work) ) {
				TaskWorker tmpWorker = new TaskWorker(taskThreadGroup, work);
				tmpWorker.start();
			} */
		}
	}

	/**
	 * Task를 등록하고 실행을 한다.
	 * 
	 * @param work
	 *            등록되어 실행될 Task
	 */
	public static final boolean isRegistTask(Task work) {
		if( !init_flag ) {
			throw new RuntimeException("TaskManager is not Initialize... First init TaskManager");
		}

		synchronized (__INNER_LOCK__) {
			String __TARGET_TASK_ID__ = work.getTaskID();

			if( __TARGET_TASK_ID__ == null )
				return true;

			for (Iterator iter = singleTaskManager.getTaskList().iterator(); iter.hasNext();) {
				Task __CHECK_TASK__ = (Task) iter.next();

				if( __TARGET_TASK_ID__.equals(__CHECK_TASK__.getTaskID()) ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Task를 등록하고 실행을 한다.
	 * 
	 * @param work
	 *            등록되어 실행될 Task
	 */
	public static final boolean isRegistTask(String __TARGET_TASK_ID__) {
		if( !init_flag ) {
			throw new RuntimeException("TaskManager is not Initialize... First init TaskManager");
		}

		synchronized (__INNER_LOCK__) {
			if( __TARGET_TASK_ID__ == null )
				return true;
			if (log.isDebugEnabled()) 
				if (log.isDebugEnabled()) 
					log.debug("CHECK ID", __TARGET_TASK_ID__);
			for (Iterator iter = singleTaskManager.getTaskList().iterator(); iter.hasNext();) {
				Task __CHECK_TASK__ = (Task) iter.next();
				if (log.isDebugEnabled()) 
					log.debug("REGIST CHECK", __CHECK_TASK__.getTaskID());
				if( __TARGET_TASK_ID__.equals(__CHECK_TASK__.getTaskID()) ) {
					return true;
				}
			}
		}

		return false;
	}

	public static String getTaskNames() {

		StringBuffer buffer = new StringBuffer(128);
		singleTaskManager.getInternalTaskNames(buffer);
		return buffer.toString();
	}

	public void getInternalTaskNames(StringBuffer __BUFFER__) {
		synchronized (__INNER_LOCK__) {
			for (Iterator it = this.getTaskList().iterator(); it.hasNext();) {
				__BUFFER__.append(it.next().toString());
				__BUFFER__.append("\r\n");
			}
		}
	}

	/**
	 * 등록된 Task를 점검한다. 일하고 있는 녀석들의 시간을 체크하여 삐꾸는 바로 죽인다. 또한 선행작업이 모두 끝난 녀석들에 대해서는
	 * START 시켜준다.
	 */
	protected void check() throws Exception {

		synchronized (__INNER_LOCK__) {

			boolean start_flag = false;
			Task task;
			for (Iterator iter = this.taskList.iterator(); iter.hasNext();) {

				task = (Task) iter.next();
				start_flag = task.isStarted();
				if( start_flag ) {
					if( task.isEnd() ) {
						log.debug("{} is destroy ", task.getTaskID());
						task.destroy();
						iter.remove();
						continue;
					}

					if( task.isValidTime() ) {
						log.debug("{} valid time... so execute...", task.getName());
						TaskWorker worker = new TaskWorker(taskThreadGroup, task);
						worker.start();
						continue;
					}

					if (log.isDebugEnabled()) {
						log.debug("{} not valid time... so skip...", task.getName());
					}
				}
				else {
					// 선행작업이 모두 끝났다면 마침내 Task는 시작된다.
					if( timeToStart(task) ) {
						TaskWorker worker = new TaskWorker(taskThreadGroup, task);
						worker.start();
					}
				}
			}
		}
	}

	/**
	 * Loop를 돌면서 task의 depends list가 작업 중인지 검사
	 * 
	 * @param task
	 * @return
	 */
	private boolean timeToStart(Task task) {

		Task tmpTask2;
		boolean check_result = false;
		for (Iterator inner_iter = taskList.iterator(); inner_iter.hasNext();) {
			tmpTask2 = (Task) inner_iter.next();
			check_result = check_result || task.depends(tmpTask2.getTaskID());
			// 안 끝난 게 있다면 더 이상 체크해 볼 필요 없다.
			if( check_result )
				break;
		}
		return !check_result;
	}
	
	public List<Task> getTaskList() {
		return this.taskList;
	}
}
