/*
 * CommonTask.java
 *
 * Created on 2003년 3월 24일 월, 오전 11:08
 */

package pluto.schedule;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import lombok.extern.slf4j.Slf4j;
import pluto.io.eMsFileWriter;
import pluto.lang.Name;
import pluto.util.Cal;

/**
 *
 * @author  Administrator
 * @version
 *
 */
@Slf4j
public abstract class Task extends Name {
	
	
	
	
	/**건별 발송등 한판만 도는거
	 */
	public static final short TYPE_TRANSACTION = 1;
	
	/**일정주기로 체크하는 Task
	 */
	public static final short TYPE_INTERVAL = 2;
	
	/**일정 주기별 실행할때 기본 주기값
	 */
	public static final long DEFAULT_EXECUTE_INTERVAL = 60*1000L;
	
	/**흐름 제어 파라미터
	 */
	protected boolean work_flag = true;
	
	/** 작업이 시작되었는지 반환 
	 */
	protected boolean start_flag = false;
	
	/**다음에 시작할 시간을 정한다.
	 */
	protected long next_execute_time = -1L;
	
	/** 작업이 진행되기 위해 선행되어야 할 작업 아이디의 리스트를 넣는다.
	 */
	protected List<String> dependsList = new ArrayList<String>();
		
	protected Properties TASK_PROPERTY = null;
	
	private short TASK_TYPE = TYPE_TRANSACTION;
	
	private long TASK_EXEC_INTERVAL = DEFAULT_EXECUTE_INTERVAL;
	
	private String TASK_ID = "1";
	
	/**내부 로그를 작성하는 Writer
	 */
	// XXX 'private'으로 했을 경우 상속받는 클래스에서는 인식을 하지 못해서 'protected'로 바꿈.
	protected eMsFileWriter LOG_CHANNEL = null;
	
	/** Creates new Task */
	protected Task() {
		this( TYPE_TRANSACTION , DEFAULT_EXECUTE_INTERVAL );
	}
	
	protected Task(short type) {
		this( type , DEFAULT_EXECUTE_INTERVAL );
	}
	
	/** Creates new CommonTask */
	protected Task(short type,long interval) {
		this.TASK_TYPE = type;
		this.TASK_EXEC_INTERVAL = interval;
	}
	
	public final void setExecutionType(short type,String interval) {
		this.TASK_TYPE = type;
		try{
			this.TASK_EXEC_INTERVAL = Long.parseLong( interval );
		}
		catch( Exception ignore ){
			this.TASK_EXEC_INTERVAL = DEFAULT_EXECUTE_INTERVAL;
		}
	}
	
	public final void setExecutionType(short type,long interval) {
		this.TASK_TYPE = type;
		this.TASK_EXEC_INTERVAL = interval;
	}
	
	/**해당 Task의 로그를 작성할 Writer를 지정한다.
	 *이 Writer는 지정을 하고 나면 Task가 종료될때 flush() / close()가 자동으로 진행된다.
	 *그러므로 release_Resource() 메소드에서 추가로 지정하거나 할 필요가 없다.
	 */
	protected final void setLogWrtier( eMsFileWriter out ){
		this.LOG_CHANNEL = out;
	}
	
	// XXX log함수가 protected로 되어 있어서 다른 패키지 에서 사용못하게 되어 있음. 그래서 public으로 수정함. kckim 2005Sep01
	public void log( String logStr ){
		if( this.LOG_CHANNEL == null ){
			log.debug( logStr );
			return;
		}
		try{
			this.LOG_CHANNEL.write( Cal.getDate() );
			this.LOG_CHANNEL.write( "\t" );
			this.LOG_CHANNEL.println( logStr );
		}
		catch( Throwable e ){
			log.error("error", e );
		}
	}
	// XXX 인자값이 Throwable인 함수가 없어서 추가함.
	public void log(Throwable thw) {
		if( this.LOG_CHANNEL == null ) {
			log.error("default logging:", thw);
			return;
		}
		try {
			this.LOG_CHANNEL.print(Cal.getDate());
			this.LOG_CHANNEL.print(" ---- STACK TRACE ----");
			this.LOG_CHANNEL.println();
		}
		catch(Throwable e) {
			log.error("log write error", e);
		}
	}
	
	/**Task의 ID를 세팅한다.
	 *Task ID는 unique하며 만일 중복된 ID를생성하여 TaskManager에 실행을 하면 실행되지 않는다.
	 */
	public void setTaskID( String ID ) {
		if (ID != null && !"".equals(ID)) {
			this.TASK_ID = ID;
		}
	}
	
	/**TaskID를 반환한다.
	 */
	public String getTaskID() {
		return this.TASK_ID;
	}
	
	/**Task실행에 필요한 Property를 세팅한다.
	 */
	public void setTaskProperty( Properties prop ){
		this.TASK_PROPERTY = prop;
		this.setTaskID( this.TASK_PROPERTY.getProperty( "TASK_ID") );
		this.setName( this.TASK_PROPERTY.getProperty( "TASK_NAME") );
	}
	
	/**
	 * 선행 작업 아이디 리스트를 넣는다.
	 * @param list
	 */
	public void setDependsList(List<String> list ) {
		
		if( list == null) {
			dependsList = new ArrayList<String>();
		} else {
			dependsList = list;
		}
	}
	
	/**
	 * 입력된 task_id가 이 Task의 선행 작업인지를 판단한다.
	 * @param taskId
	 * @return
	 */
	public boolean depends( String taskId ) {
		
		return (dependsList == null )? false:dependsList.contains(taskId);
	}
	
	public void internal_execute() {
		long startTime = System.currentTimeMillis();
		this.start_flag = true;
		this.work_flag = true;
		//초기화 를 실행하고 에러가나면 에러처리 메소드를 호출하고 그만 둔다.
		try{
			if (log.isDebugEnabled())
				log.debug("[{}] execute_initiate start", getName());
			this.execute_initiate();
			if (log.isDebugEnabled())
				log.debug("[{}] execute_initiate end", getName());
		}
		catch( Throwable thw ){
			log.error("[{}], execute_initiate error", getName(), thw);
			execute_initiateError( thw );
			release_Resource();
			setEnd();
			return;
		}
		
		// 본 실행 메소드를 실행하고 결과에 따라서 수행 메소드를 실행한다.
		try {
			this.execute();
			this.doSuccessProcess();
		} catch( Throwable thw ){
			log.error("[{}], execute error", getName(), thw);
			this.doErrorProcess( thw );
		}
		finally {
			long elapseTime = System.currentTimeMillis() - startTime;
			if (elapseTime > 5000) {
				log.info("{} execute  [elapseTime:{} ms]", getTaskID(), elapseTime);
			}
			release_Resource();
			setEnd();
		}
	}
	
	/**
	 * execute() 가 false를 반환하였을때(실패) 실행하는 로직이 구현된다.
	 * 여기서는 아무일도 하지 않으며 성공시 후속 조치가 필요한 Task는 이 method를 오버라이드 하면 된다.
	 * @param thw
	 */
	public void doErrorProcess( Throwable thw ){
		log.error( getName() , thw );
	}
	
	/** 
	 * execute() 가 true를 반환하였을때(성공) 실행하는 로직이 구현된다.
	 * 여기서는 아무일도 하지 않으며 성공시 후속 조치가 필요한 Task는 이 method를 오버라이드 하면 된다.
	 */
	public void doSuccessProcess(){
	}
	
	/** 
	 * Task의 완료 유무를 반환
	 * @return true : 완료된 Task 이며 제거대상이 되는 Task임<br>
	 * false : 종료되지 않았으며 재실행을 기다리는 Task
	 */
	public final boolean isEnd() {
		if( this.TASK_TYPE == TYPE_INTERVAL ) {
			return false;
		}
		
		return !this.work_flag;
	}
	
	/**
	 * Task의 작업이 시작되었는지를 반환 
	 * 선행 작업이 있는 경우 바로 시작되지 않고 대기 상태로 TaskManager에 등록되어 있다.
	 * @return
	 */
	public final boolean isStarted() {
		
		return this.start_flag;
	}
	
	/**main 실행 로직이 종료될때 호출되어 실행에 필요했던 값을 정리한다.
	 */
	private void setEnd() {
		if( this.LOG_CHANNEL != null ) {
			this.LOG_CHANNEL.close();
		}
		
		this.work_flag = false;
		
		if( this.TASK_TYPE == TYPE_INTERVAL ) {
			this.next_execute_time = System.currentTimeMillis() + this.TASK_EXEC_INTERVAL;
		}
	}
	
	/** 에러가 발생하여 일정 시간을 두고 기다려야 하는 경우에<br>
	 * 매니져가 이 메소드를 호출하여 true를 반환할 경우에<br>
	 * <B>execute()</B> 메소드를 호출한다.
	 * @return true : 실행할 시간이 됨<br>
	 * false : 아직 더 있어야 함
	 */
	public boolean isValidTime() {
		/** for debug
		 * log.debug( "WORK FLAG:" + this.work_flag );
		 * log.debug( "NEXT:" + Cal.getDate( this.next_execute_time ) );
		 */
		// debug end
		if( this.TASK_TYPE == TYPE_TRANSACTION ) {
			return false;
		}
		if( this.next_execute_time < 0 ) {
			return false;
		}
		return !this.work_flag &&System.currentTimeMillis() > this.next_execute_time ;
	}
	
	/** 스케쥴 패턴을 반환한다.<br>
	 * [TYPE_TRANSACTION / TYPE_INTERVAL ]
	 * @return [TYPE_TRANSACTION / TYPE_INTERVAL ]
	 */
	public short getType() {
		return TYPE_INTERVAL;
	}
	
	/** 작업중 생성된 Object와 자원을 반환한다.
	 * 매니져에서 unregist 될때 호출된다.
	 */
	public void destroy() {
	}
	
	@Override
	public String toString() {
		return getName();
	}
	
	/**
	 * 실제적인 일을 처리하는 비지니스로직
	 * @throws Exception
	 */
	public abstract void execute() throws Exception;
	
	/**Task를 초기화하는 로직을 구현한다.
	 *Throwable이 발생하게 되면 execute_initiateError() 를 호출하도록 되어있다.
	 * 초기에는 abstract로 해 놓았었는데.. getInstance() & recycleInstance() 의 미묘한 녀석들을 처리하기 위해서
	 * 공란으로 만들어놓고 하부에서 Task를초기화 할때 무조건 super.execute_initiate() 를 호출하는 방식으로 처리하도록 함.
	 * @throws Exception
	 */
	public void execute_initiate() throws Exception{
		// do nothing...
	}
	
	/**초기화할때 Throwable이 뛰쳐나왔을때 처리하는 로직을 구현한다.
	 * Exception을 절대로 반환하면 안된다. 그러면 ㅠㅠ
	 * 에러처리는 구현안에 모두 포함하여 하는 것을 권장한다.
	 * @param thw
	 */
	public abstract void execute_initiateError( Throwable thw );
	
	/**에러가 발생하거나 정상적으로 끝나거나 언제나 실행이된다.
	 * 할당받은 자원을 free 시키는 로직을 구현한다.
	 * execute_initiateError()와 마찬가지로 Exception을 반환하면 안되고 처리로직을 전체 포함하여야 한다.
	 */
	public abstract void release_Resource();
}
