/*
 * Created on 2005. 6. 22
 */
package pluto.nio;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;

import pluto.io.eMsByteArray;
import pluto.util.FIFOBuffer;

/**
 * @author 이상근
 */
public class ReusableByteArrayOutputStream extends java.io.OutputStream {
	//자주 사용되기 때문에 재사용을 위한 버퍼를 제공한다.
	private static final int MAX_CONTAIN_SIZE = 10;

	private static FIFOBuffer INNER_CONTAINER = null;

	static {
		INNER_CONTAINER = new FIFOBuffer(MAX_CONTAIN_SIZE);
	}

	public static ReusableByteArrayOutputStream getInstance() {
		Object returnValue = INNER_CONTAINER.pop();

		if (returnValue == null) {
			return new ReusableByteArrayOutputStream(1024);
		}

		return (ReusableByteArrayOutputStream) returnValue;
	}

	public static void recycleInstance(ReusableByteArrayOutputStream tmp) {
		if (tmp == null)
			return;
		try {
			tmp.reset();
		}
		catch (Exception e) {
		}
		INNER_CONTAINER.push(tmp);
	}

	/**
	 * The buffer where data is stored.
	 */
	protected byte buf[];

	/**
	 * The number of valid bytes in the buffer.
	 */
	protected int count;

	/** Creates a new instance of eMsByteArrayOutputStream */
	public ReusableByteArrayOutputStream() {
		this(32);
	}

	/**
	 * Creates a new byte array output stream, with a buffer capacity of the
	 * specified size, in bytes.
	 * 
	 * @param size
	 *            the initial size.
	 * @exception IllegalArgumentException
	 *                if size is negative.
	 */
	public ReusableByteArrayOutputStream(int size) {
		if (size < 0) {
			throw new IllegalArgumentException("Negative initial size: " + size);
		}
		buf = new byte[size];
	}

	public synchronized void write(int b) throws java.io.IOException {
		int newcount = count + 1;
		if (newcount > buf.length) {
			byte newbuf[] = new byte[Math.max(buf.length << 1, newcount)];
			System.arraycopy(buf, 0, newbuf, 0, count);
			buf = newbuf;
		}
		buf[count] = (byte) b;
		count = newcount;
	}

	/**
	 * Writes <code>len</code> bytes from the specified byte array starting at
	 * offset <code>off</code> to this byte array output stream.
	 * 
	 * @param b
	 *            the data.
	 * @param off
	 *            the start offset in the data.
	 * @param len
	 *            the number of bytes to write.
	 */
	public synchronized void write(byte b[], int off, int len) {
		if ((off < 0) || (off > b.length) || (len < 0) || ((off + len) > b.length) || ((off + len) < 0)) {
			throw new IndexOutOfBoundsException("TARGET : " + b.length + " OFF: " + off + " LEN: " + len);
		}
		else if (len == 0) {
			return;
		}
		int newcount = count + len;
		if (newcount > buf.length) {
			byte newbuf[] = new byte[Math.max(buf.length << 1, newcount)];
			System.arraycopy(buf, 0, newbuf, 0, count);
			buf = newbuf;
		}
		System.arraycopy(b, off, buf, count, len);
		count = newcount;
	}

	/**
	 * Creates a newly allocated byte array. Its size is the current size of
	 * this output stream and the valid contents of the buffer have been copied
	 * into it.
	 * 
	 * @return the current contents of this output stream, as a byte array.
	 * @see java.io.ByteArrayOutputStream#size()
	 */
	public synchronized int toByteArray(eMsByteArray toBuffer) {
		if (toBuffer.length() < count) {
			toBuffer.setArray(new byte[count]);
		}
		System.arraycopy(buf, 0, toBuffer.getArray(), 0, count);
		return count;
	}

	/**
	 * Resets the <code>count</code> field of this byte array output stream to
	 * zero, so that all currently accumulated output in the ouput stream is
	 * discarded. The output stream can be used again, reusing the already
	 * allocated buffer space.
	 * 
	 * @see java.io.ByteArrayInputStream#count
	 */
	public synchronized void reset() {
		count = 0;
	}

	/**
	 * Returns the current size of the buffer.
	 * 
	 * @return the value of the <code>count</code> field, which is the number
	 *         of valid bytes in this output stream.
	 * @see java.io.ByteArrayOutputStream#count
	 */
	public int size() {
		return count;
	}

	/**
	 * Converts the buffer's contents into a string, translating bytes into
	 * characters according to the platform's default character encoding.
	 * 
	 * @return String translated from the buffer's contents.
	 * @since JDK1.1
	 */
	public String toString() {
		return new String(buf, 0, count);
	}

	/**
	 * Converts the buffer's contents into a string, translating bytes into
	 * characters according to the specified character encoding.
	 * 
	 * @param enc
	 *            a character-encoding name.
	 * @return String translated from the buffer's contents.
	 * @throws UnsupportedEncodingException
	 *             If the named encoding is not supported.
	 * @since JDK1.1
	 */
	public String toString(String enc) throws UnsupportedEncodingException {
		return new String(buf, 0, count, enc);
	}

	public void writeTo(OutputStream out) throws IOException {
		out.write(buf, 0, count);
	}

	/**
	 * Closing a <tt>ByteArrayOutputStream</tt> has no effect. The methods in
	 * this class can be called after the stream has been closed without
	 * generating an <tt>IOException</tt>.
	 * <p>
	 *  
	 */
	public void close() throws IOException {
	}

}
