/*
 * Created on 2005. 6. 22
 */
package pluto.nio;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * @author 이상근
 */
public class ByteLineReader {
	private static final int	INVALIDATED				= -2;

	private static final int	UNMARKED				= -1;

	private static final byte	CR						= (byte) '\r';

	private static final byte	LF						= (byte) '\n';

	private InputStream			in;

	private byte				cb[];

	private int					nChars;

	private int					nextChar;

	private int					markedChar				= UNMARKED;

	/**
	 * Valid only when markedChar > 0
	 */
	private int					readAheadLimit			= 0;

	/** If the next character is a line feed, skip it */
	private boolean				skipLF					= false;

	private static int			defaultByteBufferSize	= 8192;

	/**
	 * Create a buffering character-input stream that uses an input buffer of
	 * the specified size.
	 * 
	 * @param in
	 *            A Reader
	 * @param sz
	 *            Input-buffer size
	 * 
	 * @exception IllegalArgumentException
	 *                If sz is <= 0
	 */
	public ByteLineReader(InputStream in, int sz) {
		this.in = in;
		cb = new byte[sz];
		nextChar = nChars = 0;
	}

	/**
	 * Create a buffering character-input stream that uses a default-sized input
	 * buffer.
	 * 
	 * @param in
	 *            A Reader
	 */
	public ByteLineReader(InputStream in) {
		this(in, defaultByteBufferSize);
	}

	/** Check to make sure that the stream has not been closed */
	private void ensureOpen() throws IOException {
		if( in == null ) {
			throw new IOException("Stream closed");
		}
	}

	/**
	 * Fill the input buffer, taking the mark into account if it is valid.
	 */
	private void fill() throws IOException {
		int dst;
		if( markedChar <= UNMARKED ) {
			/* No mark */
			dst = 0;
		}
		else {
			/* Marked */
			int delta = nextChar - markedChar;
			if( delta >= readAheadLimit ) {
				/* Gone past read-ahead limit: Invalidate mark */
				markedChar = INVALIDATED;
				readAheadLimit = 0;
				dst = 0;
			}
			else {
				if( readAheadLimit <= cb.length ) {
					/* Shuffle in the current buffer */
					System.arraycopy(cb, markedChar, cb, 0, delta);
					markedChar = 0;
					dst = delta;
				}
				else {
					/* Reallocate buffer to accommodate read-ahead limit */
					byte ncb[] = new byte[readAheadLimit];
					System.arraycopy(cb, markedChar, ncb, 0, delta);
					cb = ncb;
					markedChar = 0;
					dst = delta;
				}
				nextChar = nChars = delta;
			}
		}

		int n;
		do {
			n = in.read(cb, dst, cb.length - dst);
		} while (n == 0);
		if( n > 0 ) {
			nChars = dst + n;
			nextChar = dst;
		}
	}

	public boolean ready() throws IOException {
		synchronized (this.in) {
			ensureOpen();

			/*
			 * If newline needs to be skipped and the next char to be read is a
			 * newline character, then just skip it right away.
			 */
			if( skipLF ) {
				/*
				 * Note that in.ready() will return true if and only if the next
				 * read on the stream will not block.
				 */
				if( nextChar >= nChars && in.available() > 0 ) {
					fill();
				}
				if( nextChar < nChars ) {
					if( cb[nextChar] == '\n' )
						nextChar++;
					skipLF = false;
				}
			}
			return (nextChar < nChars) || in.available() > 0;
		}
	}

	/**
	 * Read a line of text. A line is considered to be terminated by any one of
	 * a line feed ('\n'), a carriage return ('\r'), or a carriage return
	 * followed immediately by a linefeed.
	 * 
	 * @param ignoreLF
	 *            If true, the next '\n' will be skipped
	 * 
	 * @return A String containing the contents of the line, not including any
	 *         line-termination characters, or null if the end of the stream has
	 *         been reached
	 * 
	 * @see java.io.LineNumberReader#readLine()
	 * 
	 * @exception IOException
	 *                If an I/O error occurs
	 */
	public void writeLineToStream(OutputStream buffer, int limit) throws IOException {
		int startChar;

		boolean omitLF = skipLF;

		int iAppendSize = 0;

		synchronized (this.in) {
			ensureOpen();

			bufferLoop: for (;;) {

				if( nextChar >= nChars ) {
					fill();
				}
				if( nextChar >= nChars ) { /* EOF */
					return;
				}

				boolean eol = false;

				byte c = 0;

				int i;

				/* Skip a leftover '\n', if necessary */
				if( omitLF && (cb[nextChar] == LF) ) {
					nextChar++;
				}

				skipLF = false;
				omitLF = false;

				charLoop: for (i = nextChar; i < nChars; i++) {
					c = cb[i];
					if( (c == LF) || (c == CR) ) {
						eol = true;
						break charLoop;
					}
				}

				startChar = nextChar;
				nextChar = i;

				if( eol ) {
					iAppendSize += (i - startChar);
					if( limit > 0 && iAppendSize > limit ) {
						throw new LimitSizeExceedException("LINE LIMIT OVERFLOW:" + String.valueOf(iAppendSize) + " LIMIT:" + limit);
					}
					buffer.write(cb, startChar, i - startChar);
					nextChar++;
					if( c == '\r' ) {
						skipLF = true;
					}
					return;
				}

				iAppendSize += (i - startChar);
				if( limit > 0 && iAppendSize > limit ) {
					throw new LimitSizeExceedException("LINE LIMIT OVERFLOW:" + String.valueOf(iAppendSize) + " LIMIT:" + limit);
				}
				buffer.write(cb, startChar, i - startChar);
			}
		}
	}

	/**
	 * 다 쓰고 닫기.
	 *  
	 */
	public void close() {
		try {
			this.in.close();
		}
		catch(Throwable thw) {

		}
		this.in = null;
	}
}
