/*
 * SocketAgent.java
 *
 * Created on 2003년 3월 5일 수, 오후 10:27
 */

package pluto.net;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;

import pluto.lang.Name;
import lombok.extern.slf4j.Slf4j;
import pluto.lang.eMsLocale;
import pluto.log.ErrorSpoolLogger;
import pluto.log.Log;
import pluto.mail.mx.LookupUtil;
import pluto.util.Cal;

/**
 * 소켓으로 통신하는 기본 룰을 정의한다.
 * 
 * @author 이상근
 * @version 3.0
 * @modified by keidao (eMs 6.0)
 */
@Slf4j
public abstract class SocketAgent extends Name implements Log, NetworkMonitorable {

	private static final boolean	HOST_DEBUG				= false;

	private static final boolean	JDK_VERSION_12_ENABLE	= false;

	private static final boolean	SESSION_DEBUG			= false;

	protected static final int		STRING_TYPE				= 1;

	protected static final int		BINARY_TYPE				= 2;

	// 로컬 호스트의 InetAddress 를 미리 하나 받아놓고 사용한다.
	// private InetAddress LOCAL_INET_ADDRESS = null;

	protected Socket				CONNECTED_SOCKET		= null;

	protected int					TIME_OUT				= 0;

	protected long					COMM_START_TIME			= 0;

	/** Holds value of property connectHost. */
	protected String				CONNECT_HOST;

	/** Holds value of property targetPort. */
	protected int					targetPort;

	private boolean					__IS_MONITORED__		= false;

	private InnerTmpMonitor			INNER_TEMP_MONITOR		= null;

	/**
	 * 기본적으로 모니터링의 대상이 되지 않는 SocketAgent를 생성한다.
	 */
	protected SocketAgent() {
		this(false);
	}

	protected SocketAgent(boolean regist) {
		this.__IS_MONITORED__ = regist;
		if( this.__IS_MONITORED__ ) {
			SocketAgentMonitor.registSocketAgent(this);
		}
	}

	/**
	 * 모니터링을 중지한다.
	 */
	protected void unregist() {
		if( this.__IS_MONITORED__ ) {
			SocketAgentMonitor.removeSocketAgent(this);
		}
	}

	/**
	 * 기준 시간 이상 통신이 이루어졌는 지를 반환한다.
	 * 
	 * @return true : 기준 시간 이상 idle 상태 <br>
	 *         false : 통신중이거나 유휴상태
	 */
	public boolean isIdle() {
		if( isConnect() && (System.currentTimeMillis() - this.COMM_START_TIME) > this.TIME_OUT ) {
			if( SESSION_DEBUG )
				log("S:" + Cal.getDate(this.COMM_START_TIME) + " L:" + String.valueOf(this.TIME_OUT) + " N:" + String.valueOf(System.currentTimeMillis()) + "/" + String.valueOf(this.COMM_START_TIME));
			return true;
		}

		return false;
	}

	private void foo() {
	}

	public void killSession() {
		if( this.CONNECTED_SOCKET != null ) {
			if( SESSION_DEBUG )
				log("KILL SESSION CALL smtp.shutdownInput() ");
			try {
				if( JDK_VERSION_12_ENABLE )
					foo();
				else
					this.CONNECTED_SOCKET.shutdownInput();
			}
			catch(Throwable e) {
				if( SESSION_DEBUG )
					log("KILL SESSION ERROR", e);
			}

			if( SESSION_DEBUG )
				log("KILL SESSION CALL smtp.shutdownOutput() ");
			try {
				if( JDK_VERSION_12_ENABLE )
					foo();
				else
					this.CONNECTED_SOCKET.shutdownOutput();
			}
			catch(Throwable e) {
				if( SESSION_DEBUG )
					log("KILL SESSION ERROR", e);
			}
			if( SESSION_DEBUG )
				log("KILL SESSION CALL smtp.close() ");
			try {
				this.CONNECTED_SOCKET.close();
			}
			catch(Exception e) {
				if( SESSION_DEBUG )
					log("KILL SESSION ERROR", e);
			}

			// 연결이 안된걸루 판단한다.
			// this.IN_COMM = false;
		}
	}

	/**
	 * 모든 스트림을 종료하고 Socket 연결을 종료한다.
	 */
	public void close() {
		this.CONNECT_HOST = null;

		// this.IN_COMM = false;
		this.COMM_START_TIME = -1;

		if( INNER_TEMP_MONITOR != null ) {
			INNER_TEMP_MONITOR.interrupt();
			INNER_TEMP_MONITOR = null;
		}

		killSession();

		this.CONNECTED_SOCKET = null;
	}

	public boolean isConnect() {
		//return this.IN_COMM;
		// 현재 소켓의 상태를 점검하는 방식으로 수정
		// isConnected() : 소켓에 성공적으로 연결이 된적이 있으면 true
		// isClose() : 소켓이 닫혀 있는지 판단
		// isClosed()의 초기 값은 false이다.
		if(this.CONNECTED_SOCKET == null){
			return false;
		} else {
			return (this.CONNECTED_SOCKET.isConnected() && !this.CONNECTED_SOCKET.isClosed());
		}		
	}

	/**
	 * 지정 서버의 특정 포트로 Socket을 연결한다.
	 * 
	 * @param host
	 *        연결 호스트
	 * @param port
	 *        연결 포트
	 * @throws IOException
	 *         연결중에러
	 */
	protected final void connect(String host, int port) throws IOException {
		this.COMM_START_TIME = System.currentTimeMillis();

		if (log.isDebugEnabled()) {
			log("Connect Host: " + host);
			log("Connect Port: " + String.valueOf(port));
		}

		this.CONNECT_HOST = null;
		// this.IN_COMM = false;

		String sTargetAddr = null;
		String sTargetIP = null;

		if( host.indexOf("=") > 0 ) {
			sTargetIP = host.substring(0, host.indexOf("="));
			sTargetAddr = host.substring(host.indexOf("=") + 1);
			if( HOST_DEBUG ) {
				log("IP:" + sTargetIP);
				log("HOST:" + sTargetAddr);
			}
		}
		else {
			if( HOST_DEBUG ) {
				log("HOST ONLY:" + host);
			}
			sTargetAddr = host;
		}

		//IP가 null이면 호스트 이름으로 접속을 한다.
		this.CONNECTED_SOCKET = new Socket(sTargetIP == null ? sTargetAddr : sTargetIP, port);

		this.CONNECT_HOST = sTargetAddr.concat("[").concat(this.CONNECTED_SOCKET.getInetAddress().getHostAddress()).concat("]");
		targetPort = this.getLocalPort();

		// this.IN_COMM = true;

		openConnection();

	}

	/**
	 * [modify JOO]
	 * TIMEOUT 속성을 주고 연결을 조정하기 위해서 추가되는 메소드
	 * 
	 * @param host
	 * @param iIP
	 * @param port
	 * @param timeoutConnect
	 * @throws IOException
	 */
	protected final void connect(String host, int iIP, int port, int timeoutConnect) throws IOException {
		this.COMM_START_TIME = System.currentTimeMillis();

		this.CONNECT_HOST = null;

		//-------------------------------
//		InetAddress address = LookupUtil.getInetAddress(iIP);
//		this.CONNECT_HOST = host.concat("[").concat(address.getHostAddress()).concat("]");
//
////		새롭게 수정한 커넥트 방법
//		this.CONNECTED_SOCKET = new Socket();
//		this.CONNECTED_SOCKET.connect(new InetSocketAddress(address, port), timeoutConnect);
//
//		targetPort = this.getLocalPort();
//		openConnection();
		
		//-----------------------------
		
		InetAddress address = LookupUtil.getInetAddress(iIP);
		try{
			/****************** 신규 소켓생성 방식(Muti LanCard) **********************/
			InetAddress localAddress = null;		
			
			localAddress = InetAddress.getByName(eMsLocale.LOCAL_ADDR_IP[0]);
		
			// [kckim] 멀티 네트워크 카드를 이용한 발송을 위한것 검증안됨 (테스트 요망..)  ... [JOO] 분석			
			InetSocketAddress localSa = new InetSocketAddress(localAddress,0); //발송 서버 
			InetSocketAddress remoteSa = new InetSocketAddress(address,port); //받는 서버 
			 
			if(log.isDebugEnabled()){
				log.debug("[LOCAL SET ADDRESS]" + eMsLocale.LOCAL_ADDR_IP);
				log.debug("[RECEIVE ADDRESS][port]"+port+"[iIP]"+address.toString());
				log.debug("[SENDER_InetSocketAddress]"  + localSa.getAddress().getHostAddress() );
			}
			
			this.CONNECTED_SOCKET = new Socket();
			this.CONNECTED_SOCKET.bind(localSa);    //발송 서버 세팅
			this.CONNECTED_SOCKET.connect(remoteSa,timeoutConnect);//받는 서버 연결
			
			if(log.isDebugEnabled()){
				log.debug(" [SENDER_Socket_Address]" + this.CONNECTED_SOCKET.getLocalAddress().getHostAddress() );
			}
			/****************** end (Muti LanCard) **********************/		
			
			this.CONNECT_HOST = host.concat("[").concat(address.getHostAddress()).concat("]");
			
			targetPort = this.getLocalPort();
		
		}catch(IOException ie ){
			//log.debug("[Conn Fail]=>"+address.toString());
			log.error("[Conn Fail]=>"+address.toString());
			throw ie;
		}
		openConnection();
	}

	/**
	 * 지정된 Socket으로 연결을 생성한다.
	 * 
	 * @param host
	 *        연결 호스트
	 * @param port
	 *        연결 포트
	 * @throws IOException
	 *         연결중에러
	 */
	protected final void connect(Socket host) throws IOException {
		this.COMM_START_TIME = System.currentTimeMillis();

		CONNECT_HOST = host.getInetAddress().getHostAddress();

		if (log.isDebugEnabled()) {
			log("Connect Host: " + CONNECT_HOST);
		}

		// this.IN_COMM = true;
		this.CONNECTED_SOCKET = host;

		openConnection();
	}

	protected final void resetConnectionTime() {
		this.COMM_START_TIME = System.currentTimeMillis();
	}

	protected final long getElapsedTime() {
		return this.COMM_START_TIME > 0 ? System.currentTimeMillis() - this.COMM_START_TIME : 0;
	}

	/**
	 * 메일서버와의 Connection으로 메세지를 전송한다.
	 * 
	 * @param source
	 *        전송할 메세지
	 * @throws IOException
	 *         전송중 에러발생
	 */
	protected void send(Object source) throws IOException {
		if( source == null )
			return;

		if( source instanceof String ) {
			send((String) source);
		}
		else if( source instanceof byte[] ) {
			send((byte[]) source);
		}
		else {
			send(source.toString());
		}
	}

	/**
	 * 통신 Socket의 TimeOut을 설정한다. <br>
	 * Monitor는 설정된 시간의 2배가 지나면 idle로 간주한다.
	 * 
	 * @return true : 설정 성공 <br>
	 *         false : 설정 실패
	 * @param timeOfTimeout
	 *        amount of TimeOut
	 */
	protected boolean setConnectionTimeout(int timeOfTimeout) {
		this.TIME_OUT = timeOfTimeout;
		try {
			this.CONNECTED_SOCKET.setSoTimeout(timeOfTimeout);
			if( this.__IS_MONITORED__ ) {
				// 모니터링 되고 있으면 아무것도 할일이 없고
			}
			else {
				// 모니터링 되는 녀석이 아니면 이정도 시간이 흐른후에 인터럽트 하는 녀석을 띄운다.
				INNER_TEMP_MONITOR = new SocketAgent.InnerTmpMonitor(this, timeOfTimeout);
				INNER_TEMP_MONITOR.start();
			}
		}
		catch(Exception e) {
			return false;
		}

		return true;
	}

	/**
	 * 연결된 호스트 정보를 반환한다.
	 * 
	 * @return 연결된 호스트 정보
	 */
	public String getConnectHost() {
		// return this.isConnect() ? this.CONNECT_HOST : "NC";

		if(this.CONNECTED_SOCKET == null){
			return "NC";
		} else if(!isConnect()){
			return "NC-"+this.CONNECT_HOST;
		} else {
			return this.CONNECT_HOST;
		}
	}

	/**
	 * 연결된 포트번호를 반환한다.
	 * 
	 * @return 연결된 포트번호
	 */
	public int getTargetPort() {
		if( this.CONNECTED_SOCKET == null )
			return -1;
		return this.CONNECTED_SOCKET.getPort();
	}

	/**
	 * 연결된 Local 포트번호를 반환한다.
	 * 
	 * @return 연결된 포트번호
	 */
	public int getLocalPort() {
		if( this.CONNECTED_SOCKET == null )
			return -1;
		return this.CONNECTED_SOCKET.getLocalPort();
	}

	/**
	 * 로깅을 해야쥐..
	 * 
	 * @param source
	 *        로깅할 String
	 */
	public void log(String logStr) {
		log.debug(getName() + "=>" + logStr);
	}

	/**
	 * 로깅을 해야쥐..
	 * 
	 * @param source
	 *        로깅할 String
	 */
	public void log(String logStr, Throwable e) {
		log.debug(getName() + "=>" + logStr);
		log.error(e.getMessage(), e);
	}

	protected abstract void openConnection() throws IOException;

	/**
	 * 메세지를 전송한다.
	 * 
	 * @param source
	 *        전송할 메세지
	 * @throws IOException
	 *         전송중 에러
	 */
	protected abstract void send(String source) throws IOException;

	/**
	 * byte 배열의 메세지를 전송한다.
	 * 
	 * @param source
	 *        전송할 byte 배열
	 * @throws IOException
	 *         전송중 에러
	 */
	protected abstract void send(byte[] source) throws IOException;

	static class InnerTmpMonitor extends Thread {
		SocketAgent	agent		= null;

		int			TIME_OUT	= 0;

		InnerTmpMonitor(SocketAgent a, int timeout) {
			this.agent = a;
			this.TIME_OUT = timeout;
		}

		public void run() {
			if (log.isDebugEnabled()) 
				log.debug("MONITOOR START");
			try {
				Thread.currentThread().sleep(this.TIME_OUT);
				if( this.agent.isIdle() ) {
					this.agent.killSession();
				}
			}
			catch(Exception e) {
				if (log.isDebugEnabled()) 
					log.error(e.getMessage());
			}

			if (log.isDebugEnabled()) 
				log.debug("MONITOOR END");
		}
	}
}
