/*
 * Created on 2005-12-14
 */
package pluto.net;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.Socket;

import lombok.extern.slf4j.Slf4j;
import pluto.io.eMsByteArrayOutputStream;
import pluto.lang.ByteArrayContainer;
import pluto.lang.Name;
import pluto.mail.mx.LookupUtil;

/**
 * @author 이상근
 */
@Slf4j
public class MonitorableSocket extends Name implements NetworkMonitorable {
	private static final int	BYTE_CR					= (int) '\r';

	private static final int	BYTE_LF					= (int) '\n';

	private int					TIME_OUT				= 0;

	private boolean				IN_COMM					= false;

	private long				COMM_START_TIME			= 0;

	protected Socket			CONNECTED_SOCKET		= null;

	private InputStream			SOCKET_INPUT_STREAM		= null;

	private OutputStream		SOCKET_OUTPUT_STREAM	= null;

	private String				CHAR_SET				= null;

	/** Holds value of property connectHost. */
	protected String			CONNECT_HOST;

	public MonitorableSocket(String charset) {
		try {
			"abc".getBytes(charset);
			this.CHAR_SET = charset;
		}
		catch(UnsupportedEncodingException e) {
			log.error(e.getMessage());
			this.CHAR_SET = null;
		}
	}

	public String getConnectHost() {
		return this.CONNECT_HOST;
	}

	public boolean isIdle() {
		if( this.IN_COMM && (System.currentTimeMillis() - this.COMM_START_TIME) > this.TIME_OUT ) {
			return true;
		}

		return false;
	}

	public void killSession() {
		if( this.CONNECTED_SOCKET != null ) {
			// shutdown Input
			try {
				this.CONNECTED_SOCKET.shutdownInput();
			}
			catch(Throwable e) {
			}

			// shutdown output
			try {
				this.CONNECTED_SOCKET.shutdownOutput();
			}
			catch(Throwable e) {
			}

			// close socket
			try {
				this.CONNECTED_SOCKET.close();
			}
			catch(Exception e) {
			}

			// 연결이 안된걸루 판단한다.
			this.IN_COMM = false;

			// 소켓을 null 처리
			this.CONNECTED_SOCKET = null;
		}
	}

	/**
	 * 모든 스트림을 종료하고 Socket 연결을 종료한다.
	 */
	public void close() {
		killSession();
		this.CONNECT_HOST = null;
		this.COMM_START_TIME = -1;
	}

	/**
	 * Lookup 바뀌면서 추가되는 메소드
	 * 
	 * @param host
	 * @param iIP
	 * @param port
	 * @throws IOException
	 */
	protected final void connect(String host, int iIP, int port) throws IOException {
		this.COMM_START_TIME = System.currentTimeMillis();

		this.IN_COMM = false;

		this.CONNECT_HOST = null;

		InetAddress address = LookupUtil.getInetAddress(iIP);

		//IP가 null이면 호스트 이름으로 접속을 한다.
		this.CONNECTED_SOCKET = new Socket(address, port);

		this.CONNECT_HOST = host.concat("[").concat(address.getHostAddress()).concat("]");

		this.IN_COMM = true;

		// 스트림 생성
		this.SOCKET_INPUT_STREAM = this.CONNECTED_SOCKET.getInputStream();

		this.SOCKET_OUTPUT_STREAM = this.CONNECTED_SOCKET.getOutputStream();
	}

	public String receiveLine() throws IOException {
		eMsByteArrayOutputStream out = null;
		byte[] buffer = null;
		try {
			out = eMsByteArrayOutputStream.getInstance();
			buffer = ByteArrayContainer.getInstance();


			while (true) {
				this.SOCKET_INPUT_STREAM.read(buffer);
				break;
			}

			return this.CHAR_SET == null ? out.toString() : out.toString(this.CHAR_SET);
		}
		finally {
			eMsByteArrayOutputStream.recycleInstance(out);
			ByteArrayContainer.recycleInstance(buffer);
		}
	}

	/**
	 * flagTelnet 을 true로 설정하면 str을 전송하고 자동으로 CR LF 를 추가로 전송한다. <br>
	 * 그러니까 뒤에 CR/LF를 붙여서 보내지 않아야 한다.
	 * 
	 * @param sendBytes
	 * @param flagTelnet
	 * @throws IOException
	 */
	public void send(byte[] sendBytes, boolean flagTelnet) throws IOException {

		if( flagTelnet ) {
			// Telnet Protocol로 전달해야한다면
			byte[] transfer = null;
			int count = 0;
			int lastb = 0;
			try {
				transfer = ByteArrayContainer.getInstance();

				for (int i = 0; i < sendBytes.length; i++) {
					if( sendBytes[i] == BYTE_CR ) {
						if( count > 0 ) {
							this.SOCKET_OUTPUT_STREAM.write(transfer, 0, count);
							count = 0;
						}
						this.SOCKET_OUTPUT_STREAM.write(BYTE_CR);
						this.SOCKET_OUTPUT_STREAM.write(BYTE_LF);
						this.SOCKET_OUTPUT_STREAM.flush();
					}
					else if( i == BYTE_LF ) {
						if( lastb != BYTE_CR ) {
							if( count > 0 ) {
								this.SOCKET_OUTPUT_STREAM.write(transfer, 0, count);
								count = 0;
							}
							this.SOCKET_OUTPUT_STREAM.write(BYTE_CR);
							this.SOCKET_OUTPUT_STREAM.write(BYTE_LF);
							this.SOCKET_OUTPUT_STREAM.flush();
						}
					}
					else {
						transfer[count++] = sendBytes[i];

						if( count >= transfer.length ) {
							this.SOCKET_OUTPUT_STREAM.write(transfer, 0, count);
							count = 0;
						}
					}

					lastb = sendBytes[i];
				}

				this.SOCKET_OUTPUT_STREAM.write(BYTE_CR);
				this.SOCKET_OUTPUT_STREAM.write(BYTE_LF);
				this.SOCKET_OUTPUT_STREAM.flush();
			}
			finally {
				ByteArrayContainer.recycleInstance(transfer);
			}
		}
		else {
			this.SOCKET_OUTPUT_STREAM.write(sendBytes);
			this.SOCKET_OUTPUT_STREAM.flush();
		}
	}

	/**
	 * flagTelnet 을 true로 설정하면 str을 전송하고 자동으로 CR LF 를 추가로 전송한다. <br>
	 * 그러니까 뒤에 CR/LF를 붙여서 보내지 않아야 한다.
	 * 
	 * @param str
	 * @param flagTelnet
	 * @throws IOException
	 */
	public void send(String str, boolean flagTelnet) throws IOException {
		byte[] sendBytes = this.CHAR_SET == null ? str.getBytes() : str.getBytes(this.CHAR_SET);
		send(sendBytes, flagTelnet);
	}
}
