/*
 * Created on 2005. 12. 2
 */
package pluto.mail.mx;

import java.io.File;
import java.io.IOException;

import lombok.extern.slf4j.Slf4j;
import pluto.lang.eMsLocale;
import pluto.lang.eMsStringBuffer;
import pluto.log.DisabledMXLogger;
import pluto.util.Cal;
import pluto.util.eMsStringTokenizer;

/**
 * @author 이상근
 */
@Slf4j
public class MXSearchResult {
	

	public String					TARGET_DOMAIN						= null;

	public String					TARGET_NS_DOMAIN					= null;

	public long						SEARCH_TIME							= Long.MIN_VALUE;

	public Throwable				SEARCH_ERROR						= null;

	/**
	 * 도메인 리스트를 담는다.
	 */
	//private transient String[]		domainDATA							= null;

	/**
	 * 도메인 아이피 리스트를 담는다.
	 */
	//private transient int[]			domainIP							= null;

	/**
	 * 도메인 객체
	 */
	public transient MXDomain[]	mxDomain								= null;
	
	/**
	 * 가장적은 MX값을 가진 도메인 리스트 개수
	 */
	protected int					INDEX_OF_FIRST_REFERENCE_SEQUENCE	= 0;

	/**
	 * 현재 인덱스
	 */
	protected int					domainIndex	= 0;

	/**
	 * 도메인 리스트 총 개수
	 */
	protected int					size								= 0;

	int								countErrorRaise						= 0;

	/**
	 * 결과에 대한 코드
	 */
	int								ERROR_TYPE							= DNS.ERROR_INITIALIZE;

	/**
	 * 에러시 발생한 Throwable
	 */
	Throwable						ERROR								= null;

	String							fullName							= null;

	/**
	 * 
	 * @param domain
	 * @throws IOException
	 */
	public MXSearchResult(String domain) {
		super();
		this.TARGET_DOMAIN = domain;
	}

	/**
	 * MXSearchResult 가 expire되어 다시 Lookup을 초청하는 Method <br>
	 * 
	 * @param result
	 *        MXSearchResult
	 * @throws Exception
	 *         SearchError / Lookup과정의 에러는 Search Error가 되지만 그외의 에러를 반환한다.
	 */
	public synchronized final void touch() {
		MXLookup lookup = null;
		try {
			lookup = MXLookup.getInstance(); // return UDPMXLookup
			lookup.refesh(this);
			if(log.isDebugEnabled()) log.debug("[touch_in]"+this.toString());
		}
		catch(Throwable thw) {
			setError(DNS.ERROR_DURING_LS_LOOKUP, thw);
		}
		finally {
			MXLookup.recycleInstance(lookup);
		}
	}

	/**
	 * 디스크 케쉬할 파일이름을 반환한다.
	 * 
	 * @return 케쉬 파일이름
	 * @throws IOException
	 *         하부디렉토리를 만드는 과정에서 에러가 발행한다면..
	 */
	public String getDiskCacheFileName() throws IOException {
		// -- cache 할 파일 이름을 정한다. -- //
		String domain = this.TARGET_DOMAIN;
		String cacheFileName = null;
		String reverseDomain = null;
		eMsStringTokenizer tokenDomainString = null;
		eMsStringBuffer buffer = null;
		try {
			int i = domain.indexOf(".");
			if( i < 0 ) {
				throw new RuntimeException(domain + " has no '.'");
			}

			buffer = eMsStringBuffer.getInstance();

			// 마스터도메인이름을 뺀 나머지의 거꾸로 도메인 구하고.
			buffer.setLength(0);
			buffer.append(domain.substring(i + 1));
			reverseDomain = buffer.reverseString();

			// 하부 디렉토리 구조를 만든다.
			buffer.setLength(0);
			buffer.append(DiskCacheController.sBaseDirectory);

			tokenDomainString = eMsStringTokenizer.getInstance();
			tokenDomainString.parse(reverseDomain, ".");

			while (tokenDomainString.hasMoreTokens()) {
				buffer.append("/");
				buffer.appendReverse(tokenDomainString.nextToken());

				File checkFile = new File(buffer.toString());
				if( checkFile.exists() && checkFile.isDirectory() ) {
					// 기존 디렉토리가 있으면 다음으로 진행하고.
					continue;
				}
				else if( checkFile.exists() && checkFile.isFile() ) {
					// 만약 파일이면 삭제하고
					boolean fileDel = checkFile.delete();
					if(!fileDel) log.error("File deletion failed");
				}

				// 디렉토리를 만든다.
				checkFile.mkdir();
			}
			buffer.append("/");
			buffer.append(domain);
			cacheFileName = buffer.toString();
		}
		catch(Throwable thw){
			log.error("error", thw);
			throw new RuntimeException(thw);
		}
		finally {
			eMsStringTokenizer.recycleInstance(tokenDomainString);
			tokenDomainString = null;
			eMsStringBuffer.recycleInstance(buffer);
			buffer = null;
		}
		return cacheFileName;
	}

	/**
	 * 에러처리되는 루틴.
	 * 
	 * @param thw
	 */
	public void setError(int type, Throwable thw) {
		// 에러가 발생하게 되면 디스크 케쉬는 하지 않고 메모리 상에만 존재하게 된다.
		// 에러가 발생한 도메인에 대해서 계속해서 lookup하는 것은 좀 낭비 이므로 지정횟수까지 동일한 Excepton이 발생한다면
		// 지정 시간동안은 다시 재 lookup하는 프로세스가 실행되지 않도록 프래그 처리를 해 두어야한다.
		this.SEARCH_TIME = System.currentTimeMillis();
		this.ERROR = thw;
		this.ERROR_TYPE = type;

		countErrorRaise++;

//		if( countErrorRaise > LookupCacheManager.SEARCH_ERROR_DIRECT_SEARCH_COUNT)  
//			countErrorRaise = 0;
//		}
	}
	
	/**
	 * [JOO] 초기화를 알기 위해
	 * @return
	 */
	public int getError_type(){
		return ERROR_TYPE;
	}
	public int getCountErrorRaise(){
		return countErrorRaise;
	}

	public Throwable getError() {
		return this.ERROR;
	}

	public boolean isError() {
		return this.ERROR != null;
	}

	
	
//	boolean isValidTime() {
//		// 검색시간에 구간 TTL을 더한 값이 현재시간보다 크면 그냥 사용해도 된다.
//		// 이 결과가 에러라면 에러듀레이션을 더한 시간
//		// 결과가 성공이라면 성공 듀레이션을 더한 시간을 반환한다.
//		return (this.SEARCH_TIME + (this.isError() ? LookupCacheManager.SEARCH_ERROR_DOMAIN_TTL : LookupCacheManager.SEARCH_SUCCESS_DOMAIN_TTL)) > System.currentTimeMillis();
//	}
	
	/**
	 * @return
	 * [JOO][modify] 
	 * 현재 result의 갱신 여부를 확인할때 사용한다.
	 * 기존 보다 많이 복잡해졌다. 이로인해서 발송속도에 영향이 있을 수도 있겠다.
	 * 공통메소드로 영역을 변경하였다.
	 */
	public boolean isValidTime() {
		// 검색시간에 구간 TTL을 더한 값이 현재시간보다 크면 그냥 사용해도 된다.
		// 이 결과가 에러라면 에러듀레이션을 더한 시간
		// 결과가 성공이라면 성공 듀레이션을 더한 시간을 반환한다.
		
		//에러일경우에는 에러ttl만 지나고 다시 touch를 한다.
		if(log.isDebugEnabled()) log.debug("[MXSearchResult_isValidTime]"+this.toString()+"[term_ttl]"+this.SEARCH_TIME+"[isError]"+this.isError());
		if(this.isError()){
			if(log.isDebugEnabled()) log.debug("[isValidTime]... isError! ["+Cal.getDate()+"][LOOKUP_DAY_HH24=>"+LookupCacheManager.LOOKUP_DAY_HH24+"]["+Cal.getDate(this.SEARCH_TIME + LookupCacheManager.SEARCH_ERROR_DOMAIN_TTL)+"]");
			
			boolean re_flag =(this.SEARCH_TIME + LookupCacheManager.SEARCH_ERROR_DOMAIN_TTL) > System.currentTimeMillis();
			
			//countErrorRaise 초기화
			if(!re_flag){
				this.countErrorRaise=0;
			}
			
			return re_flag;
		}else{
			//성공일경우에는 lookup.day.flag에 따라 설정한다.
			long term_ttl = this.SEARCH_TIME + LookupCacheManager.SEARCH_SUCCESS_DOMAIN_TTL;
			
			//아직 유효하다.
			if( term_ttl > System.currentTimeMillis()){
				if(log.isDebugEnabled()) log.debug("[isValidTime]... use ok! ["+Cal.getDate()+"][LOOKUP_DAY_HH24=>"+LookupCacheManager.LOOKUP_DAY_HH24+"]["+Cal.getDate(term_ttl)+"]");
				return true;
			}
			
			//touch를 지정한 시간에 진행하기를 원할때
			if(LookupCacheManager.LOOKUP_DAY_FLAG){
				int now_hh   = 0; //현재시간
				int now_day  = 0; //현재일자
				int term_hh  = 0; //term시간 = 유효시간이 지난 최초 시간
				int term_day = 0; //term일자 = 유효시간이 지난 최초 일자
					
				//유효한 시간이 아니지만 지정한 시간에 실행되기 위해 확인한다.
				// term_hh = 유효시간이 지난 최초 시간
				term_hh = Integer.parseInt(Cal.getHour(term_ttl));
				
				//term시간이 지정한 시간보다 클경우 다음날에 진행되어야한다. 
				if(term_hh > LookupCacheManager.LOOKUP_DAY_HH24 ){
					now_day  = Integer.parseInt(Cal.getDay()); 
					term_day = Integer.parseInt(Cal.getDay(term_ttl));
					//term_day가 오늘과 다르면 하루가 지난 것이다.
					if(term_day == now_day){
						//오늘 하루는 유효한 값으로 유지한다.
						if(log.isDebugEnabled()) log.debug("[isValidTime_do_next_day]... use ok! ["+Cal.getDate()+"][LOOKUP_DAY_HH24=>"+LookupCacheManager.LOOKUP_DAY_HH24+"]["+Cal.getDate(term_ttl)+"]");
						return true;
					}
				}
				
				
				//지정한 시간이 되기 전에는 유효하다.
				now_hh   = Integer.parseInt(Cal.getHour());
				if(log.isDebugEnabled()) log.debug("[isValidTime_do__today]...["+Cal.getDate()+"][LOOKUP_DAY_HH24=>"+LookupCacheManager.LOOKUP_DAY_HH24+"]["+Cal.getDate(term_ttl)+"]"+(now_hh < LookupCacheManager.LOOKUP_DAY_HH24));
				return now_hh < LookupCacheManager.LOOKUP_DAY_HH24 ;
			}else{
				return false;
			}
		}
	}
	
	/**
	 * @return
	 * [JOO][modify]
	 * 성공일 경우에 한해서 현재 result의 재사용 여부를 결정할때 사용한다.
	 */
	public boolean isValidTimeSucc(long search_time) {
		//성공일경우에는 lookup.day.flag에 따라 설정한다.
		long term_ttl = search_time + LookupCacheManager.SEARCH_SUCCESS_DOMAIN_TTL;
		
		if(log.isDebugEnabled()) log.debug("[MXSearchResult_isValidTimeSucc]"+this.toString()+"[term_ttl]"+term_ttl+"[isError]"+this.isError());
		
		//아직 유효하다.
		if( term_ttl > System.currentTimeMillis()){
			return true;
		}
		
		//touch를 지정한 시간에 진행하기를 원할때
		if(LookupCacheManager.LOOKUP_DAY_FLAG){
			int now_hh   = 0; //현재시간
			int now_day  = 0; //현재일자
			int term_hh  = 0; //term시간 = 유효시간이 지난 최초 시간
			int term_day = 0; //term일자 = 유효시간이 지난 최초 일자
				
			//유효한 시간이 아니지만 지정한 시간에 실행되기 위해 확인한다.
			// term_hh = 유효시간이 지난 최초 시간
			term_hh = Integer.parseInt(Cal.getHour(term_ttl));
			
			//term시간이 지정한 시간보다 클경우 다음날에 진행되어야한다. 
			if(term_hh > LookupCacheManager.LOOKUP_DAY_HH24 ){
				now_day  = Integer.parseInt(Cal.getDay()); 
				term_day = Integer.parseInt(Cal.getDay(term_ttl));
				//term_day가 오늘과 다르면 하루가 지난 것이다.
				if(term_day == now_day){
					//오늘 하루는 유효한 값으로 유지한다.
					if(log.isDebugEnabled()) log.debug("[isValidTimeSucc_do_next_day]... use ok! ["+Cal.getDate()+"][LOOKUP_DAY_HH24=>"+LookupCacheManager.LOOKUP_DAY_HH24+"]["+Cal.getDate(term_ttl)+"]");
					return true;
				}
			}
			
			//지정한 시간이 되기 전에는 유효하다.
			now_hh   = Integer.parseInt(Cal.getHour());
			if(log.isDebugEnabled()) log.debug("[isValidTimeSucc_do__today]...["+Cal.getDate()+"][LOOKUP_DAY_HH24=>"+LookupCacheManager.LOOKUP_DAY_HH24+"]["+Cal.getDate(term_ttl)+"]"+(now_hh < LookupCacheManager.LOOKUP_DAY_HH24));
			return now_hh < LookupCacheManager.LOOKUP_DAY_HH24 ;
		}else{
			return false;
		}
	}

	public void init(int initialCapacity) {
		/* 
		if( this.domainDATA != null ) {
			this.clear();
		}

		this.domainDATA = new String[initialCapacity];
		this.domainIP = new int[initialCapacity];
		*/
		
		/*
		 * 새롭게 도메인 관리를 위해 수정
		 * for eMs 6.0 upgrade
		 */
		if( this.mxDomain != null ) {
			this.clear();
		}

		this.mxDomain = new MXDomain[initialCapacity];
		for(int i=0; i<initialCapacity; i++) {
			this.mxDomain[i] = new MXDomain();
		}
		
		this.size = initialCapacity;
		this.fullName = null;
		// 세팅을 시작하는 것이므로 에러가 아니다.
		this.ERROR = null;
		this.ERROR_TYPE = DNS.ERROR_INITIALIZE;
		countErrorRaise = 0;
	}

	/*
	public void clear() {
		// Let gc do its work
		for (int i = 0; i < this.size; i++) {
			this.domainDATA[i] = null;
			this.domainIP[i] = 0;
		}
		this.fullName = null;
		this.domainDATA = null;
		this.domainIP = null;
		this.SEARCH_TIME = Long.MIN_VALUE;
	}
	*/
	
	
	public void clear() {
		// Let gc do its work
		for (int i = 0; i < this.size; i++) {
			this.mxDomain[i].reset();
		}
		this.fullName = null;
		this.SEARCH_TIME = Long.MIN_VALUE;
	}

	public synchronized void set(int idx, String domain, int ip) {
		this.mxDomain[idx].setHost(domain);
		this.mxDomain[idx].setIp(ip);
		// this.mxDomain[idx].setErrorTime(System.currentTimeMillis());
		// this.mxDomain[idx].setMx(0);
		return;
	}
	
	public long getErrorTime(int idx){
		return this.mxDomain[idx].getErrorTime();
	}

	/**
	 * 해당 idx에 해당하는 MX에 값에 에러카운트와 에러발생시간을 설정한다.
	 * 
	 * @param idx
	 * @param errorTime
	 */
	public synchronized void setMXError(int idx, long errorTime){
		this.mxDomain[idx].setErrorCnt(this.mxDomain[idx].getErrorCnt()+1);		
		this.mxDomain[idx].setErrorTime(errorTime);		
	}

	/**
	 * 해당 idx에 해당하는 MX에 값을 초기화 시킨다.
	 * @param idx
	 */
	public synchronized void resetMXError(int idx){
		this.mxDomain[idx].setErrorTime(-1);		
		this.mxDomain[idx].setErrorCnt(0);		
	}
	
	public synchronized int getMXErrorCnt(int idx){
		return this.mxDomain[idx].getErrorCnt();
	}

	public int getTargetIndex(int AGENT_INDEX, int SERVER_INDEX) {
		if( size < SERVER_INDEX )
			return -1;
		int TARGET_INDEX = 0;
		if( SERVER_INDEX > this.INDEX_OF_FIRST_REFERENCE_SEQUENCE ) {
			if (log.isDebugEnabled()) {
				LookupCacheManager.log("DNSList - TARGET direct IDX:" + String.valueOf(SERVER_INDEX - 1));
			}
			TARGET_INDEX = SERVER_INDEX - 1;
		}
		else {
			if (log.isDebugEnabled()) {
				LookupCacheManager.log("DNSList - TARGET rotate IDX:" + ((AGENT_INDEX + SERVER_INDEX) % this.INDEX_OF_FIRST_REFERENCE_SEQUENCE));
			}
			TARGET_INDEX = (AGENT_INDEX + SERVER_INDEX) % this.INDEX_OF_FIRST_REFERENCE_SEQUENCE;
		}

		TARGET_INDEX = TARGET_INDEX < size ? TARGET_INDEX : size - 1;
		TARGET_INDEX = TARGET_INDEX < 0 ? 0 : TARGET_INDEX;

		return TARGET_INDEX;
	}
	
	/**
	 * 등록된 도메인 리스트를 MX preference 순번에 의하여 로테이션 시키면서 index 를 반환
	 **/
	public synchronized int getTargetIndex() {
			
		int TARGET_INDEX = -1;

		for(int i=0; i<this.INDEX_OF_FIRST_REFERENCE_SEQUENCE; i++){
			
			this.domainIndex = this.domainIndex % this.INDEX_OF_FIRST_REFERENCE_SEQUENCE;
			TARGET_INDEX = this.domainIndex++;
			
			// 에러가 20번 이상 발생한 MX에서,
			// by hmall 2009.06.17 50 -> 100
			if(getMXErrorCnt(TARGET_INDEX) > 100) {

				// 에러가 발생한 시간부터 사용되지 못하도록 설정된 시간이 지나지 않았고,
				if(getErrorTime(TARGET_INDEX) + eMsLocale.DISABLED_TIME_OF_MX > System.currentTimeMillis()) {

					// MX의 개수가  여러개이면, 다음 MX 값을 가지고 온다.
					if(this.INDEX_OF_FIRST_REFERENCE_SEQUENCE > 0){
						DisabledMXLogger.put(this.mxDomain[TARGET_INDEX].getHost() + " is disabled. Use Next mxDomain. ErrorCount : " + this.mxDomain[TARGET_INDEX].getErrorCnt() + ", ErrorTime : " + this.mxDomain[TARGET_INDEX].getErrorTime() );
						TARGET_INDEX = -1;
						continue;
						
					// MX의 개수가  1개면, 지정된 시간 안이면 에러를 발생시킨다.
					} else if(this.INDEX_OF_FIRST_REFERENCE_SEQUENCE == 1){
						DisabledMXLogger.put(this.mxDomain[TARGET_INDEX].getHost() + " is disabled. ErrorCount : " + this.mxDomain[TARGET_INDEX].getErrorCnt() + ", ErrorTime : " + this.mxDomain[TARGET_INDEX].getErrorTime() );
						TARGET_INDEX = -1;
						break;
					}

				// 시간이 지나면은 초기화 시킨다.. 
				} else {
					resetMXError(TARGET_INDEX);
					break;
				}
				
			} else {	
				break;
			}			
		} // End for

		return TARGET_INDEX;
	}
	
	public int getDomainSize() {
		return this.mxDomain.length;
	}

	/**
	 * 등록된 도메인 리스트를 MX preference 순번에 의하여 로테이션 시키면서 스트링 배열로 반환
	 * 
	 * @return MX Record 스트링 배열
	 */
	public synchronized String getDomain(int idx) {
		return this.mxDomain[idx].getHost();
	}

	/**
	 * 등록된 도메인 리스트를 MX preference 순번에 의하여 로테이션 시키면서 스트링 배열로 반환
	 * 
	 * @return MX Record 스트링 배열
	 */
	public synchronized int getIP(int idx) {
		return this.mxDomain[idx].getIp();
	}

	public synchronized String toString() {
		if( fullName != null ) {
			return fullName;
		}

		StringBuffer buffer = null;
		try {
			buffer = new StringBuffer();
			buffer.append(this.TARGET_DOMAIN);
			buffer.append(":");
			if( isError() ) {
				buffer.append(this.ERROR.toString());
			}
			else {
				for (int i = 0; i < this.size; i++) {
					buffer.append("(");
					buffer.append(this.mxDomain[i].getHost());
					buffer.append("/");
					buffer.append(LookupUtil.getIntIPToString(this.mxDomain[i].getIp()));
					buffer.append(")");
				}
			}
			this.fullName = buffer.toString();
		}
		finally {
			buffer = null;
		}
		return this.fullName;
	}
}
