/*
 * Created on 2005. 12. 1
 */
package pluto.mail.mx;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;

import lombok.extern.slf4j.Slf4j;
import pluto.mail.mx.exception.CanonicalNameFoundException;
import pluto.mail.mx.exception.NoMXRecordFoundException;
import pluto.net.NetAddress;

/**
 * @author 이상근
 */
@Slf4j
public class MXResultSorter {

	static final CanonicalNameFoundException	CANONICAL_NAME_FOUND_EXCEPTION	= new CanonicalNameFoundException("final");

	/**
	 * 갱신할지 여부를 판단
	 */
	long										serial							= -1;

	long										ttl								= -1;

	/**
	 * MX Record preference중 가장 작은 인덱스를 저장
	 */
	int											minref							= -1;

	String										SOA_MASTER_DNS					= null;

	String										NORMAL_DNS						= null;

	int											SOA_MASTER_DNS_IP				= 0;

	boolean										exactSOA						= false;

	// -- MX List Part Member -- //

	/**
	 * List의 기준 헤더 member
	 */
	protected Entry								header							= new Entry("header", -1, null, null);

	public MXResultSorter() {
		// -- MX List Part init -- //
		header.next = header.previous = header;
	}

	public synchronized void putMailExchanger(DNSInputStream dnsIn) throws IOException {
		int preference = dnsIn.readShort();

		String mx = dnsIn.readDomainName();

		// MailExchanger를 등록한다.
		addMailExchanger(mx, preference);
	}

	public synchronized void putAddress(String domain, DNSInputStream dnsIn) throws IOException {
		int i1 = dnsIn.readByte();
		int i2 = dnsIn.readByte();
		int i3 = dnsIn.readByte();
		int i4 = dnsIn.readByte();

		// 추가한다.
		addAddress(domain, LookupUtil.getIntIPToInetInt(i1, i2, i3, i4));
	}

	public synchronized void putStartOfAuth(DNSInputStream dnsIn) throws IOException {
		if (log.isDebugEnabled()) {
			log.debug("MXResultSorter put SOA");
		}
		this.SOA_MASTER_DNS = dnsIn.readDomainName();
		if (log.isDebugEnabled()) {
			log.debug("origin:" + this.SOA_MASTER_DNS);
		}

		if (log.isDebugEnabled()) {
			String mailAddress = dnsIn.readDomainName();
			log.debug("mailAddress:" + mailAddress);
		}
		else {
			dnsIn.readDomainName();
		}

		// serial 을 저장한다.
		this.serial = dnsIn.readInt();
		if (log.isDebugEnabled()) {
			log.debug("serial:" + serial);
		}

		if (log.isDebugEnabled()) {
			long refresh = dnsIn.readInt();
			log.debug("refresh:" + refresh);
		}
		else {
			dnsIn.readInt();
		}

		if (log.isDebugEnabled()) {
			long retry = dnsIn.readInt();
			log.debug("retry:" + retry);
		}
		else {
			dnsIn.readInt();
		}

		if (log.isDebugEnabled()) {
			long expire = dnsIn.readInt();
			log.debug("expire:" + expire);
		}
		else {
			dnsIn.readInt();
		}

		this.ttl = dnsIn.readInt();
		if (log.isDebugEnabled()) {
			log.debug("ttl:" + ttl);
		}
	}

	public synchronized void addNameServer(String domain) {
		this.NORMAL_DNS = domain;

		if( this.NORMAL_DNS.equalsIgnoreCase(this.SOA_MASTER_DNS) ) {
			this.exactSOA = true;
		}
	}

	public synchronized void addAddress(String domain, int iIP) {
		// 마스터 도메인의 A Record가 들어온다면...
		if( this.SOA_MASTER_DNS != null && domain.equalsIgnoreCase(this.SOA_MASTER_DNS) ) {
			this.SOA_MASTER_DNS_IP = iIP;
		}

		// NULL 이 올수는 없지만 온다면 스킵해야한다.
		if( domain == null ) {
			if (log.isDebugEnabled()) {
				throw new RuntimeException("DOMAIN IS NULL");
			}
			return;
		}

		Entry nextEntry = header.next;

		do {
			if( nextEntry == header ) {
				// 찾지 못했는데 헤더를 만나면 pref = -1로 M 을 만들어야한다.
				//				Entry newEntry = new Entry(domain, -1, nextEntry.next, nextEntry);
				//				newEntry.previous.next = newEntry;
				//				newEntry.next.previous = newEntry;
				//				newEntry.listArecord.add(iIP);
				return;
			}

			if( domain.equalsIgnoreCase(nextEntry.domain) ) {
				// 같은 도메인으로 등록된 녀석을 찾으면 append 해야한다.
				nextEntry.listArecord.add(iIP);
				return;
			}

			nextEntry = nextEntry.next;
		} while (true);
	}

	boolean	hasCanonicalName	= false;

	/**
	 * A Record중에 CanonicalName을 가진 녀석이 있는지를 반환한다.
	 * 
	 * @return
	 */
	public boolean hasCanonical() {
		return hasCanonicalName;
	}

	/**
	 * CanonicalName 처리를 하고 호출된다.
	 *  
	 */
	public void clearCanonical() {
		this.hasCanonicalName = false;
	}

	public synchronized void addCanonicalName(String fromDomain, String toDomain) {
		// 두도메인이 같으면 Canonical Name이 아니다.
		if( fromDomain.equalsIgnoreCase(toDomain) ) {
			return;
		}
		hasCanonicalName = true;

		// -- 만일 Canonical Name을 가진 녀석이 MX안에 동시에 존재할 경우를 대비해서 치환할 대상을 저장할 공간을 만든다 -- //
		Entry targetEntry = null;
		Entry sourceEntry = null;

		// -- 엔트리 중에서 fromDomain을 가진 녀석들은 toDomain으로 치환해야한다.
		Entry nextEntry = header.next;

		while (nextEntry != header) {
			// 만일 목적이 되는 entry가 있으면 그녀석을 엔트리에서 삭제한다.
			if( nextEntry.domain.equalsIgnoreCase(toDomain) ) {
				targetEntry = nextEntry;
			}

			if( nextEntry.domain.equalsIgnoreCase(fromDomain) ) {
				sourceEntry = nextEntry;
			}

			nextEntry = nextEntry.next;
		}

		if( sourceEntry == null ) {
			// 과연 이런 경우가 있을 수 있을까?
			log("HAS CANONICAL BUT SOURCE ENTRY IS NULL");
			return;
		}

		if( targetEntry == null ) {
			// source만 존재하므로 소스만 고치면 된다.
			sourceEntry.domain = toDomain;
		}
		else {
			// 둘다 존재하므로 pref가 작은거를 남기고 큰거를 없앤다.
			if( targetEntry.preference < sourceEntry.preference ) {
				sourceEntry.next.previous = sourceEntry.previous;
				sourceEntry.previous.next = sourceEntry.next;
				sourceEntry.destroy();
			}
			else {
				targetEntry.next.previous = targetEntry.previous;
				targetEntry.previous.next = targetEntry.next;
				targetEntry.destroy();
			}

			throw CANONICAL_NAME_FOUND_EXCEPTION;
		}
	}

	public synchronized void addMailExchanger(String domain, int preference) {
		// NULL 이 올수는 없지만 온다면 스킵해야한다.
		if( domain == null ) {
			if (log.isDebugEnabled()) {
				throw new RuntimeException("DOMAIN IS NULL");
			}
			return;
		}

		Entry nextEntry = header.next;

		do {
			// 만일 MX 보다 A 가 먼저 들어와서 만들어 졌을 경우를 생각해야한다.
			// A가 들어와서 MX가 없으면 preference를 -1 로 설정해서 만들어 놓을 것이니까... 그걸 지금의 preference로 치환하는
			// 로직이 있어야 한다.

			if( nextEntry == header || preference < nextEntry.preference ) {
				Entry newEntry = new Entry(domain, preference, nextEntry, nextEntry.previous);
				newEntry.previous.next = newEntry;
				newEntry.next.previous = newEntry;
				return;
			}

			nextEntry = nextEntry.next;
		} while (true);
	}

	public void clear() {
		// 기본값들 초기화하고
		serial = -1;
		minref = -1;
		this.SOA_MASTER_DNS = null;
		this.NORMAL_DNS = null;
		SOA_MASTER_DNS_IP = 0;

		Entry nextEntry = header.next;

		while (nextEntry != header) {
			// 마지막 인덱스 잠시 저장하고.
			Entry targetEntry = nextEntry;

			// 다음 인덱스 저장하고.
			nextEntry = targetEntry.next;

			targetEntry.next.previous = targetEntry.previous;
			targetEntry.previous.next = targetEntry.next;

			// 현재 인덱스 clear한다.
			targetEntry.clear();
		}
	}

	public void destroy() {
		// 파기할 때에는 헤더도 null 처리해준다.
		clear();
		header = null;
	}

	public void printResult() {
		log.debug("min:" + String.valueOf(this.minref));
		log.debug("serial:" + String.valueOf(this.serial));

		Entry nextEntry = header.next;

		while (nextEntry != header) {
			log.debug((nextEntry.preference < 0 ? "ETC:" : "MailExchanger:") + nextEntry.domain + "/" + String.valueOf(nextEntry.preference));
			for (int iIPs = nextEntry.listArecord.size; iIPs > 0; iIPs--) {
				int iIP = nextEntry.listArecord.get(iIPs - 1);
				byte[] byteIP = LookupUtil.getIntToInetByteArray(iIP);
				log.debug("Address:" + ((byteIP[0] & 0xFF) + "." + (byteIP[1] & 0xFF) + "." + (byteIP[2] & 0xFF) + "." + (byteIP[3] & 0xFF)));
			}

			nextEntry = nextEntry.next;
		}
	}

	/**
	 * 초기 MX 레코드 수를 기록한다.
	 */
	public int	minMxCount	= 0;

	/**
	 * 초기 MX가 아닌 녀석들을 저장한다.
	 */
	public int	etcMxCount	= 0;

	public void calculateMxCounts() throws NoMXRecordFoundException {
		this.minMxCount = 0;
		this.etcMxCount = 0;

		// -- MX 중에서 IP를 하나도 가지지 않은 레코드는 제거하고 인덱스를 다시 조정한다. -- //
		
		Entry nextEntry = header.next;
		
		this.minref = Integer.MAX_VALUE;
		while (nextEntry != header) {
			if(nextEntry.listArecord.size==0){
				// Bingo!!!
				nextEntry.previous.next = nextEntry.next;
				nextEntry.next.previous = nextEntry.previous;
				nextEntry.destroy();
				
				// 처음 헤더부터 다시 돌아야 한다.
				this.minref = Integer.MAX_VALUE;
				nextEntry = header.next;
				continue;
			}
			
			if( nextEntry.preference < this.minref ) {
				this.minref = nextEntry.preference;
			}
			nextEntry = nextEntry.next;
		}

		// -- 가장 낮은 Preference와 그외것들을 정리한다. -- //
		nextEntry = header.next;
		
		while (nextEntry != header) {
			if( nextEntry.preference == this.minref ) {
				this.minMxCount += nextEntry.listArecord.size;
			}
			else if( nextEntry.preference >= 0 ) {
				this.etcMxCount += nextEntry.listArecord.size;
			}

			nextEntry = nextEntry.next;
		}

		if( this.minMxCount == 0 ) {
			throw new NoMXRecordFoundException("NO MX Record or MX Record has no A Record");
		}
	}

	public int getMXDomainCount() {
		int returnValue = 0;
		Entry nextEntry = header.next;

		while (nextEntry != header) {
			if( nextEntry.preference >= 0 ) {
				returnValue++;
			}

			nextEntry = nextEntry.next;
		}

		return returnValue;
	}

	public void setResult(MXSearchResult result) {
		int idx = 0;
		Entry nextEntry = header.next;

		// 최초 인덱스를 저장해줘야한다.
		result.INDEX_OF_FIRST_REFERENCE_SEQUENCE = this.minMxCount;

		result.SEARCH_TIME = System.currentTimeMillis();

		result.setError(DNS.ERROR_NO_ERROR, null);

		while (nextEntry != header) {
			if (log.isDebugEnabled()) {
				log.debug((nextEntry.preference < 0 ? "ETC:" : "MailExchanger:") + nextEntry.domain + "/" + String.valueOf(nextEntry.preference));
			}

			if( nextEntry.preference >= 0 ) {
				for (int iIPs = nextEntry.listArecord.size; iIPs > 0; iIPs--) {
					int iIP = nextEntry.listArecord.get(iIPs - 1);
					if (log.isDebugEnabled()) {
						byte[] byteIP = LookupUtil.getIntToInetByteArray(iIP);
						log.debug("Address:" + ((byteIP[0] & 0xFF) + "." + (byteIP[1] & 0xFF) + "." + (byteIP[2] & 0xFF) + "." + (byteIP[3] & 0xFF)));
					}
					// 해당 인덱스에 domain / ip를 세팅한다.
					result.set(idx, nextEntry.domain, iIP);
					idx++;
				}
			}

			nextEntry = nextEntry.next;
		}
	}

	public InetAddress getSOAInetAddress() throws UnknownHostException {
		// 도메인은 있지만 아이피가 없을 경우에는 가져와야한다.
		if( this.SOA_MASTER_DNS_IP == 0 && this.SOA_MASTER_DNS != null ) {
			if (log.isDebugEnabled()) {
				log("no soa contain ip:" + this.SOA_MASTER_DNS);
			}
			InetAddress inet = InetAddress.getByName(this.SOA_MASTER_DNS);
			if( this.exactSOA ) {
				inet = InetAddress.getByName(this.SOA_MASTER_DNS);
			}
			else {
				inet = InetAddress.getByName(this.NORMAL_DNS);
			}
			String hostip = inet.getHostAddress();
			int[] address = NetAddress.getIntArray(hostip);
			this.SOA_MASTER_DNS_IP = LookupUtil.getIntIPToInetInt(address[0], address[1], address[2], address[3]);
		}

		if( this.SOA_MASTER_DNS_IP == 0 ) {
			return null;
		}

		byte[] address = LookupUtil.getIntToInetByteArray(this.SOA_MASTER_DNS_IP);

		return InetAddress.getByAddress(address);
	}

	static final void log(String log) {
		LookupCacheManager.log(log);
	}

	static final void log(Throwable log) {
		LookupCacheManager.log(log);
	}

	protected static class Entry {
		protected String			domain;

		protected int				preference	= -1;

		protected Entry				next;

		protected Entry				previous;

		protected AddressArrayList	listArecord	= null;

		Entry(String d, int pref, Entry n, Entry p) {
			this.domain = d;
			this.next = n;
			this.previous = p;
			this.preference = pref;
			this.listArecord = new AddressArrayList();
		}

		protected void putARecord(int aRecord) {
			this.listArecord.add(aRecord);
		}

		void setPreference(int pref) {
			this.preference = pref;
		}

		void clear() {
			this.domain = null;
			this.next = null;
			this.previous = null;
			this.listArecord.clear();
		}

		void destroy() {
			this.domain = null;
			this.next = null;
			this.previous = null;
			this.listArecord.destroy();
		}
	}

	static class AddressArrayList {
		private int[]	elementData;

		private int		size;

		AddressArrayList(int initialCapacity) {
			this.elementData = new int[initialCapacity];
		}

		AddressArrayList() {
			this(10);
		}

		void ensureCapacity(int minCapacity) {
			int oldCapacity = elementData.length;
			if( minCapacity > oldCapacity ) {
				int oldData[] = elementData;
				int newCapacity = (oldCapacity * 3) / 2 + 1;
				if( newCapacity < minCapacity ) {
					newCapacity = minCapacity;
				}
				elementData = new int[newCapacity];
				System.arraycopy(oldData, 0, elementData, 0, this.size);
			}
		}

		int size() {
			return size;
		}

		int get(int index) {
			return elementData[index];
		}

		void add(int o) {
			// 같은 아이피가 들어오면 있을 필요가 없으므로..
			for (int i = 0; i < size; i++) {
				if( elementData[i] == o ) {
					return;
				}
			}
			ensureCapacity(size + 1);
			elementData[size++] = o;
		}

		void clear() {
			this.size = 0;
		}

		void destroy() {
			this.size = -1;
			this.elementData = null;
		}
	}
}
