/*
 * Created on 2005. 11. 29
 */
package pluto.mail.mx;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.net.InetAddress;

import pluto.io.eMsByteArrayOutputStream;
import lombok.extern.slf4j.Slf4j;
import pluto.mail.mx.MXResultSorter.Entry;
import pluto.mail.mx.exception.CanonicalNameFoundException;
import pluto.mail.mx.exception.FormatErrorException;
import pluto.mail.mx.exception.LookupUnknownException;
import pluto.mail.mx.exception.NameNotKnownException;
import pluto.mail.mx.exception.NotImplementedException;
import pluto.mail.mx.exception.ResultArrayContainEOFException;
import pluto.mail.mx.exception.ServerFailureException;
import pluto.mail.mx.exception.ServerRefusedException;
import pluto.mail.mx.exception.TooManyMissMatchQueryIDException;
import pluto.net.NetworkMonitorable;
import pluto.net.SocketAgentMonitor;
import pluto.util.FIFOBuffer;
import pluto.util.eMsStringTokenizer;

/**
 * @author 이상근
 */
@Slf4j
public abstract class MXLookup implements NetworkMonitorable {

	protected String				CONNECT_HOST		= null;

	protected boolean				IN_COMM				= false;

	protected int					TIME_OUT			= 0;

	protected long					COMM_START_TIME		= 0;

	/**
	 * DNS 쿼리시 기본 타임아웃을 설정한다.
	 */
	public static final int			DEFAULT_TIME_OUT	= 10 * 1000;

	private static int				globalID			= 0;

	static ByteArrayOutputStream	byteArrayOut		= null;

	static DataOutputStream			dataOut				= null;

	static eMsStringTokenizer		token				= null;

	//자주 사용되기 때문에 재사용을 위한 버퍼를 제공한다.
	private static final int		MAX_CONTAIN_SIZE	= 10;

	private static FIFOBuffer		INNER_CONTAINER		= null;

	static {
		// -- initialize byte array buffers -- //
		byteArrayOut = new ByteArrayOutputStream();
		dataOut = new DataOutputStream(byteArrayOut);
		token = new eMsStringTokenizer();

		INNER_CONTAINER = new FIFOBuffer(MAX_CONTAIN_SIZE);
	}

	synchronized static final byte[] getQuery(String queryHost, int queryID, int type) throws IOException {
		dataOut.flush();

		byteArrayOut.flush();
		byteArrayOut.reset();

		try {
			dataOut.writeShort(queryID);
			dataOut.writeShort((0 << DNS.SHIFT_QUERY) | (DNS.OPCODE_QUERY << DNS.SHIFT_OPCODE) | (1 << DNS.SHIFT_RECURSE_PLEASE));
			dataOut.writeShort(1); // # queries
			dataOut.writeShort(0); // # answers
			dataOut.writeShort(0); // # authorities
			dataOut.writeShort(0); // # additional
			token.parse(queryHost, ".");
			while (token.hasMoreTokens()) {
				String label = token.nextToken();
				dataOut.writeByte(label.length());
				dataOut.writeBytes(label);
			}
			dataOut.writeByte(0);
			dataOut.writeShort(type);
			dataOut.writeShort(DNS.CLASS_ANY);
		}
		catch(IOException ignored) {
		}

		byte[] query = byteArrayOut.toByteArray();

		return query;
	}

	static final MXLookup getInstance() {
		Object tmpValue = INNER_CONTAINER.pop();
		
		MXLookup returnValue = null;
		
		if( tmpValue == null ) {
			log("create new MXLookup");
			 returnValue = new UDPMXLookup();	// hmall 20090615
		}
		else{
			log("pop up old MXLookup");
			returnValue = (MXLookup) tmpValue;
		}

		return returnValue;
	}

	public static void recycleInstance(MXLookup tmp) {
		if( tmp == null ) {
			return;
		}

		if( INNER_CONTAINER.push(tmp) ) {
			// 재활용대상이 되면 내용만지우고
			tmp.reset();
		}
		else {
			// 재활용이 안되면 자원 전체를 반환해야한다.
			tmp.destroy();
		}
	}

	public static void log(String log) {
		LookupCacheManager.log(log);
	}

	public static void log(Throwable log) {
		LookupCacheManager.log(log);
	}

	String					domain	= null;

	public MXResultSorter	sorter	= new MXResultSorter();

	MXLookup() {
		// 모니터에 등록해야 하기 때문에
		SocketAgentMonitor.registSocketAgent(this);

		this.sorter = new MXResultSorter();
	}

	void set(String d) {
		this.domain = d;
	}

	final void reset() {
		this.domain = null;

		// sorter에서 사용했던 것들을 지워준다.
		this.sorter.clear();
	}

	final void destroy() {
		this.domain = null;

		// sorter의 모든 것을 날려버린다.
		this.sorter.destroy();
	}

	/**
	 * 
	 * @param result
	 * @throws Exception
	 */
	final void refesh(MXSearchResult result) throws Exception {
		String sDomain = result.TARGET_DOMAIN;

		// -- 디렉토리에서 먼저 찾아 놓은 결과가 있는지를 가져온다. -- //
		switch (DiskCacheController.checkCacheResult(result)) {
			case DiskCacheController.STATE_READ_FROM_CACHE: {
				// 케쉬에서 읽었으므로 그냥 돌아간다.
				if (log.isDebugEnabled()) {
					log(sDomain + " is read from cache");
				}
				return;
			}

			case DiskCacheController.STATE_CREATE_CACHE:
			case DiskCacheController.STATE_READ_FROM_CACHE_INVALID: {
				// 파일이 없거나 expire된 녀석 이라면 다시 lookup해야한다.
				if (log.isDebugEnabled()) {
					log(sDomain + " is not in cache or expire");
				}
				break;
			}

			default: {
				break;
			}
		}

		eMsByteArrayOutputStream buffer = null;
		try {
			buffer = eMsByteArrayOutputStream.getInstance();
			Throwable ex = null;
			//**** 먼저 NS를 Lookup 해서 serial / ttl을 가져온다. ****//
			for (int i = 0; i < LookupCacheManager.DNS_RESOLVER_LIST.length; i++) {
				try {
					// DNS Server 수만큼 반복해서 해야한다.
					refreshToTargetDNS(buffer, LookupCacheManager.DNS_RESOLVER_LIST[i], result);
					
					//[JOO]위에서 Exception 이 발생하지 않았다면 여기서 멈춘다.
					//하지만 Exception 이 발생했다면 for문이 계속 진행된다.
					ex = null;//앞의 DNS에서 오류가 있었다면 삭제한다. 성공했기 때문에...
					break;
					
				}catch(Throwable thw) {
					// TODO-ErrorProcess
					// thw.printStackTrace();
					//log.debug("[refesh-ERROR]"+result.toString()+"[thw]"+thw.toString());
					ex = thw;
				}
			}
			
			//[JOO] modify - 어차피 최종 exception이 error로 처리된다.
			if(ex != null){
				result.setError(DNS.ERROR_REFRESH_TO_TARGET_DNS, ex);
			}
			
			
		}finally {
			eMsByteArrayOutputStream.recycleInstance(buffer);
			buffer = null;
		}
	}

	public abstract void close();

	protected abstract int getDNSResult(InetAddress address, byte[] query, eMsByteArrayOutputStream buffer) throws Throwable;

	final void refreshQueryTargetType(eMsByteArrayOutputStream buffer, InetAddress address, String domain, int type, MXSearchResult result) throws Throwable {
		int responseLength = -1;

		// -- 먼저 MX 를 룩업해서 기타 정보를 먼저 가져온다. -- //
		boolean bSearchResult = false;
		int iTryCount = 0;
		do {
			iTryCount++;

			int query_id = (++globalID) % 65536;

			byte[] queryNS = getQuery(domain, query_id, type);

			responseLength = getDNSResult(address, queryNS, buffer);

			// -- 수신한 데이터를 분석하여 구조체 작성을 시작한다. -- //
			bSearchResult = receiveResponse(query_id, this.sorter, buffer, responseLength, result);

			if( iTryCount > 3 ) {
				// 삼세번 실행해서 안되면 에러를 발생시킨다.
				throw new TooManyMissMatchQueryIDException(domain + ":Too Many Miss QueryID");
			}
		} while (bSearchResult);

	}

	final void refreshToTargetDNS(eMsByteArrayOutputStream buffer, InetAddress address, MXSearchResult result) throws Throwable {
		int responseLength = -1;

		try {
			// -- 먼저 MX 를 룩업해서 기타 정보를 먼저 가져온다. -- //
			//			if (log.isDebugEnabled()) {
			//				log.debug("MX Search");
			//			}
			refreshQueryTargetType(buffer, address, result.TARGET_DOMAIN, DNS.TYPE_MX, result);

			if( this.sorter.getMXDomainCount() == 0 ) {
				this.sorter.addMailExchanger(result.TARGET_DOMAIN, 10);
			}

			// -- A record Lookup -- //
			do {
				//				if (log.isDebugEnabled()) {
				//					log.debug("A Search");
				//				}
				// Canonical flag를 초기화하고 A Record를 갱신하는 동안 Canonical Name을 가지지 않을 때까지 반복한다.
				this.sorter.clearCanonical();
				refreshARecords(buffer, address, result);
			} while (this.sorter.hasCanonical());

			// -- 결과를 모두 sorter에 저장하였으므로 sorter의 정보를 result로 세팅한다. -- //
			// sorter에 저장된 리스트의 카운트를 정비하고
			this.sorter.calculateMxCounts();
		}
		catch(Throwable thw) {
			throw thw;
		}
		finally {
			close();
		}

		// sorter에 저장된 리스트의 카운트로 result를 초기화하고
		result.init(this.sorter.minMxCount + this.sorter.etcMxCount);

		// sorter의 정보를 result로 세팅한다.
		this.sorter.setResult(result);

		// 정보가 다 세팅되었으면 Cache로 저장해야한다.
		DiskCacheController.storeResultToCache(result);
	}

	final void refreshARecords(eMsByteArrayOutputStream buffer, InetAddress address, MXSearchResult result) throws Throwable {
		Entry header = this.sorter.header;

		Entry nextEntry = header.next;

		while (nextEntry != header) {
			//			if (log.isDebugEnabled()) {
			//				log.debug();
			//				log.debug((nextEntry.preference < 0 ? "ETC:" : "MailExchanger:") + nextEntry.domain + "/" + String.valueOf(nextEntry.preference));
			//			}
			String domain = nextEntry.domain;

			if( nextEntry.preference >= 0 ) {
				try {
					int iIP = LookupUtil.getPureDigitIP(domain);
					// -- 도메인에 해당하는 A Record를 가져온다. -- //
					if( iIP != 0 ) {
						// MX에 아이피로 지정되어 있으면 룩업할 필요없이 그냥 세팅한다.
						this.sorter.addAddress(domain, iIP);
					}
					else {
						refreshQueryTargetType(buffer, address, domain, DNS.TYPE_A, result);
					}
				}
				catch(CanonicalNameFoundException e) {
					// CanonicalName을 찾았으면 첨부터 다시 룩업한다.
					nextEntry = this.sorter.header.next;
				}
				catch(Exception e) {
					// TODO ErrorProcess
					// log.debug("A Record Search Error" + e.toString());
				}
			}

			nextEntry = nextEntry.next;
		}
	}

	static boolean receiveResponse(int query_id, MXResultSorter sorter, eMsByteArrayOutputStream buffer, int length, MXSearchResult result) throws Throwable {
		byte[] rawBuffer = buffer.getRawByteArray();

		// 원조 데이터 구조체
		DNSInputStream dnsIn = new DNSInputStream(rawBuffer, 0, length);
		int id = dnsIn.readShort();

		if( id != query_id ) {
			//			if (log.isDebugEnabled()) {
			//				log.debug("####################################DIFFRENT QUERY ID ERROR ");
			//			}
			return true;
		}

		int flags = dnsIn.readShort();
		decodeFlags(flags, result);
		int numQueries = dnsIn.readShort();
		int numAnswers = dnsIn.readShort();
		int numAuthorities = dnsIn.readShort();
		int numAdditional = dnsIn.readShort();

		while (numQueries-- > 0) { // discard questions
			String queryName = dnsIn.readDomainName();

			int queryType = dnsIn.readShort();
			int queryClass = dnsIn.readShort();
		}

		try {
			//			if (log.isDebugEnabled()) {
			//				log.debug("numAnswers");
			//			}
			while (numAnswers-- > 0) {
				readRR(sorter, dnsIn);
			}
			//			if (log.isDebugEnabled()) {
			//				log.debug("numAuthorities");
			//			}
			while (numAuthorities-- > 0) {
				readRR(sorter, dnsIn);
			}
			//			if (log.isDebugEnabled()) {
			//				log.debug("numAdditional");
			//			}
			while (numAdditional-- > 0) {
				readRR(sorter, dnsIn);
			}
		}
		catch(EOFException ex) {
			// 명시적인 에러가 올라온다면..
			if (log.isDebugEnabled()) {
				log(ex);
			}
			throw new ResultArrayContainEOFException(result.TARGET_DOMAIN);
		}

		return false;
	}

	static void readRR(MXResultSorter sorter, DNSInputStream dnsIn) throws IOException {
		String rrName = dnsIn.readDomainName();
		int rrType = dnsIn.readShort();
		int rrClass = dnsIn.readShort();
		long rrTTL = dnsIn.readInt();
		int rrDataLen = dnsIn.readShort();

		int iMaxDataRange = dnsIn.getPosition() + rrDataLen;

		switch (rrType) {
			case DNS.TYPE_A: {
				// Address
				//				if (log.isDebugEnabled()) {
				//					log.debug("find TYPE_A");
				//				}
				sorter.putAddress(rrName, dnsIn);
				break;
			}

			case DNS.TYPE_NS: {
				//				if (log.isDebugEnabled()) {
				//					log.debug("find TYPE_NS");
				//				}
				String domain = dnsIn.readDomainName();
				sorter.addNameServer(domain);
				break;
			}
			case DNS.TYPE_CNAME: {
				String domain = dnsIn.readDomainName();
				//				if (log.isDebugEnabled()) {
				//					log.debug("canonical from:" + rrName + " to:" + domain);
				//				}
				sorter.addCanonicalName(rrName, domain);
				break;
			}
			case DNS.TYPE_MD:
			case DNS.TYPE_MF:
			case DNS.TYPE_MB:
			case DNS.TYPE_MG:
			case DNS.TYPE_MR:
			case DNS.TYPE_PTR: {
				String domain = dnsIn.readDomainName();
				//				if (log.isDebugEnabled()) {
				//					log.debug(DNS.typeName(rrType) + ":" + domain);
				//				}
				break;
			}

			case DNS.TYPE_SOA: {
				// Auth
				//				if (log.isDebugEnabled()) {
				//					log.debug("find SOA");
				//				}
				sorter.putStartOfAuth(dnsIn);
				break;
			}

			case DNS.TYPE_NULL: {
				byte[] data = new byte[rrDataLen];
				dnsIn.read(data);
				String text = new String(data, "latin1");
				//				if (log.isDebugEnabled()) {
				//					log.debug("etc null:" + text);
				//				}
				break;
			}

			case DNS.TYPE_WKS: {
				int[] ipAddress = new int[4];
				for (int i = 0; i < 4; ++i) {
					ipAddress[i] = dnsIn.readByte();
				}
				int protocol = dnsIn.readByte();
				byte[] data = new byte[rrDataLen - 5];
				dnsIn.read(data);
				break;
			}

			case DNS.TYPE_HINFO: {
				String cpu = dnsIn.readString();
				String os = dnsIn.readString();
				break;
			}

			case DNS.TYPE_MINFO: {
				String rBox = dnsIn.readDomainName();
				String eBox = dnsIn.readDomainName();
				break;
			}

			case DNS.TYPE_MX: {
				// MailExchanger
				sorter.putMailExchanger(dnsIn);
				break;
			}

			case DNS.TYPE_TXT: {
				String s;
				do {
					s = dnsIn.readString();
					//					if (log.isDebugEnabled()) {
					//						log.debug("etc text:" + s);
					//					}
				} while (iMaxDataRange > dnsIn.getPosition());
				break;
			}

			default: {
				log(rrName + " receive unknown:" + rrType);
				dnsIn.plusPosition(rrDataLen);
			}
		}
	}

	/**
	 * DNS Response Flag 분석.
	 * 
	 * @param flags
	 * @throws IOException
	 */
	static void decodeFlags(int flags, MXSearchResult result) throws Exception {
		boolean isResponse = ((flags >> DNS.SHIFT_QUERY) & 1) != 0;
		if( !isResponse ) {
			throw new IOException("Response flag not set");
		}

		int opcode = (flags >> DNS.SHIFT_OPCODE) & 15;
		// could check opcode
		boolean authoritative = ((flags >> DNS.SHIFT_AUTHORITATIVE) & 1) != 0;
		boolean truncated = ((flags >> DNS.SHIFT_TRUNCATED) & 1) != 0;
		boolean recurseRequest = ((flags >> DNS.SHIFT_RECURSE_PLEASE) & 1) != 0;
		// could check recurse request
		boolean recursive = ((flags >> DNS.SHIFT_RECURSE_AVAILABLE) & 1) != 0;
		int code = (flags >> DNS.SHIFT_RESPONSE_CODE) & 15;
		// 응답이 0 이 아니면 뭔가 에러가 발생한 거다.
		if( code != 0 ) {
			switch (code) {
				case DNS.ERROR_FORMAT_ERROR: {
					throw new FormatErrorException(result.TARGET_DOMAIN + ":" + DNS.codeName(code));
				}

				case DNS.ERROR_SERVER_FAILURE: {
					throw new ServerFailureException(result.TARGET_DOMAIN + ":" + DNS.codeName(code));
				}

				case DNS.ERROR_NAME_NOT_KNOWN: {
					throw new NameNotKnownException(result.TARGET_DOMAIN + ":" + DNS.codeName(code));
				}

				case DNS.ERROR_NOT_IMPLEMENTED: {
					throw new NotImplementedException(result.TARGET_DOMAIN + ":" + DNS.codeName(code));
				}

				case DNS.ERROR_REFUSED: {
					throw new ServerRefusedException(result.TARGET_DOMAIN + ":" + DNS.codeName(code));
				}
				default: {
					throw new LookupUnknownException(result.TARGET_DOMAIN + ":" + DNS.codeName(code));
				}

			}
		}
	}

	public String getConnectHost() {
		return null;
	}

	public String getName() {
		return null;
	}

	public boolean isIdle() {
		if( this.IN_COMM && (System.currentTimeMillis() - this.COMM_START_TIME) > this.TIME_OUT ) {
			return true;
		}

		return false;
	}

	public abstract void killSession();

	protected abstract void setConnectionTimeout(int timeOfTimeout);
}
