/*
 * Created on 2005. 12. 4
 */
package pluto.mail.mx;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.Properties;

import lombok.extern.slf4j.Slf4j;
import pluto.io.eMsByteArrayOutputStream;
import pluto.ioutil.FileUtil;
import pluto.lang.Bits;
import pluto.lang.eMsLocale;
import pluto.mail.mx.exception.InvalidCacheFileFormatException;
import pluto.util.Cal;

/**
 * @author 이상근 modify [JOO]
 */
@Slf4j
public class DiskCacheController {
	

	/**
	 * 갱신해야 하는 cache
	 */
	public static final short		STATE_READ_FROM_CACHE_INVALID	= 0;

	/**
	 * 갱신이 필요없는 cache
	 */
	public static final short		STATE_READ_FROM_CACHE			= 1;

	/**
	 * cache가 존재하지 않음 생성해야함.
	 */
	public static final short		STATE_CREATE_CACHE				= 2;

	static String					sBaseDirectory					= null;

	static File						fileBaseDirectory				= null;

	static String[]					tmpDomainStringArray			= null;

	static final void init(Object tmp) throws Exception {
		Properties prop = (Properties) tmp;

		//로그 체널을 세팅한다.
		sBaseDirectory = prop.getProperty("disk.cache.base");

		if( sBaseDirectory == null ) {
			throw new RuntimeException("[disk.cache.base] parameter is null");
		}

		fileBaseDirectory = new File(sBaseDirectory);

		if( !fileBaseDirectory.exists() ) {
			// 디렉토리가 존재하지 않음.
			throw new FileNotFoundException("Directory Dose Not Exist:" + sBaseDirectory);
		}

		if( fileBaseDirectory.isFile() ) {
			// 디렉토리가 존재하지 않음.
			throw new FileNotFoundException("Target Is Not Directory Is File:" + sBaseDirectory);
		}

		// 디렉토리를 만들때 거꾸로 스트링을 배열할때 사용한다.
		// 하다가 정말 10개 보다 큰게 나온다면 그때 늘려 잡아도 된다.
		tmpDomainStringArray = new String[10];
	}

	static final void log(String log) {
		LookupCacheManager.log(log);
	}

	static final void log(Throwable log) {
		LookupCacheManager.log(log);
	}

	/**
	 * 
	 * @param result
	 * @return
	 * @throws Exception
	 */
	static short checkCacheResult(MXSearchResult result) throws Exception {
		// -- 이 모든 작업은 MXSearchResult result 에 대한 lock을 가진 Thread가 진행하기 때문에 locking은 필요가 없다. -- //

		// 먼저 디렉토리를 체크하여 대상 파일 이름을 만든다.
		String sTargetFileName = result.getDiskCacheFileName();
		File sTargetFile = new File(sTargetFileName);

		if( !sTargetFile.exists() ) {
			// 파일이 존재 하지 않으면 cache파일을 생성해야한다.
			if (log.isDebugEnabled()) {
				log(result.TARGET_DOMAIN + ":create cache file target : " + sTargetFileName);
			}
			return STATE_CREATE_CACHE;
		}

		long lSearchTime = -1;
		int resultType = -1;

		byte[] buffer = null;

		int readSize = -1;

		FileInputStream stream = null;

		try {
			buffer = new byte[4];
			stream = new FileInputStream(sTargetFile);

			// -- read search time -- //
			// first 4 byte
			readSize = stream.read(buffer);

			if( readSize != 4 ) {
				throw new InvalidCacheFileFormatException("search time high", 4, readSize);
			}

			lSearchTime = LookupUtil.getLong(buffer, true);

			// next 4 byte
			readSize = stream.read(buffer);

			if( readSize != 4 ) {
				throw new InvalidCacheFileFormatException("search time low", 4, readSize);
			}

			lSearchTime += LookupUtil.getLong(buffer, false);

			// -- read search result code -- //
			// next 4 byte
			readSize = stream.read(buffer);

			if( readSize != 4 ) {
				throw new InvalidCacheFileFormatException("read search result", 4, readSize);
			}

			resultType = Bits.getInt(buffer, 0);

			// -- search result code 에 따라서 다른 루틴으로 읽어 들여야한다. -- //
			// 성공이고 아직 success ttl 이 지나지 않은 경우에는 Cash를 재사용한다.
			//if( resultType == DNS.ERROR_NO_ERROR && (lSearchTime + LookupCacheManager.SEARCH_SUCCESS_DOMAIN_TTL) > System.currentTimeMillis() ) {
			
			//[JOO][modify] 재사용할 valid time을 정교하게 구하도록 수정함
			if( (resultType == DNS.ERROR_NO_ERROR) && result.isValidTimeSucc(lSearchTime) ) { 
				// -- 케쉬파일에서 읽어 들인다. -- //
				// 먼저 읽은 정보를 세팅하고
				result.SEARCH_TIME = lSearchTime;

				// 나머지 정보를 스트림으로 부터 읽는다.
				readFromCacheToResult(buffer, result, stream);
				return STATE_READ_FROM_CACHE;
			}
			
			//[JOO][modify] 실패의 경우 - 성공(DNS.ERROR_NO_ERROR) 도 아니고 초기화(DNS.ERROR_INITIALIZE) 도 아니면서 룩업시간이 초과하지 않을 경우
			// 에러 난 도메인을 error ttl 시간동안 재사용한다.
			if(  resultType != DNS.ERROR_NO_ERROR 
			 &&  resultType != DNS.ERROR_INITIALIZE 
			 && (lSearchTime + LookupCacheManager.SEARCH_ERROR_DOMAIN_TTL) > System.currentTimeMillis() ) {
				// -- 케쉬파일에서 읽어 들인다. -- //
				// 먼저 읽은 정보를 세팅하고
				result.SEARCH_TIME = lSearchTime;
				result.ERROR_TYPE = resultType;
				// 나머지 정보를 스트림으로 부터 읽는다.
				readFromCacheToResult(buffer, result, stream);
				return STATE_READ_FROM_CACHE;
			}
		}
		catch(InvalidCacheFileFormatException e) {
			if( e instanceof InvalidCacheFileFormatException ) {
				log("invalid format file : " + sTargetFileName + "->" + e.toString());
			}else{
				log("checkCacheResult Exception : " + sTargetFileName + "->" + e.toString());
				//e.printStackTrace();
				log.error(e.getMessage());
			}
			return STATE_READ_FROM_CACHE_INVALID;
		}
		finally {
			buffer = null;
			FileUtil.close(stream);
		}
		
		// serial이 다르거나 현재 시간이 검색 제한시간 보다 커지면 재검색을 해야한다.
		return STATE_READ_FROM_CACHE_INVALID;
	}

	
	static void readFromErrorCacheToResult(byte[] buffer, MXSearchResult result, InputStream stream) throws Exception {
		int readSize = -1;

		// 가장 낮은 도메인의 갯수를 읽는다.
		readSize = stream.read(buffer);

		if( readSize != 4 ) {
			throw new InvalidCacheFileFormatException("get min preference count", 4, readSize);
		}

		int messageSize = Bits.getInt(buffer, 0);

		byte[] messageBuffer = new byte[messageSize];

		readSize = stream.read(messageBuffer);

		if( messageSize != readSize ) {
			// 지정 메세지 사이즈와 읽어들인 사이즈가 다르다면?
			if (log.isDebugEnabled()) {
				throw new RuntimeException("ERROR MESSAGE READ ERROR SIZE:" + String.valueOf(messageSize) + "bytes but read:" + String.valueOf(readSize) + "bytes");
			}
			// TODO 이럴경우가 생긴다면 에러처리를 해야한다. 그러나 특별히 필요할지는 모르겠다.
			log(result.TARGET_DOMAIN + ":ERROR MESSAGE READ ERROR SIZE:" + String.valueOf(messageSize) + "bytes but read:" + String.valueOf(readSize) + "bytes");
		}
		
		return;
	}

	static void readFromCacheToResult(byte[] buffer, MXSearchResult result, InputStream stream) throws Exception {
		int readSize = -1;

		// 가장 낮은 도메인의 갯수를 읽는다.
		readSize = stream.read(buffer);

		if( readSize != 4 ) {
			throw new InvalidCacheFileFormatException("get min preference count", 4, readSize);
		}

		int iMinPreferenceDomainCount = Bits.getInt(buffer, 0);

		result.INDEX_OF_FIRST_REFERENCE_SEQUENCE = iMinPreferenceDomainCount;

		readSize = stream.read(buffer);

		if( readSize != 4 ) {
			throw new InvalidCacheFileFormatException("get etc preference count", 4, readSize);
		}

		int iTotalMXList = iMinPreferenceDomainCount + Bits.getInt(buffer, 0);

		// result를 초기화 한다.
		result.init(iTotalMXList);

		eMsByteArrayOutputStream out = null;
		try {
			out = eMsByteArrayOutputStream.getInstance();
			byte[] bufferDomainread = out.getRawByteArray();
			for (int i = 0; i < iTotalMXList; i++) {
				// 도메인 스트링 카운트를 읽는다.
				readSize = stream.read(buffer);

				if( readSize != 4 ) {
					throw new InvalidCacheFileFormatException("get domain size", 4, readSize);
				}

				int iDomainSize = Bits.getInt(buffer, 0);

				// 도메인 사이즈가 1024를 넘으면 이상하다.
				// 버퍼가 1024 기반으로 되어 있기도하지만...쩝.
				if( iDomainSize > 1024 ) {
					throw new InvalidCacheFileFormatException("1024 byte Domain is Exist=>" + iDomainSize);
				}

				readSize = stream.read(bufferDomainread, 0, iDomainSize);

				// 읽을 수와 읽은 수가 다르면 에러.
				if( readSize != iDomainSize ) {
					throw new InvalidCacheFileFormatException("different read size and expect size", iDomainSize, readSize);
				}

				String domain = new String(bufferDomainread, 0, readSize);

				// 도메인 아이피를 읽어온다.
				readSize = stream.read(buffer);

				if( readSize != 4 ) {
					throw new InvalidCacheFileFormatException("get domain ip", 4, readSize);
				}

				int iIP = Bits.getInt(buffer, 0);

				// 해당 인덱스에 세팅한다.
				result.set(i, domain, iIP);
			}
		}
		finally {
			eMsByteArrayOutputStream.recycleInstance(out);
			out = null;
		}
	} // -- bit 연산들 -- //

	/**
	 * 결과를 파일로 저장.
	 * 
	 * @param result
	 * @throws Exception
	 */
	static void storeResultToCache(MXSearchResult result) throws Exception {
		FileOutputStream stream = null;
		String sCacheFileName = null;
		byte[] buffer = null;
		try {
			sCacheFileName = result.getDiskCacheFileName();
			if (log.isDebugEnabled()) {
				log(result+":save to cache:"+sCacheFileName);
			}
			buffer = new byte[4];
			File sTargetFile = new File(sCacheFileName);
			stream = new FileOutputStream(sTargetFile, false);
			writeFromResultToCache(buffer, result, stream);
		}
		catch(Exception e){
			//e.printStackTrace();
			log.error(e.getMessage());
			throw e;
		}
		finally {
			buffer = null;
			FileUtil.close(stream);
			stream = null;
		}

		// history 작성 //
		//		try {
		//			buffer = new byte[4];
		//			File sTargetFile = new File(sCacheFileName + "." + Cal.getYYYYMM());
		//			stream = new FileOutputStream(sTargetFile, false);
		//			writeFromResultToNormal(buffer, result, stream);
		//		}
		//		finally {
		//			buffer = null;
		//			FileUtil.close(stream);
		//			stream = null;
		//		}
	}

	static void writeFromResultToCache(byte[] buffer, MXSearchResult result, OutputStream stream) throws Exception {
		int readSize = -1;

		// write search time
		LookupUtil.putLong(result.SEARCH_TIME, buffer, true);
		stream.write(buffer, 0, 4);
		LookupUtil.putLong(result.SEARCH_TIME, buffer, false);
		stream.write(buffer, 0, 4);

		// write search result
		Bits.putInt(buffer, 0, result.ERROR_TYPE);
		stream.write(buffer, 0, 4);

		// -- 에러와 정상에 따라서 구분해야함 -- //
		// 에러는 에러 메세지만 작성하고 돌아간다.
		if( result.isError() ) {
			String errResult = result.getError().getMessage();
			byte[] errByteResult = errResult.getBytes(eMsLocale.CHAR_SET);

			// write message length
			Bits.putInt(buffer, 0, errByteResult.length);
			stream.write(buffer, 0, 4);

			// write message byte array
			stream.write(errByteResult);

			return;
		}

		// 가장 낮은 도메인의 갯수를 쓴다.
		Bits.putInt(buffer, 0, result.INDEX_OF_FIRST_REFERENCE_SEQUENCE);

		stream.write(buffer, 0, 4);

		// 나머지 도메인의 갯수를 쓴다. ( 총 사이즈 - min domain size )
		Bits.putInt(buffer, 0, (result.size - result.INDEX_OF_FIRST_REFERENCE_SEQUENCE));

		stream.write(buffer, 0, 4);

		for (int i = 0; i < result.size; i++) {
			// 도메인을 준비하고
			String domain = result.getDomain(i);
			byte[] byteArrayDomain = domain.getBytes(eMsLocale.CHAR_SET);

			// 도메인 사이즈를 기록한다.
			Bits.putInt(buffer, 0, byteArrayDomain.length);
			stream.write(buffer, 0, 4);

			// 도메인 내용을 기록한다.
			stream.write(byteArrayDomain, 0, byteArrayDomain.length);

			// 도메인 IP를 기록한다.
			Bits.putInt(buffer, 0, result.getIP(i));
			stream.write(buffer, 0, 4);
		}
	} // -- bit 연산들 -- //

	static void writeFromResultToNormal(byte[] buffer, MXSearchResult result, OutputStream stream) throws Exception {
		int readSize = -1;

		PrintStream out = new PrintStream(stream, true, "KSC5601");

		out.println("=================== START ===========================");

		// write search time
		out.println("search time:" + Cal.getDate(result.SEARCH_TIME));

		int searchResult = result.ERROR_TYPE;

		// write search time
		out.println("search result:" + DNS.codeName(searchResult));

		// -- 에러와 정상에 따라서 구분해야함 -- //
		// 에러는 에러 메세지만 작성하고 돌아간다.
		if( result.isError() ) {
			String errResult = result.getError().getMessage();
			out.println("error message:" + DNS.codeName(searchResult));
			return;
		}

		// 가장 낮은 도메인의 갯수를 쓴다.
		out.println("min reference size:" + String.valueOf(result.INDEX_OF_FIRST_REFERENCE_SEQUENCE));

		// 나머지 도메인의 갯수를 쓴다. ( 총 사이즈 - min domain size )
		out.println("etc reference size:" + String.valueOf(result.size - result.INDEX_OF_FIRST_REFERENCE_SEQUENCE));

		out.println("=== domain list===");
		for (int i = 0; i < result.size; i++) {
			// 도메인을 준비하고
			String domain = result.getDomain(i);
			int ip = result.getIP(i);
			byte[] byteIP = LookupUtil.getIntToInetByteArray(ip);
			out.print(String.valueOf(i));
			out.print(":");
			out.print(domain);
			out.print("/");
			out.println(((byteIP[0] & 0xFF) + "." + (byteIP[1] & 0xFF) + "." + (byteIP[2] & 0xFF) + "." + (byteIP[3] & 0xFF)));
		}
		out.println("=================== END ===========================");
	} // -- bit 연산들 -- //
}
