/*
 * Created on 2005. 12. 6
 */
package pluto.mail.mx;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.IOException;

import lombok.extern.slf4j.Slf4j;

/**
 * @author 이상근
 */
@Slf4j
public class DNSInputStream extends ByteArrayInputStream {
	

	protected DataInputStream		dataIn;

	public DNSInputStream(byte[] data, int off, int len) {
		super(data, off, len);
		dataIn = new DataInputStream(this);
	}

	public int getPosition() {
		return super.pos;
	}

	public void plusPosition(int add) {
		super.pos += add;
	}

	public int readByte() throws IOException {
		return dataIn.readUnsignedByte();
	}

	public int readShort() throws IOException {
		return dataIn.readUnsignedShort();
	}

	public long readInt() throws IOException {
		return dataIn.readInt() & 0xffffffffL;
	}

	public String readString() throws IOException {
		int len = readByte();
		if( len == 0 ) {
			return "";
		}
		else {
			byte[] buffer = new byte[len];
			dataIn.readFully(buffer);
			return new String(buffer, "latin1");
		}
	}

	public String readDomainName() throws IOException {
		if( pos >= count )
			throw new EOFException("EOF reading domain name");
		if( (buf[pos] & 0xc0) == 0 ) {
			String label = readString();
			if( label.length() > 0 ) {
				String tail = readDomainName();
				if( tail.length() > 0 )
					label = label + '.' + tail;
			}
			return label;
		}
		else {
			if( (buf[pos] & 0xc0) != 0xc0 )
				throw new IOException("Invalid domain name compression offset");
			int offset = readShort() & 0x3fff;
			DNSInputStream dnsIn = new DNSInputStream(buf, offset, buf.length - offset);
			return dnsIn.readDomainName();
		}
	}

	public void readRR(MXResultSorter sorter) throws IOException {
		String rrName = readDomainName();
		int rrType = readShort();
		int rrClass = readShort();
		long rrTTL = readInt();
		int rrDataLen = readShort();
		DNSInputStream rrDNSIn = new DNSInputStream(buf, pos, rrDataLen);
		pos += rrDataLen;
		switch (rrType) {
			case DNS.TYPE_A: {
				// Address
				if (log.isDebugEnabled()) {
					log.debug("find TYPE_A");
				}
				sorter.putAddress(rrName, rrDNSIn);
				break;
			}

			case DNS.TYPE_NS: {
				if (log.isDebugEnabled()) {
					log.debug("find TYPE_NS");
				}
				if (log.isDebugEnabled()) {
					log.debug("read TYPE_NS domain START");
				}
				String domain = rrDNSIn.readDomainName();
				if (log.isDebugEnabled()) {
					log.debug("read TYPE_NS domain END");
				}
				sorter.addNameServer(domain);
				break;
			}
			case DNS.TYPE_CNAME: {
				String domain = rrDNSIn.readDomainName();
				if (log.isDebugEnabled()) {
					log.debug("canonical from:" + rrName + " to:" + domain);
				}
				sorter.addCanonicalName(rrName, domain);
				break;
			}
			case DNS.TYPE_MD:
			case DNS.TYPE_MF:
			case DNS.TYPE_MB:
			case DNS.TYPE_MG:
			case DNS.TYPE_MR:
			case DNS.TYPE_PTR: {
				if (log.isDebugEnabled()) {
					log.debug("read ETC domain START");
				}
				String domain = rrDNSIn.readDomainName();
				if (log.isDebugEnabled()) {
					log.debug("read ETC domain START");
				}
				if (log.isDebugEnabled()) {
					log.debug(DNS.typeName(rrType) + ":" + domain);
				}
				break;
			}

			case DNS.TYPE_SOA: {
				// Auth
				if (log.isDebugEnabled()) {
					log.debug("find SOA");
				}
				sorter.putStartOfAuth(rrDNSIn);
				break;
			}

			case DNS.TYPE_NULL: {
				byte[] data = new byte[rrDataLen];
				rrDNSIn.read(data);
				String text = new String(data, "latin1");
				if (log.isDebugEnabled()) {
					log.debug("etc null:" + text);
				}
				break;
			}

			case DNS.TYPE_WKS: {
				int[] ipAddress = new int[4];
				for (int i = 0; i < 4; ++i) {
					ipAddress[i] = rrDNSIn.readByte();
				}
				int protocol = rrDNSIn.readByte();
				byte[] data = new byte[rrDataLen - 5];
				rrDNSIn.read(data);
				break;
			}

			case DNS.TYPE_HINFO: {
				String cpu = rrDNSIn.readString();
				String os = rrDNSIn.readString();
				break;
			}

			case DNS.TYPE_MINFO: {
				if (log.isDebugEnabled()) {
					log.debug("read TYPE_MINFO1 domain START");
				}
				String rBox = rrDNSIn.readDomainName();
				if (log.isDebugEnabled()) {
					log.debug("read TYPE_MINFO1 domain END");
				}
				if (log.isDebugEnabled()) {
					log.debug("read TYPE_MINFO2 domain START");
				}
				String eBox = rrDNSIn.readDomainName();
				if (log.isDebugEnabled()) {
					log.debug("read TYPE_MINFO2 domain END");
				}
				break;
			}

			case DNS.TYPE_MX: {
				// MailExchanger
				sorter.putMailExchanger(rrDNSIn);
				break;
			}

			case DNS.TYPE_TXT: {
				String s;
				while ((s = rrDNSIn.readString()) != null) {
					if (log.isDebugEnabled()) {
						log.debug("TEXT:" + s);
					}
				}
				break;
			}

			default: {
				LookupCacheManager.log(rrName + " receive unknown:" + rrType);
			}
		}
		return;
	}
}
