/*
 * LookupContainer.java
 *
 * Created on 2004년 7월 8일 (목), 오후 4:10
 */

package pluto.mail;

import java.net.InetAddress;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Properties;

import lombok.extern.slf4j.Slf4j;
import pluto.log.LogChannel;
import pluto.log.LogChannelContainer;
import pluto.net.NetAddress;
import pluto.schedule.ScheduledMonitor;
import pluto.util.AmailHashtable;
import pluto.util.FIFOBuffer;
import pluto.util.KeyValueEntry;
import pluto.util.eMsStringTokenizer;

/**
 * 
 * @author 이상근
 */
@Slf4j
public class LookupContainer extends ScheduledMonitor {

	/**
	 * 정상 도메인들을 보관한다.
	 */
	private static AmailHashtable	HASH_OF_DNS_LIST			= null;

	/**
	 * DNS Server List
	 */
	public static int				DNS_RESOLVER_LIST_LENGTH	= 0;

	/**
	 * MX Record 갱신주기
	 */
	public static long				MX_REFRESH_CYCLE			= 3 * 12 * 60 * 60 * 1000L;

	/**
	 * A Record 갱신주기
	 */
	public static long				A_REFRESH_CYCLE				= 12 * 60 * 60 * 1000L;

	/**
	 * DNS Server InetAddress List
	 */
	protected static InetAddress[]		DNS_RESOLVER_LIST			= null;

	/**
	 * 룩업에러트레이스를 남길 로그체널
	 */
	private static LogChannel		LOG_CHANNEL					= null;

	/**
	 * DNSGroupResolver 버퍼를 담아두는거..
	 */
	private static FIFOBuffer		RESOLVER_CONTAINER			= null;

	private static LookupContainer	INNER_CONTAINER_INSTANCE	= null;

	/**
	 * 본서버가 Relay Server를 통해서 발송하는 설정 flag
	 */
	private static boolean			RELAY_SERVER_PRESENT		= false;

	/**
	 * 릴레이서버 아이피를 지정한다.
	 */
	private static String			RELAY_SERVER_IP				= null;

	/**
	 * 릴레이서버의 DNSList Instance를 가지고 있는 녀석이다.
	 */
	private static DNSList			RELAY_DNS_LIST				= null;

	/**
	 * 각각의 헤쉬테이블을 초기화한다. TODO 포함하는 도메인의 수가 많아지기 때문에 Hashtable을 좀 수정하여 효율적으로 관리할수 있도록 해야할거 같다.
	 */
	static {
		// TODO 적정한 사이즈를 산정해야 효율이 올라간다.
		// 그러기 위해서는 기존에 찾았던 도메인 리스트나 초기 세팅시 도메인 헤쉬멥을 가지고 있는것도
		// 하나의 방법일수 있다.
		HASH_OF_DNS_LIST = new AmailHashtable(1000);
	}

	/**
	 * LookupContainer를 초기화한다.
	 * 
	 * @param prop
	 *        초기화 파라미터
	 * @throws Exception
	 */
	public synchronized static void init(Object prop) throws Exception {
		Properties tmp = (Properties) prop;

		//로그 체널을 세팅한다.
		String sChannelID = tmp.getProperty("logger");

		if( sChannelID != null ) {
			LOG_CHANNEL = LogChannelContainer.get(sChannelID);
		}

		RELAY_SERVER_IP = tmp.getProperty("relay.server");

		RELAY_SERVER_PRESENT = RELAY_SERVER_IP != null;

		if( RELAY_SERVER_PRESENT ) {
			RELAY_DNS_LIST = new RelayDNSList(RELAY_SERVER_IP);
		}
		String tmpSize = tmp.getProperty("max.connection.contain", "50");

		try {
			RESOLVER_CONTAINER = new FIFOBuffer(Integer.parseInt(tmpSize));
		}
		catch(Exception e) {
			//e.printStackTrace();
			log.error(e.getMessage());
			RESOLVER_CONTAINER = new FIFOBuffer(50);
		}

		String tmpRefreshCycle = tmp.getProperty("refresh.cycle");

		if( tmpRefreshCycle == null )
			throw new RuntimeException("refresh.cycle parameter is not set");

		// 리프레쉬 싸이클 세팅
		try {
			A_REFRESH_CYCLE = Long.parseLong(tmpRefreshCycle);
			MX_REFRESH_CYCLE = A_REFRESH_CYCLE * 3;
		}
		catch(Exception e) {
			log.error("LookupContainer", "exec error", e);
		}

		String tmplist = tmp.getProperty("server.list");

		if( tmplist == null )
			throw new RuntimeException("server.list parameter is not set");

		// 서버리스트 세팅
		eMsStringTokenizer TOKEN = new eMsStringTokenizer(tmplist, ";");
		LinkedList list = new LinkedList();
		while (TOKEN.hasMoreTokens()) {
			list.add(TOKEN.nextToken());
		}

		DNS_RESOLVER_LIST_LENGTH = list.size();

		DNS_RESOLVER_LIST = new InetAddress[DNS_RESOLVER_LIST_LENGTH];

		for (int i = 0; i < DNS_RESOLVER_LIST_LENGTH; i++) {
			DNS_RESOLVER_LIST[i] = NetAddress.getInetAddress(list.get(i).toString());
		}

		try {
			int tmpInitSize = Integer.parseInt(tmp.getProperty("init.connection.contain", "10"));

			DNSGroupResolver[] arraySimpleResolver = new DNSGroupResolver[tmpInitSize];

			for (int i = 0; i < tmpInitSize; i++) {
				arraySimpleResolver[i] = getInstance();
			}

			for (int i = 0; i < tmpInitSize; i++) {
				recycleInstance(arraySimpleResolver[i]);
			}
		}
		catch(Exception e) {
			//e.printStackTrace();
			log.error(e.getMessage());
			System.exit(1);
		}
		String tmpMonitorCycle = tmp.getProperty("dns.monitor.cycle");

		if( tmpMonitorCycle == null )
			throw new RuntimeException("dns.monitor.cycle parameter is not set");

		// 리프레쉬 싸이클 세팅
		try {
			INNER_CONTAINER_INSTANCE = new LookupContainer(Long.parseLong(tmpMonitorCycle));
			INNER_CONTAINER_INSTANCE.start();
		}
		catch(Exception e) {
			log.error("LookupContainer", "exec error", e);
		}
	}

	/**
	 * Resolver Instance를 반환한다.
	 * 
	 * @return
	 */
	protected synchronized static DNSGroupResolver getInstance() {
		Object returnValue = RESOLVER_CONTAINER.pop();

		if( returnValue != null ) {
			return (DNSGroupResolver) returnValue;
		}

		try {
			DNSGroupResolver res = new DNSGroupResolver(DNS_RESOLVER_LIST, 53);
			return res;
		}
		catch(Throwable e) {
			log.error("LookupContainer", "resolver create error", e);
			log("DNSGroupResolver creation error");
			log(e);
			return null;
		}
	}

	/**
	 * Resolver Instance를 재활용한다.
	 * 
	 * @param tmp
	 *        DNSGroupResolver instance
	 */
	protected static void recycleInstance(DNSGroupResolver tmp) {
		if( tmp == null )
			return;

		RESOLVER_CONTAINER.push(tmp);
	}

	/**
	 * 로그를 작성한다.
	 * 
	 * @param log
	 */
	public synchronized static void log(String logStr) {
		if( LOG_CHANNEL == null ) {
			log.info("LookupContainer", "log:" + logStr);
		}
		else {
			try {
				LOG_CHANNEL.write(logStr);
			}
			catch(Exception e) {
				log.error("LookupContainer", "log write error", e);
			}
		}
	}

	/**
	 * 로그를 작성한다.
	 * 
	 * @param log
	 */
	public synchronized static void log(Throwable thr) {
		if( LOG_CHANNEL == null ) {
			log.info("log channel is null", thr);
		}
		else {
			try {
				LOG_CHANNEL.write(thr);
			}
			catch(Exception e) {
				log.error("log write error", e);
			}
		}
	}

	/**
	 * 현재 도메인이 등록되어 있는지를 반환
	 * 
	 * @param domain
	 *        검색 도메인
	 * @return true : 이미 찾아 놓은 도메인 <br>
	 *         false : 찾지 않은 도메인
	 */
	public static Throwable isInvalidDomain(String domain) {
		if( RELAY_SERVER_PRESENT ) {
			return null;
		}

		DNSList tmpList = (DNSList) HASH_OF_DNS_LIST.get(domain);

		if( tmpList != null && tmpList.isError() ) {
			return tmpList.getErrorMsg();
		}

		return null;
	}

	private static final Object	LOCK_OF_LIST_CREATE	= new Object();

	/**
	 * 등록된 DNSList를 반환한다.
	 * 
	 * @param domain
	 *        도메인
	 * @return 등록된 DNSList
	 */
	public static DNSList getDNSList(String domain) {
		if( RELAY_SERVER_PRESENT ) {
			return RELAY_DNS_LIST;
		}
		DNSList tmpList = (DNSList) HASH_OF_DNS_LIST.get(domain);

		if( tmpList != null ) {
			return tmpList;
		}
		// 없으면 신규 Instance를 생성한다.
		if (log.isDebugEnabled()) {
			log.debug("LookupContainer", "MissDomain-getDNSList", domain);
		}
		synchronized (LOCK_OF_LIST_CREATE) {
			tmpList = new DNSList(domain);
			HASH_OF_DNS_LIST.put(domain, tmpList);

			return tmpList;
		}
	}

	/**
	 * 해당 도메인의 DNSList를 등록한다.
	 * 
	 * @param domain
	 *        도메인
	 * @param list
	 *        해당 도메인의 DNSList
	 */
	public synchronized static void registeDNSList(String domain, DNSList list) {
		if( RELAY_SERVER_PRESENT ) {
			return;
		}
		HASH_OF_DNS_LIST.put(domain, list);
	}

	/** Creates a new instance of LookupContainer */
	LookupContainer(long inter) {
		super(inter, "DNS Inner Monitor");
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see pluto.schedule.ScheduledMonitor#check()
	 */
	protected void check() throws Exception {
		if (log.isDebugEnabled()) {
			log.debug("INNER DNS LIST CHECK START");
		}
		StringBuffer tmpBuffer = new StringBuffer(128);
		try {
			for (Iterator iter = HASH_OF_DNS_LIST.iterator(); iter.hasNext();) {
				KeyValueEntry a = (KeyValueEntry) iter.next();

				String key = a.getKey().toString();
				DNSList value = (DNSList) a.getValue();
				if (log.isDebugEnabled()) {
					log.debug("CHECK", key);
				}
				value.refreshMXRecored(tmpBuffer);
			}
		}
		finally {
			tmpBuffer = null;
		}
	}
}
