/*
 * DNSList.java
 *
 * Created on 2003년 3월 5일 수, 오후 10:22
 */

package pluto.mail;

import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import lombok.extern.slf4j.Slf4j;
import pluto.DNS.DClass;
import pluto.DNS.Lookup;
import pluto.DNS.Name;
import pluto.DNS.TextParseException;
import pluto.lang.eMsLocale;
import pluto.util.Cal;
import pluto.util.PlutoLinkedList;

/**
 * MX Record 리스트 정보를 저장하는 데이터 구조 <br>
 * Preference순으로 rotate하여 리스트를 관리한다. <br>
 * TODO 도메인 정보를 저장하는 리스트의 흐름을 한번 정리해봐야한다.
 * 
 * @author 이상근
 * @version
 */
@Slf4j
public class DNSList extends pluto.lang.Name {
	

	/**
	 * 대상 도메인을 설정한다.
	 */
	private String				TARGET_DOMAIN						= null;

	/**
	 * MX Record Referencd 사이즈를 맞추기 위한 사이즈
	 */
	public static final int		MX_REFERENCE_MAX_SIZE				= 4;

	/**
	 * 복구불가 에러도메인인지를 점검한다.
	 */
	private boolean				PERMANENT_ERROR_FLAG				= false;

	/**
	 * 정상도메인인지 에러도메인인지를 지정한다.
	 */
	private boolean				error								= false;

	/**
	 * 에러도메인일 경우 마지막 에러사항이 기록된다.
	 */
	private Throwable			error_msg							= null;

	/**
	 * 프라이머리 MX레코드 사이즈를 저장한다.
	 */
	private int					INDEX_OF_FIRST_REFERENCE_SEQUENCE	= 0;

	/**
	 * MX Record리스트를 저장한다.
	 */
	private DNSSearchResult		SERVER_LIST							= null;

	/**
	 * MX Recored 마지막 갱신시간을 저장한다.
	 */
	private long				MX_REFRESH_TIME						= 0;

	/**
	 * A Recored 마지막 갱신시간을 저장한다.
	 */
	private long				A_REFRESH_TIME						= 0;

	private LinkedList			MX_LIST								= null;

	private Lookup				SEARCH_LOOKUP						= null;

	protected DNSList() {

	}

	/**
	 * Creates new DNSList
	 */
	public DNSList(String domain) {
		super.setName("DNSList-" + domain);
		log(super.getName() + " create");
		if (log.isDebugEnabled()) {
			log.debug("DNSList", "Start create Instance");
		}
		this.MX_REFRESH_TIME = -1;
		this.A_REFRESH_TIME = -1;
		this.MX_LIST = new LinkedList();
		this.TARGET_DOMAIN = domain;
		try {
			this.SEARCH_LOOKUP = new pluto.DNS.Lookup(pluto.DNS.Name.fromString(domain), pluto.DNS.Type.MX, DClass.IN);
			PERMANENT_ERROR_FLAG = false;
		}
		catch(TextParseException te) {
			//이녀석이 올라오면 도메인이 잘못된거다.
			PERMANENT_ERROR_FLAG = true;
			this.error_msg = new DomainNotFoundException(te);
			return;
		}

		if (log.isDebugEnabled()) {
			log.debug("DNSList", "pass Create Lookup Instance");
		}

		StringBuffer tmpBuffer = null;
		try {
			tmpBuffer = new StringBuffer(128);
			this.refreshMXRecored(tmpBuffer);
		}
		finally {
			tmpBuffer = null;
		}

		if (log.isDebugEnabled()) {
			log.debug("DNSList", "Start create Instance ... [OK]");
		}
	}

	/**
	 * 등록된 도메인 리스트를 MX preference 순번에 의하여 로테이션 시키면서 스트링 배열로 반환
	 * 
	 * @return MX Record 스트링 배열
	 */
	public synchronized String getList(int AGENT_INDEX, int SERVER_INDEX) {
		if( this.SERVER_LIST.size() < SERVER_INDEX )
			return null;
		int TARGET_INDEX = 0;
		if( SERVER_INDEX > this.INDEX_OF_FIRST_REFERENCE_SEQUENCE ) {
			if (log.isDebugEnabled()) {
				log.debug("DNSList", "TARGET direct IDX:" + String.valueOf(SERVER_INDEX - 1));
			}
			TARGET_INDEX = SERVER_INDEX - 1;
		}
		else {
			if (log.isDebugEnabled()) {
				log.debug("DNSList", "TARGET rotate IDX:" + String.valueOf((AGENT_INDEX + SERVER_INDEX) % this.INDEX_OF_FIRST_REFERENCE_SEQUENCE));
			}
			TARGET_INDEX = (AGENT_INDEX + SERVER_INDEX) % this.INDEX_OF_FIRST_REFERENCE_SEQUENCE;
		}

		TARGET_INDEX = TARGET_INDEX < this.SERVER_LIST.size() ? TARGET_INDEX : this.SERVER_LIST.size() - 1;
		TARGET_INDEX = TARGET_INDEX < 0 ? 0 : TARGET_INDEX;

		return this.SERVER_LIST.get(TARGET_INDEX);
	}

	void refreshMXRecored(StringBuffer TMP_BUFFER) {
		if (log.isDebugEnabled()) {
			log.debug("DNSList", "MX REFESH TIME:" + Cal.getDate(this.MX_REFRESH_TIME));
		}
		// 마지막 MX 레코드 갱신시간에 갱신주기를 더한것이 현재시간보다 크면 갱신을 하지 않아도 된다.
		// 그래서 그냥 돌아간다.
		if( this.MX_REFRESH_TIME + LookupContainer.MX_REFRESH_CYCLE > System.currentTimeMillis() ) {
			if (log.isDebugEnabled()) {
				log.debug("DNSList", "REFESH SKIP");
			}
			return;
		}

		//MX 레코드를 갱신한다.
		// Exception last_exception = null;
		DNSGroupResolver __RESOLVE_SERVERS = null;
		List answers = null;
		int answer_result = -1;
		String answer_msg = null;
		try {
			__RESOLVE_SERVERS = LookupContainer.getInstance();

			if( __RESOLVE_SERVERS == null ) {
				//만일 resolver가 null이 반환되었다면 네트웍에 문제가 발생한 것이다.
				log.error("DNSList", "getting DNS Resolver error");
				this.setMXList(new ResolverCreationException("Resover Creation error"));
				return;
			}

			TMP_BUFFER.setLength(0);
			answer_result = __RESOLVE_SERVERS.process(SEARCH_LOOKUP, TMP_BUFFER);
			if( answer_result == Lookup.SUCCESSFUL ) {
				answers = SEARCH_LOOKUP.getResultMxList();
			}
			else {
				answer_msg = SEARCH_LOOKUP.getErrorString();
			}

		}
		catch(Throwable thw) {
			//로깅하고
			log(thw);
			// 한번이라도 성공을한 녀석이라면 이전에 찾았던걸로 계속한다.
			if( this.MX_REFRESH_TIME > 0 ) {
				this.MX_REFRESH_TIME = System.currentTimeMillis();
			}
			else {
				// 애초에 에러므로 에러로 마킹한다.
				this.setMXList(new DNSSearchFailException(thw));
			}
			//더이상 진행이 의미가 없다.
			return;
		}
		finally {
			LookupContainer.recycleInstance(__RESOLVE_SERVERS);
		}

		switch (answer_result) {
			case Lookup.SUCCESSFUL: {
				// 일단 DNSList는 재활용 하지 않는 걸로 바꾼다.
				this.setMXList(answers);
				break;
			}

			case Lookup.HOST_NOT_FOUND: {
				if (log.isDebugEnabled()) {
					log.info("LokupContainer", "HOST_NOT_FOUND:" + answer_msg);
				}
				this.setMXList(new DomainNotFoundException(this.TARGET_DOMAIN + ":" + answer_msg));
				break;
			}

			case Lookup.TYPE_NOT_FOUND: {
				if (log.isDebugEnabled()) 
					log.info("LookupContainer", "TYPE_NOT_FOUND:" + answer_msg);
				
				if( eMsLocale.NO_MX_RECORED_REJECT_FLAG ) {
					// MX Record를 가지지 않으면 유효 도메인으로 보지 않는다.
					this.setMXList(new DomainNotFoundException(this.TARGET_DOMAIN + ":HAS NO MX RECORD"));
				}
				else {
					// MX Record를 가지고 있지 않더라도 기본 도메인으로 처리한다.
					answers = new PlutoLinkedList();
					answers.add("10 " + this.TARGET_DOMAIN);
					this.setMXList(answers);
				}
				break;
			}

			// 기타에러에대해서 처리한다.
			default: {
				// 한번이라도 성공을한 녀석이라면 이전에 찾았던걸로 계속한다.
				if( this.MX_REFRESH_TIME > 0 ) {
					this.MX_REFRESH_TIME = System.currentTimeMillis();
				}
				else {
					// 애초에 에러므로 에러로 마킹한다.
					this.setMXList(new DNSSearchFailException(this.TARGET_DOMAIN + ":" + answer_msg));
				}
				//더이상 진행이 의미가 없다.
				return;
			}
		}// end of result switch

		//초기화 시간을 기록한다.
		this.MX_REFRESH_TIME = System.currentTimeMillis();
		//MX Recored를 갱신하였으므로 A Recored 갱신시간을 초기화한다.
		//그래야 refreshARecord() 요녀석이 움직인다.
		this.A_REFRESH_TIME = -1;

		// A Record를 갱신한다.
		refreshARecord(TMP_BUFFER);
	}

	void refreshARecord(StringBuffer TMP_BUFFER) {
		if (log.isDebugEnabled()) {
			log.debug("DNSList", "A REFESH TIME:" + Cal.getDate(this.A_REFRESH_TIME));
		}
		// 마지막 A 레코드 갱신시간에 갱신주기를 더한것이 현재시간보다 크면 갱신을 하지 않아도 된다.
		// 그래서 그냥 돌아간다.
		if( this.A_REFRESH_TIME + LookupContainer.A_REFRESH_CYCLE > System.currentTimeMillis() ) {
			if (log.isDebugEnabled()) {
				log.debug("DNSList", "A RECORD REFESH SKIP");
			}
			return;
		}

		// A Recored를 갱신하도록 한다.
		this.setAList(TMP_BUFFER);

		this.A_REFRESH_TIME = System.currentTimeMillis();
	}

	void setMXList(Object __SERVER_LIST) {
		PlutoLinkedList listServers = null;
		String sServer = null;
		if( __SERVER_LIST instanceof Throwable ) {
			this.error = true;
			this.error_msg = (Throwable) __SERVER_LIST;
			return;
		}

		listServers = (PlutoLinkedList) __SERVER_LIST;

		//일단 에러는 아니다.
		this.error = false;

		// MX List를 지운다.
		this.MX_LIST.clear();

		// MX Reference index size를 맞춘다.
		for (Iterator iter = listServers.iterator(); iter.hasNext();) {
			sServer = iter.next().toString();

			int idx = sServer.indexOf(" ");

			if( idx < 0 ) {
				// 구분자가 없으면 잘못된거다.
				log.info("DNSList", "Search Result Domain Invalid:" + sServer);
				continue;
			}

			for (int plus = MX_REFERENCE_MAX_SIZE - idx; plus > 0; plus--) {
				sServer = "0".concat(sServer);
			}

			if( sServer.endsWith(".") ) {
				this.MX_LIST.add(sServer.substring(0, sServer.length() - 1));
			}
			else {
				this.MX_LIST.add(sServer);
			}
		}

		if( this.MX_LIST.size() < 1 ) {
			this.error = true;
			this.error_msg = new DNSSerarchResultZeroException("VALID DOMAIN IS NOT PRESENT");
			return;
		}

		/**
		 * 정렬을 한다.
		 */
		Collections.sort(this.MX_LIST);
	}

	/**
	 * MX Record리스트만 세팅한다.
	 * 
	 * @param __SERVER_LIST__
	 *            MX Record 리스트 저장 Vector
	 * @param state
	 *            도메인의 상태
	 */
	void setAList(StringBuffer TMP_BUFFER) {
		// 만일 MX size 가 0 이라면 에러다...
		if( this.MX_LIST.size() < 1 ) {
			this.error = true;
			return;
		}
		/**
		 * 첫번째 녀석이 아이디가 된다.
		 */
		String sServer = this.MX_LIST.get(0).toString();
		String ID = sServer.substring(0, sServer.indexOf(" "));

		if (log.isDebugEnabled()) {
			log.debug("DNSList", "MX LIST:" + this.MX_LIST.toString());
		}
		/**
		 * 내부적으로 사용할 서버 리스트
		 */
		this.SERVER_LIST = new DNSSearchResult(this.MX_LIST.size() + 1);

		DNSGroupResolver __RESOLVE_SERVERS = LookupContainer.getInstance();
		/**
		 * 정렬을 했으니까 서버 이름들을 배열에 넣으면서 첫번째 구간을 결정한다.
		 */
		for (Iterator iter = this.MX_LIST.iterator(); iter.hasNext();) {
			sServer = (String) iter.next();

			String __TMP_ID__ = sServer.substring(0, sServer.indexOf(" "));

			String sServerDomain = sServer.substring(sServer.indexOf(" ") + 1).trim();

			//도메인에 대해서 A Recored를 가져와야 한다.
			List answers = this.getArecoredList(__RESOLVE_SERVERS, sServerDomain, TMP_BUFFER);
			if( answers == null ) {
				if (log.isDebugEnabled()) {
					log.debug("DNSList", "lookup error:" + TMP_BUFFER.toString());
				}
				//널이 반환되면 뭔가가 에러가 있는거다.
				LookupContainer.log(TMP_BUFFER.toString());
				continue;
			}
			for (Iterator iter2 = answers.iterator(); iter2.hasNext();) {
				String sIP = iter2.next().toString();
				this.SERVER_LIST.add(sIP + "=" + sServerDomain);
				if( __TMP_ID__.equals(ID) ) {
					this.INDEX_OF_FIRST_REFERENCE_SEQUENCE++;
				}
			}
		}

		this.SERVER_LIST.trimToSize();

		return;
	}

	/**
	 * 해당 도메인에 A Record 리스트를 추출한다.
	 * 
	 * @param __RESOLVE_SERVERS
	 * @param sServerDomain
	 * @param TMP_BUFFER
	 * @return
	 */
	List getArecoredList(DNSGroupResolver __RESOLVE_SERVERS, String sServerDomain, StringBuffer TMP_BUFFER) {
		TMP_BUFFER.setLength(0);
		// String answer_msg = null;
		Lookup SEARCH_LOOKUP = null;
		try {
			SEARCH_LOOKUP = new pluto.DNS.Lookup(Name.fromString(sServerDomain), pluto.DNS.Type.A, DClass.IN);
		}
		catch(Throwable e) {
			// 에러상황을 기록한다.
			TMP_BUFFER.append("[");
			TMP_BUFFER.append(sServerDomain);
			TMP_BUFFER.append("]LookupCreationError-");
			TMP_BUFFER.append(e.toString());
			return null;
		}

		//해당 도메인에 대한 접근을 제한해야한다.
		synchronized (SEARCH_LOOKUP) {
			int answer_result = __RESOLVE_SERVERS.process(SEARCH_LOOKUP, TMP_BUFFER);
			if( answer_result == Lookup.SUCCESSFUL ) {
				return SEARCH_LOOKUP.getResultMxList();
			}

			// 에러상황을 기록한다.
			TMP_BUFFER.append("[");
			TMP_BUFFER.append(sServerDomain);
			TMP_BUFFER.append("]error-");
			TMP_BUFFER.append(SEARCH_LOOKUP.getErrorString());
			return null;
		}
	}

	/**
	 * 등록된 상태를 반환한다.
	 * 
	 * @return 00: 정상 도메인 etc: 에러코드
	 */
	public boolean isError() {
		return this.error || this.PERMANENT_ERROR_FLAG;
	}

	public Throwable getErrorMsg() {
		return this.error_msg;
	}

	public String toString() {
		return this.SERVER_LIST.toString();
	}

	public void log(String log) {
		LookupContainer.log(log);
	}

	public void log(Throwable thw) {
		LookupContainer.log(thw);
	}
}
