/*
 * eMsStringBuffer.java
 *
 * Created on 2004년 6월 19일 (토), 오후 3:26
 */

package pluto.lang;

import lombok.extern.slf4j.Slf4j;
import pluto.util.FIFOBuffer;

/**
 * java.lang.StringBuffer는 재상용에 있어서 지속적인 char 배열을 재생성하기 때문에 이를 보완하여 재사용 가능한 형태로 변형 단 이 Buffer는 java.lang.StringBuffer 처럼 동기화가 이루어 지지 않기 때문에 멀티 Thread에서 동일한 내용을 참조하게 되면 내용의 변형이 되기 때문에 Thread 공용변수로 사용은
 * 하지 않는 것이 좋다.
 * 
 * 결과적으로 String을 조립하여 결과 String을 만드는 경우에만 사용이 가능할듯 하다...^^
 * 
 * @author EMS
 */
@Slf4j
public class eMsStringBuffer extends Name {

	//자주 사용되기 때문에 재사용을 위한 버퍼를 제공한다.
	private static final int		MAX_CONTAIN_SIZE	= 10;

	private static FIFOBuffer		INNER_CONTAINER		= null;

	static {
		INNER_CONTAINER = new FIFOBuffer(MAX_CONTAIN_SIZE);
	}

	public static eMsStringBuffer getInstance() {
		Object tmpValue = INNER_CONTAINER.pop();

		eMsStringBuffer returnValue = tmpValue == null ? new eMsStringBuffer() : (eMsStringBuffer) tmpValue;

		returnValue.setCallerName(Thread.currentThread().getName());
		return returnValue;
	}

	public static void recycleInstance(eMsStringBuffer tmp) {
		if( tmp == null )
			return;

		if (log.isDebugEnabled()) {
			String tName = tmp.getCallerName();
			if( !Thread.currentThread().getName().equals(tName) ) {
				Exception e = new Exception("defferent .. worker:" + Thread.currentThread().getName() + "## getter" + tmp.getCallerName() + " => " + tmp.getName());
				log.error("error", e);
			}
		}

		if( tmp.length() > 4096 ) {
			if (log.isDebugEnabled()) 
				log.debug(Thread.currentThread().getName() + "=>" + tmp.getName() + " to large so no recycle");
			tmp = null;
			return;
		}

		if( INNER_CONTAINER.contains(tmp) ) {
			if (log.isDebugEnabled()) {
				Exception e = new Exception(Thread.currentThread().getName() + "=>" + tmp.getName() + " is already in eMsStringBuffer FIFOBuffer");
				log.error("", e);
			}
			return;
		}

		if (log.isDebugEnabled()) {
			Exception e = new Exception(Thread.currentThread().getName() + "=>" + tmp.getName() + " is recycle");
			log.error("error", e);
		}
		tmp.reset();

		INNER_CONTAINER.push(tmp);
	}

	/**
	 * 컨텐트 꼬이는거 때문에 로그를 작성할 녀석을 결정한다. private LogChannel __TRACE_LOG_CHANNEL__ = null;
	 */

	/**
	 * The value is used for character storage.
	 * 
	 * @serial
	 */
	private char	value[];

	/**
	 * The count is the number of characters in the buffer.
	 * 
	 * @serial
	 */
	private int		count;

	private String	callerName	= null;

	private boolean	calledFalg	= false;

	/**
	 * Constructs a string buffer with no characters in it and an initial capacity of 16 characters.
	 */
	private eMsStringBuffer() {
		this(16);
	}

	/**
	 * Constructs a string buffer with no characters in it and an initial capacity specified by the <code>length</code> argument.
	 * 
	 * @param length
	 *        the initial capacity.
	 * @exception NegativeArraySizeException
	 *            if the <code>length</code> argument is less than <code>0</code>.
	 */
	private eMsStringBuffer(int length) {
		value = new char[length];
		setName("pluto.lang.eMsStringBuffer:".concat(UniqueKey.get()));
	}

	/**
	 * Constructs a string buffer so that it represents the same sequence of characters as the string argument; in other words, the initial contents of the string buffer is a copy of the argument
	 * string. The initial capacity of the string buffer is <code>16</code> plus the length of the string argument.
	 * 
	 * @param str
	 *        the initial contents of the buffer.
	 * @exception NullPointerException
	 *            if <code>str</code> is <code>null</code>
	 */
	public eMsStringBuffer(String str) {
		this(str.length() + 16);
		append(str);
	}

	public void setCallerName(String name) {
		if( this.calledFalg ) {
			throw new RuntimeException("ALREADY CALL FROM:" + this.getCallerName());
		}
		this.callerName = name;
		this.calledFalg = true;
	}

	public String getCallerName() {
		return this.callerName;
	}

	/**
	 * Returns the length (character count) of this string buffer.
	 * 
	 * @return the length of the sequence of characters currently represented by this string buffer.
	 */
	public synchronized int length() {
		return count;
	}

	/**
	 * Returns the current capacity of the String buffer. The capacity is the amount of storage available for newly inserted characters; beyond which an allocation will occur.
	 * 
	 * @return the current capacity of this string buffer.
	 */
	public synchronized int capacity() {
		return value.length;
	}

	/**
	 * Ensures that the capacity of the buffer is at least equal to the specified minimum. If the current capacity of this string buffer is less than the argument, then a new internal buffer is
	 * allocated with greater capacity. The new capacity is the larger of:
	 * <ul>
	 * <li>The <code>minimumCapacity</code> argument.
	 * <li>Twice the old capacity, plus <code>2</code>.
	 * </ul>
	 * If the <code>minimumCapacity</code> argument is nonpositive, this method takes no action and simply returns.
	 * 
	 * @param minimumCapacity
	 *        the minimum desired capacity.
	 */
	public synchronized void ensureCapacity(int minimumCapacity) {
		if( minimumCapacity > value.length ) {
			expandCapacity(minimumCapacity);
		}
	}

	/**
	 * This implements the expansion semantics of ensureCapacity but is unsynchronized for use internally by methods which are already synchronized.
	 * 
	 * @see java.lang.StringBuffer#ensureCapacity(int)
	 */
	private void expandCapacity(int minimumCapacity) {
		int newCapacity = (value.length + 1) * 2;
		if( newCapacity < 0 ) {
			newCapacity = Integer.MAX_VALUE;
		}
		else if( minimumCapacity > newCapacity ) {
			newCapacity = minimumCapacity;
		}

		char newValue[] = new char[newCapacity];
		System.arraycopy(value, 0, newValue, 0, count);
		value = newValue;
	}

	/**
	 * Sets the length of this String buffer. java.lang.StringBuffer 에서는 현재 저장된 버퍼 사이즈보다 클경우에는 그만큼 버퍼 사이즈를 늘리고 '\0'으로 나머지를 채우는 로직으로 구현되었지만 여기서는 현재 저장된 버퍼사이즈 보다 크게 늘리는 것은 지원하지 않는다. append된 결과에서 마지막을
	 * 지울때 사용한다.
	 * 
	 * @param newLength
	 *        the new length of the buffer.
	 * @exception IndexOutOfBoundsException
	 *            if the <code>newLength</code> argument is negative or argument is bigger then buffer length
	 * @see java.lang.StringBuffer#length()
	 */
	public synchronized void setLength(int newLength) {
		if( (newLength < 0) || (newLength > count) ) {
			throw new StringIndexOutOfBoundsException(newLength);
		}

		count = newLength;
		/**
		 * if( this.__TRACE_LOG_CHANNEL__ != null ){ try{ this.__TRACE_LOG_CHANNEL__.write( Thread.currentThread().getName() + "=>" + this.getName() + " is call setLength -> " + String.valueOf(
		 * newLength ) ); } catch( Exception e ){ } }
		 */
	}

	/**
	 * public synchronized void setLogChannel( LogChannel channel ){ this.__TRACE_LOG_CHANNEL__ = channel; }
	 */

	private synchronized void reset() {
		count = 0;
		this.calledFalg = false;
		/**
		 * this.__TRACE_LOG_CHANNEL__ = null;
		 */
	}

	public synchronized void minusLength(int newLength) {
		if( (newLength < 0) || (newLength > count) ) {
			throw new StringIndexOutOfBoundsException(newLength);
		}

		count = count - newLength;
	}

	/**
	 * The specified character of the sequence currently represented by the string buffer, as indicated by the <code>index</code> argument, is returned. The first character of a string buffer is at
	 * index <code>0</code>, the next at index <code>1</code>, and so on, for array indexing.
	 * <p>
	 * The index argument must be greater than or equal to <code>0</code>, and less than the length of this string buffer.
	 * 
	 * @param index
	 *        the index of the desired character.
	 * @return the character at the specified index of this string buffer.
	 * @exception IndexOutOfBoundsException
	 *            if <code>index</code> is negative or greater than or equal to <code>length()</code>.
	 * @see java.lang.StringBuffer#length()
	 */
	public synchronized char charAt(int index) {
		if( (index < 0) || (index >= count) ) {
			throw new StringIndexOutOfBoundsException(index);
		}
		return value[index];
	}

	/**
	 * Appends the string to this string buffer.
	 * <p>
	 * java.lang.StringBuffer 에서는 null을 null로 append 하지만 eMsStringBuffer에서는 append를 하지 않는다.
	 * 
	 * @param str
	 *        a string.
	 */
	public synchronized void append(String str) {
		if( str == null ) {
			return;
		}

		int len = str.length();
		int newcount = count + len;

		if( newcount > value.length ) {
			expandCapacity(newcount);
		}

		str.getChars(0, len, value, count);
		count = newcount;
	}

	public synchronized String reverseString() {
		int n = count - 1;
		for (int j = (n - 1) >> 1; j >= 0; --j) {
			char temp = value[j];
			value[j] = value[n - j];
			value[n - j] = temp;
		}

		return toString();
	}

	public synchronized void append(String str, int beginIndex) {
		append(str, beginIndex, str.length());
	}

	/**
	 * Appends the string to this string buffer.
	 * <p>
	 * java.lang.StringBuffer 에서는 null을 null로 append 하지만 eMsStringBuffer에서는 append를 하지 않는다.
	 * 
	 * @param str
	 *        a string.
	 */
	public synchronized void append(String str, int beginIndex, int endIndex) {
		if( str == null ) {
			return;
		}

		// 구간이 없으면 append 하지 않는다.
		if( endIndex == beginIndex )
			return;

		// 길이를 구한다음
		int len = endIndex - beginIndex;

		// 길이가 음수이면 안된다.
		if( len < 0 ) {
			throw new StringIndexOutOfBoundsException("Begin Index is Bigger than End Index " + len);
		}

		// 종료점이 str의 길이보다 크면 안된다.
		if( endIndex > str.length() ) {
			throw new StringIndexOutOfBoundsException("End Index is bigger than source length " + endIndex);
		}

		//시작점이 음수일수는 없다.
		if( beginIndex < 0 ) {
			throw new StringIndexOutOfBoundsException("Begin Index is negative " + endIndex);
		}

		int newcount = count + len;

		if( newcount > value.length ) {
			expandCapacity(newcount);
		}

		str.getChars(beginIndex, len, value, count);
		count = newcount;
	}

	public synchronized void append(char str[], int offset, int len) {
		int newcount = count + len;
		if( newcount > value.length )
			expandCapacity(newcount);
		System.arraycopy(str, offset, value, count, len);
		count = newcount;
	}

	public synchronized void append(char str[]) {
		int len = str.length;
		int newcount = count + len;
		if( newcount > value.length )
			expandCapacity(newcount);
		System.arraycopy(str, 0, value, count, len);
		count = newcount;
	}

	public synchronized void append(char c) {
		int newcount = count + 1;
		if( newcount > value.length )
			expandCapacity(newcount);
		value[count++] = c;
		//return this;
	}

	public synchronized void appendReverse(String str) {
		for (int i = str.length(); i > 0; i--) {
			append(str.charAt(i-1));
		}
	}

	/**
	 * Removes the characters in a substring of this <code>StringBuffer</code>. The substring begins at the specified <code>start</code> and extends to the character at index <code>end - 1</code>
	 * or to the end of the <code>StringBuffer</code> if no such character exists. If <code>start</code> is equal to <code>end</code>, no changes are made.
	 * 
	 * @param start
	 *        The beginning index, inclusive.
	 * @param end
	 *        The ending index, exclusive.
	 * @return This string buffer.
	 * @exception StringIndexOutOfBoundsException
	 *            if <code>start</code> is negative, greater than <code>length()</code>, or greater than <code>end</code>.
	 * @since 1.2
	 */
	public synchronized void delete(int start, int end) {
		if( start < 0 )
			throw new StringIndexOutOfBoundsException(start);
		if( end > count )
			end = count;
		if( start > end )
			throw new StringIndexOutOfBoundsException();

		int len = end - start;
		if( len > 0 ) {
			System.arraycopy(value, start + len, value, start, count - end);
			count -= len;
		}
	}

	/**
	 * Replaces the characters in a substring of this <code>StringBuffer</code> with characters in the specified <code>String</code>. The substring begins at the specified <code>start</code>
	 * and extends to the character at index <code>end - 1</code> or to the end of the <code>StringBuffer</code> if no such character exists. First the characters in the substring are removed and
	 * then the specified <code>String</code> is inserted at <code>start</code>. (The <code>StringBuffer</code> will be lengthened to accommodate the specified String if necessary.)
	 * 
	 * @param start
	 *        The beginning index, inclusive.
	 * @param end
	 *        The ending index, exclusive.
	 * @param str
	 *        String that will replace previous contents.
	 * @return This string buffer.
	 * @exception StringIndexOutOfBoundsException
	 *            if <code>start</code> is negative, greater than <code>length()</code>, or greater than <code>end</code>.
	 * @since 1.2
	 */
	public synchronized void replace(int start, int end, String str) {
		if( start < 0 )
			throw new StringIndexOutOfBoundsException(start);
		if( end > count )
			end = count;
		if( start > end )
			throw new StringIndexOutOfBoundsException(toString() + ":" + String.valueOf(start) + ":" + String.valueOf(end) + ":" + str);

		int len = str.length();
		int newCount = count + len - (end - start);
		if( newCount > value.length ) {
			expandCapacity(newCount);
		}

		System.arraycopy(value, end, value, start + len, count - end);
		str.getChars(0, len, value, start);
		count = newCount;
	}

	public synchronized void replace(String org, String str) {
		char[] orgArray = org.toCharArray();

		int org_length = orgArray.length;
		int idx_start = 0;
		int idx_end = 0;
		while (true) {
			idx_start = indexOf(value, 0, count, orgArray, 0, orgArray.length, idx_end);

			if( idx_start < 0 )
				break;

			replace(idx_start, idx_start + org_length, str);

			idx_end = idx_start + str.length();
		}
	}

	public static void main(String[] arg) throws Exception {
		eMsStringBuffer buffer = new eMsStringBuffer();

		buffer.append("112233344556677338338990033");
		buffer.replace("33", "하하하");

		log.debug(buffer.toString());
	}

	/**
	 * Returns a new <code>String</code> that contains a subsequence of characters currently contained in this <code>StringBuffer</code> .The substring begins at the specified index and extends to
	 * the end of the <code>StringBuffer</code>.
	 * 
	 * @param start
	 *        The beginning index, inclusive.
	 * @return The new string.
	 * @exception StringIndexOutOfBoundsException
	 *            if <code>start</code> is less than zero, or greater than the length of this <code>StringBuffer</code>.
	 * @since 1.2
	 */
	public synchronized String substring(int start) {
		return substring(start, count);
	}

	/**
	 * Returns a new <code>String</code> that contains a subsequence of characters currently contained in this <code>StringBuffer</code>. The substring begins at the specified <code>start</code>
	 * and extends to the character at index <code>end - 1</code>. An exception is thrown if
	 * 
	 * @param start
	 *        The beginning index, inclusive.
	 * @param end
	 *        The ending index, exclusive.
	 * @return The new string.
	 * @exception StringIndexOutOfBoundsException
	 *            if <code>start</code> or <code>end</code> are negative or greater than <code>length()</code>, or <code>start</code> is greater than <code>end</code>.
	 * @since 1.2
	 */
	public synchronized String substring(int start, int end) {
		if( start < 0 )
			throw new StringIndexOutOfBoundsException(start);
		if( end > count )
			throw new StringIndexOutOfBoundsException(end);
		if( start > end )
			throw new StringIndexOutOfBoundsException(end - start);
		return new String(value, start, end - start);
	}

	/**
	 * Inserts the string representation of a subarray of the <code>str</code> array argument into this string buffer. The subarray begins at the specified <code>offset</code> and extends
	 * <code>len</code> characters. The characters of the subarray are inserted into this string buffer at the position indicated by <code>index</code>. The length of this
	 * <code>StringBuffer</code> increases by <code>len</code> characters.
	 * 
	 * @param index
	 *        position at which to insert subarray.
	 * @param str
	 *        A character array.
	 * @param offset
	 *        the index of the first character in subarray to to be inserted.
	 * @param len
	 *        the number of characters in the subarray to to be inserted.
	 * @exception StringIndexOutOfBoundsException
	 *            if <code>index</code> is negative or greater than <code>length()</code>, or <code>offset</code> or <code>len</code> are negative, or <code>(offset+len)</code> is greater
	 *            than <code>str.length</code>.
	 */
	public synchronized void insert(int index, char str[], int offset, int len) {
		if( (index < 0) || (index > count) )
			throw new StringIndexOutOfBoundsException();
		if( (offset < 0) || (offset + len < 0) || (offset + len > str.length) )
			throw new StringIndexOutOfBoundsException(offset);
		if( len < 0 )
			throw new StringIndexOutOfBoundsException(len);
		int newCount = count + len;
		if( newCount > value.length ) {
			expandCapacity(newCount);
		}
		System.arraycopy(value, index, value, index + len, count - index);
		System.arraycopy(str, offset, value, index, len);
		count = newCount;
	}

	/**
	 * Inserts the string into this string buffer.
	 * <p>
	 * The characters of the <code>String</code> argument are inserted, in order, into this string buffer at the indicated offset, moving up any characters originally above that position and
	 * increasing the length of this string buffer by the length of the argument. If <code>str</code> is <code>null</code>, then the four characters <code>"null"</code> are inserted into this
	 * string buffer.
	 * <p>
	 * The character at index <i>k </i> in the new character sequence is equal to:
	 * <ul>
	 * <li>the character at index <i>k </i> in the old character sequence, if <i>k </i> is less than <code>offset</code>
	 * <li>the character at index <i>k </i> <code>-offset</code> in the argument <code>str</code>, if <i>k </i> is not less than <code>offset</code> but is less than
	 * <code>offset+str.length()</code>
	 * <li>the character at index <i>k </i> <code>-str.length()</code> in the old character sequence, if <i>k </i> is not less than <code>offset+str.length()</code>
	 * </ul>
	 * <p>
	 * The offset argument must be greater than or equal to <code>0</code>, and less than or equal to the length of this string buffer.
	 * 
	 * @param offset
	 *        the offset.
	 * @param str
	 *        a string.
	 * @exception StringIndexOutOfBoundsException
	 *            if the offset is invalid.
	 */
	public synchronized void insert(int offset, String str) {
		if( str == null ) {
			return;
		}

		if( (offset < 0) || (offset > count) ) {
			throw new StringIndexOutOfBoundsException();
		}

		int len = str.length();
		int newcount = count + len;
		if( newcount > value.length ) {
			expandCapacity(newcount);
		}
		System.arraycopy(value, offset, value, offset + len, count - offset);
		str.getChars(0, len, value, offset);
		count = newcount;
	}

	/**
	 * Inserts the string representation of the <code>char</code> array argument into this string buffer.
	 * <p>
	 * The characters of the array argument are inserted into the contents of this string buffer at the position indicated by <code>offset</code>. The length of this string buffer increases by the
	 * length of the argument.
	 * <p>
	 * The overall effect is exactly as if the argument were converted to a string by the method {@link String#valueOf(char[])}and the characters of that string were then
	 * {@link #insert(int,String) inserted}into this <code>StringBuffer</code> object at the position indicated by <code>offset</code>.
	 * 
	 * @param offset
	 *        the offset.
	 * @param str
	 *        a character array.
	 * @return a reference to this <code>StringBuffer</code> object.
	 * @exception StringIndexOutOfBoundsException
	 *            if the offset is invalid.
	 */
	public synchronized void insert(int offset, char str[]) {
		if( str == null )
			return;

		if( (offset < 0) || (offset > count) ) {
			throw new StringIndexOutOfBoundsException();
		}
		int len = str.length;
		int newcount = count + len;
		if( newcount > value.length ) {
			expandCapacity(newcount);
		}

		System.arraycopy(value, offset, value, offset + len, count - offset);
		System.arraycopy(str, 0, value, offset, len);
		count = newcount;
	}

	/**
	 * Inserts the string representation of the <code>char</code> argument into this string buffer.
	 * <p>
	 * The second argument is inserted into the contents of this string buffer at the position indicated by <code>offset</code>. The length of this string buffer increases by one.
	 * <p>
	 * The overall effect is exactly as if the argument were converted to a string by the method {@link String#valueOf(char)}and the character in that string were then
	 * {@link #insert(int, String) inserted}into this <code>StringBuffer</code> object at the position indicated by <code>offset</code>.
	 * <p>
	 * The offset argument must be greater than or equal to <code>0</code>, and less than or equal to the length of this string buffer.
	 * 
	 * @param offset
	 *        the offset.
	 * @param c
	 *        a <code>char</code>.
	 * @return a reference to this <code>StringBuffer</code> object.
	 * @exception IndexOutOfBoundsException
	 *            if the offset is invalid.
	 * @see java.lang.StringBuffer#length()
	 */
	public synchronized void insert(int offset, char c) {
		int newcount = count + 1;
		if( newcount > value.length ) {
			expandCapacity(newcount);
		}
		System.arraycopy(value, offset, value, offset + 1, count - offset);
		value[offset] = c;
		count = newcount;
	}

	/**
	 * Converts to a string representing the data in this string buffer. A new <code>String</code> object is allocated and initialized to contain the character sequence currently represented by this
	 * string buffer. This <code>String</code> is then returned. Subsequent changes to the string buffer do not affect the contents of the <code>String</code>.
	 * <p>
	 * Implementation advice: This method can be coded so as to create a new <code>String</code> object without allocating new memory to hold a copy of the character sequence. Instead, the string
	 * can share the memory used by the string buffer. Any subsequent operation that alters the content or capacity of the string buffer must then make a copy of the internal buffer at that time. This
	 * strategy is effective for reducing the amount of memory allocated by a string concatenation operation when it is implemented using a string buffer.
	 * 
	 * @return a string representation of the string buffer.
	 */
	public synchronized String toString() {
		// for debug 로그 체널이 있으면 거기에 반환되는 녀석을 작성한다.
		/**
		 * if( this.__TRACE_LOG_CHANNEL__ != null ){ try{ this.__TRACE_LOG_CHANNEL__.write( Thread.currentThread().getName() + "=>" + this.getName() + " toString() " ); } catch( Exception e ){ } }
		 */

		return new String(value, 0, count);
	}

	/**
	 */
	public int indexOfCaseIgnore(String org) {
		return indexOfCaseIgnore(org, 0);
	}

	public int indexOfCaseIgnore(String org, int offset) {
		return _indexOfCaseIgnore(value, 0, count, org.toCharArray(), 0, org.length(), offset);
	}

	public int indexOfLineEnd(int offset) {
		do {
			if( value[offset] == '\r' || value[offset] == '\n' ) {
				return offset;
			}
			offset++;
		} while (offset < count);
		return count;
	}

	/**
	 * Code shared by String and StringBuffer to do searches. The source is the character array being searched, and the target is the string being searched for.
	 * 
	 * @param source
	 *        the characters being searched.
	 * @param sourceOffset
	 *        offset of the source string.
	 * @param sourceCount
	 *        count of the source string.
	 * @param target
	 *        the characters being searched for.
	 * @param targetOffset
	 *        offset of the target string.
	 * @param targetCount
	 *        count of the target string.
	 * @param fromIndex
	 *        the index to begin searching from.
	 */
	static int indexOf(char[] source, int sourceOffset, int sourceCount, char[] target, int targetOffset, int targetCount, int fromIndex) {
		if( fromIndex >= sourceCount ) {
			return (targetCount == 0 ? sourceCount : -1);
		}
		if( fromIndex < 0 ) {
			fromIndex = 0;
		}
		if( targetCount == 0 ) {
			return fromIndex;
		}

		char first = target[targetOffset];
		int i = sourceOffset + fromIndex;
		int max = sourceOffset + (sourceCount - targetCount);

		startSearchForFirstChar: while (true) {
			/* Look for first character. */
			while (i <= max && source[i] != first) {
				i++;
			}
			if( i > max ) {
				return -1;
			}

			/* Found first character, now look at the rest of v2 */
			int j = i + 1;
			int end = j + targetCount - 1;
			int k = targetOffset + 1;
			while (j < end) {
				if( source[j++] != target[k++] ) {
					i++;
					/* Look for str's first char again. */
					continue startSearchForFirstChar;
				}
			}
			return i - sourceOffset; /* Found whole string. */
		}
	}

	static int upperCase(char src) {
		int returnValue = src;

		if( returnValue >= 97 && returnValue <= 122 ) {
			return returnValue - 32;
		}

		return returnValue;
	}

	/**
	 * Code shared by String and StringBuffer to do searches. The source is the character array being searched, and the target is the string being searched for.
	 * 
	 * @param source
	 *        the characters being searched.
	 * @param sourceOffset
	 *        offset of the source string.
	 * @param sourceCount
	 *        count of the source string.
	 * @param target
	 *        the characters being searched for.
	 * @param targetOffset
	 *        offset of the target string.
	 * @param targetCount
	 *        count of the target string.
	 * @param fromIndex
	 *        the index to begin searching from.
	 */
	public static int _indexOfCaseIgnore(char[] source, int sourceOffset, int sourceCount, char[] target, int targetOffset, int targetCount, int fromIndex) {
		if( fromIndex >= sourceCount ) {
			return (targetCount == 0 ? sourceCount : -1);
		}
		if( fromIndex < 0 ) {
			fromIndex = 0;
		}
		if( targetCount == 0 ) {
			return fromIndex;
		}

		int first = upperCase(target[targetOffset]);

		int i = sourceOffset + fromIndex;
		int max = sourceOffset + (sourceCount - targetCount);

		startSearchForFirstChar: while (true) {
			/* Look for first character. */
			while (i <= max && upperCase(source[i]) != first) {
				i++;
			}
			if( i > max ) {
				return -1;
			}

			/* Found first character, now look at the rest of v2 */
			int j = i + 1;
			int end = j + targetCount - 1;
			int k = targetOffset + 1;
			while (j < end) {
				if( upperCase(source[j++]) != upperCase(target[k++]) ) {
					i++;
					/* Look for str's first char again. */
					continue startSearchForFirstChar;
				}
			}
			return i - sourceOffset; /* Found whole string. */
		}
	}

	public int findCharArray(char[] target, int i, int end_point) {

		end_point = end_point > count ? count : end_point;

		for (int idx = i; idx < end_point; idx++) {
			for (int sidx = target.length; sidx > 0; sidx--) {
				if( value[idx] == target[sidx] )
					return idx;
			}
		}

		return -1;
	}
}
