/*
 * Created on 2005. 7. 25
 */
package pluto.ioutil;

import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.lang.reflect.Method;

import lombok.extern.slf4j.Slf4j;
import pluto.lang.eMsLocale;

/**
 * @author 이상근
 */
@Slf4j
public class FileUtil {
	public static void putStringToLocalFile(File target, String sContent, String enc) throws Exception {
		try {
			try(
				FileOutputStream fos = new FileOutputStream(target);	
				OutputStreamWriter osw = new OutputStreamWriter(fos, enc);
				PrintWriter pw = new PrintWriter(osw);
				){
				pw.print(sContent);
				pw.flush();
				} catch(Exception e) {
					log.error("Exception",e);
				}
		} catch(Exception ex) {
			log.error("Exception", ex);
		}
	}

	public static void putLocalFileToStream(File target, OutputStream stream) throws Exception {
		FileInputStream in = null;
		try {
			in = new FileInputStream(target);
			byte[] buffer = new byte[256];

			int read = 0;

			while ((read = in.read(buffer)) > 0) {
				stream.write(buffer, 0, read);
			}
		}
		catch(Throwable thw) {
			log.error("error", thw);
			if( thw instanceof Exception ) {
				throw (Exception) thw;
			}
			throw new Exception(thw.toString());
		}
		finally {
			close(in);
		}
	}

	/**
	 * 원본 String을 로컬 File에 Write한다.
	 * 
	 * @param source
	 *            원본 String
	 * @param encoding
	 *            Write Charset
	 * @param escape
	 *            true:제어문자를 공백문자로 치환하여 write / false:제어문자를 그대로 적는다.
	 */
	public static void writeStringToLocalFile(String source, File target, String encoding, boolean escape) throws IOException {
		try {
			try(
				FileOutputStream fos = new FileOutputStream(target);
				OutputStreamWriter osw = new OutputStreamWriter(fos, encoding);
				BufferedWriter writer  = new BufferedWriter(osw);
				){
				// 공백 문자 치환이 아니라면 그냥 쓰고 돌아간다.
				if( !escape ) {
					writer.write(source);
					writer.flush();
					return;
				}
	
				// 아니라면 32 이하의 10/13 을 제외한 문자는 ' '로 치환된다.
				char[] sourceArray = source.toCharArray();
	
				for (int i = 0; i < sourceArray.length; i++) {
	
					if( sourceArray[i] < 32 ) {
						switch (sourceArray[i]) {
							case 10:
							case 13: {
								writer.write(sourceArray, i, 1);
								break;
							}
	
							default: {
								writer.write(" ");
								break;
							}
						}
					}
					else {
						// 제어문자가 아닌거는 그냥 쓰고.
						writer.write(sourceArray, i, 1);
					}
	
				}
				
				// 버퍼를 비운다.
				writer.flush();
			} catch(Exception e) {
				log.error("Exception", e);
			}
		} catch(Exception ex) {
			log.error("Exception", ex);
		}
	}

	/**
	 * 로컬 파일을 삭제한다.
	 * 
	 * @param url
	 *            파일 경로
	 * @return null:삭제 성공 else 실패한 Exception 내용
	 */
	public static final String deleteLocalFile(String url) {
		return deleteLocalFile(new File(url));
	}

	public static final String deleteLocalFile(File file) {
		boolean flag = false;
		try{
			flag = file.delete();
		}
		catch(Throwable thw){
			return thw.toString();
		}
		return flag?null:"RESULT FALSE";
	}

	public synchronized static final void close(Object obj) {
		if( obj == null ) {
			return;
		}

		Class classObj = obj.getClass();
		Method method = null;
		try {
			method = classObj.getMethod("close", null);
		}
		catch(Throwable thw) {
			// TODO 적당한 에러처리를 생각해야할듯 하다.
			log.error("getMethod error", thw);
		}

		try {
			if(method != null) {
				method.invoke(obj, null);
			}
		}
		catch(Throwable thw) {
			// ignore
			log.error("method invoke error", thw);
		}
	}
	
	/**
	 * Local File을 읽어 String으로 반환한다.
	 * 
	 * @param f
	 * @return
	 * @throws IOException
	 */
	public static String readLocal(File f) throws IOException {

		FileInputStream fis = new FileInputStream(f);
		byte[] b = new byte[2 << 6];

		int j;

		ByteArrayOutputStream baos = new ByteArrayOutputStream();

		try {
			while (true) {
				j = fis.read(b);
				if( j <= 0 )
					break;
	
				baos.write(b, 0, j);
			}
		} catch(Exception e) {
			log.error(e.getMessage());
			
		} finally {
			if(fis != null) {
				fis.close();
			}
		}

		return baos.toString(eMsLocale.CHAR_SET);
	}

	/**
	 * Local File을 읽어 String으로 반환한다.
	 * 
	 * @param fName
	 * @return
	 * @throws IOException
	 */
	public static String readLocal(String fName) throws IOException {

		File f = new File(fName);
		return readLocal(f);
	}	
}
