/*
 * ResultSetOutputStream.java
 *
 * Created on 2004년 9월 2일 (목), 오전 10:37
 */

package pluto.io;

import java.io.IOException;
import java.io.OutputStream;
import java.sql.ResultSetMetaData;

import lombok.extern.slf4j.Slf4j;
import pluto.db.eMsResultSet;
import pluto.lang.ByteArrayContainer;
import pluto.lang.eMsLocale;
import pluto.util.convert.BASE64;

/**
 * 
 * @author EMS
 */
@Slf4j
public class eMsResultSetOutputStream extends java.io.FilterOutputStream implements eMsResultSetStream {

	/**
	 * The internal buffer where data is stored.
	 */
	protected byte					buf[];

	/**
	 * The number of valid bytes in the buffer. This value is always in the
	 * range <tt>0</tt> through <tt>buf.length</tt>; elements
	 * <tt>buf[0]</tt> through <tt>buf[count-1]</tt> contain valid byte
	 * data.
	 */
	protected int					count;

	protected byte[]				ENC_INT_SOURCE	= new byte[3];

	protected byte[]				ENC_INT_TARGET	= new byte[4];

	/** Creates a new instance of ResultSetOutputStream */
	public eMsResultSetOutputStream(OutputStream o) {
		super(o);
		this.buf = ByteArrayContainer.getInstance();
	}

	private void flushBuffer() throws IOException {
		if( count > 0 ) {
			out.write(buf, 0, count);
			count = 0;
		}
	}

	/**
	 * Writes the specified byte to this buffered output stream.
	 * 
	 * @param b
	 *            the byte to be written.
	 * @exception IOException
	 *                if an I/O error occurs.
	 */
	public synchronized void write(int b) throws IOException {
		if( count >= buf.length ) {
			flushBuffer();
		}
		buf[count++] = (byte) b;
	}

	/**
	 * Writes <code>len</code> bytes from the specified byte array starting at
	 * offset <code>off</code> to this buffered output stream.
	 * 
	 * <p>
	 * Ordinarily this method stores bytes from the given array into this
	 * stream's buffer, flushing the buffer to the underlying output stream as
	 * needed. If the requested length is at least as large as this stream's
	 * buffer, however, then this method will flush the buffer and write the
	 * bytes directly to the underlying output stream. Thus redundant
	 * <code>BufferedOutputStream</code> s will not copy data unnecessarily.
	 * 
	 * @param b
	 *            the data.
	 * @param off
	 *            the start offset in the data.
	 * @param len
	 *            the number of bytes to write.
	 * @exception IOException
	 *                if an I/O error occurs.
	 */
	public synchronized void write(byte b[], int off, int len) throws IOException {
		if( len >= buf.length ) {
			/*
			 * If the request length exceeds the size of the output buffer,
			 * flush the output buffer and then write the data directly. In this
			 * way buffered streams will cascade harmlessly.
			 */
			flushBuffer();
			out.write(b, off, len);
			return;
		}
		if( len > buf.length - count ) {
			flushBuffer();
		}
		System.arraycopy(b, off, buf, count, len);
		count += len;
	}

	/**
	 * Flushes this buffered output stream. This forces any buffered output
	 * bytes to be written out to the underlying output stream.
	 * 
	 * @exception IOException
	 *                if an I/O error occurs.
	 * @see java.io.FilterOutputStream#out
	 */
	public synchronized void flush() throws IOException {
		flushBuffer();
		out.flush();
	}

	public void close() throws IOException {
		write(TYPE_END);
		super.close();
		ByteArrayContainer.recycleInstance(buf);
	}

	public synchronized void writeResultSet(eMsResultSet rs, boolean header_print) throws Exception {
		// 일단 들어왔으면 헤더를 찍어야지..
		if( header_print )
			appendHeader(rs);

		int count = rs.getColumnCount();

		while (rs.next()) {
			// 바디표시하고
			write(TYPE_BODY);

			// 바디에 들어있는 컬럼수를 기록하고
			writeNumericType(count);
			for (int i = 1; i <= count; i++) {
				byte[] value = rs.getStringAsByteArray(i);
				writeNumericType(value.length);
				write(value);
			}
		}
	}

	protected void appendHeader(eMsResultSet rs) throws Exception {
		// 먼저 찍었으면 그냥 돌아가야하고..
		write(TYPE_HEADER);

		int count = rs.getColumnCount();

		// short value보다 크면 안됨 : 32767
		if( count > Short.MAX_VALUE ) {
			throw new IllegalArgumentException("TOO MANY COLUMNS:" + String.valueOf(count));
		}

		// 사이즈를 append
		writeNumericType(count);
		if (log.isDebugEnabled()) {
			log.debug("HEADER LENGTH : " + count);
		}

		String charset = rs.getOUT_CHARSET();
		ResultSetMetaData rsmt = rs.getMetaData();

		for (int i = 1; i <= count; i++) {
			byte[] colname = null;
			
			if( eMsLocale.DB_TYPE.equals("mysql") ){
				colname = rsmt.getColumnLabel(i).getBytes(charset);
				
			}else{
				colname = rsmt.getColumnName(i).getBytes(charset);				
			}

			writeNumericType(colname.length);
			if (log.isDebugEnabled()) {
				log.debug("WRITE HEADER LENGTH : " + colname.length);
			}
			write(colname);
		}
	}

	protected void writeNumericType(int val) throws IOException {
		this.ENC_INT_SOURCE[0] = (byte) (val >>> 16);
		this.ENC_INT_SOURCE[1] = (byte) (val >>> 8);
		this.ENC_INT_SOURCE[2] = (byte) (val >>> 0);

		BASE64.encode_sep(this.ENC_INT_SOURCE, 3, this.ENC_INT_TARGET);

		write(this.ENC_INT_TARGET[0]);
		write(this.ENC_INT_TARGET[1]);
		write(this.ENC_INT_TARGET[2]);
		write(this.ENC_INT_TARGET[3]);
	}
}
