/*
 * eMsFileReader.java
 *
 * Created on 2003년 2월 28일 금, 오후 4:29
 */

package pluto.io;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import lombok.extern.slf4j.Slf4j;
import pluto.lang.eMsLocale;
import pluto.lang.eMsStringBuffer;

/**
 * 파일을 읽어 들일때 한글처리와 추가 기능을 구현
 * 
 * @author 이상근
 * @version 3.0
 */
@Slf4j
public class eMsFileReader extends java.io.BufferedReader {

	/**
	 * 지정된 파일을 기본 케릭터 셋으로 BufferedReader를 생성한다.
	 * 
	 * @param file
	 *            파일이름
	 * @param size
	 *            buffer 사이즈
	 * @throws IOException
	 *             생성에러
	 */
	public eMsFileReader(String file) throws IOException {
		this(file, eMsLocale.FILE_SYSTEM_IN_CHAR_SET);
	}

	public eMsFileReader(String file, String char_set) throws IOException {
		this(new File(file), char_set);
	}

	/**
	 * 지정된 파일과 기본 buffer사이즈로 BufferedReader를 생성한다.
	 * 
	 * @param file
	 *            대상파일
	 * @throws IOException
	 *             에러
	 */
	public eMsFileReader(File file) throws IOException {
		this(file, eMsLocale.FILE_SYSTEM_IN_CHAR_SET);
	}

	/**
	 * 지정된 Stream과 기본 buffer 사이즈로 BufferedReader를 생성한다.
	 * 
	 * @param file
	 *            대상 Stream
	 * @throws IOException
	 *             에러
	 */
	public eMsFileReader(File file, String char_set) throws IOException {
		this(new FileInputStream(file), char_set);
	}

	public eMsFileReader(InputStream stream) throws IOException {
		this(stream, eMsLocale.FILE_SYSTEM_IN_CHAR_SET);
	}

	public eMsFileReader(InputStream stream, String char_set) throws IOException {
		super(new java.io.InputStreamReader(stream, char_set));
	}

	/**
	 * 지정된 파일의 마지막 라인값을 반환한다.
	 * 
	 * @param filename
	 *            대상 파일 경로
	 * @throws IOException
	 *             에러
	 * @return 마지막 라인 String
	 */
	public static String getLastLine(String filename) throws IOException {
		eMsFileReader myReader = new eMsFileReader(filename);
		String tmp = null;
		String lastline = null;
		
		try {
			while (myReader.ready()) {
				tmp = myReader.readLine();
				if( tmp == null )
					break;
				lastline = tmp;
			}
		} catch(Exception e) {
			log.error(e.getMessage());
		} finally {
		
			if(myReader != null) {
				myReader.close();
			}
		}

		return lastline;
	}

	public static String getFileBody(String filename, String charset) throws IOException {
		eMsFileReader myReader = new eMsFileReader(filename, charset);
		String tmp = null;
		String lastline = null;
		eMsStringBuffer buffer = null;
		try {
			buffer = eMsStringBuffer.getInstance();
			while (myReader.ready()) {
				tmp = myReader.readLine();
				if( tmp == null )
					break;
				buffer.append(tmp);
				buffer.append("\r\n");
			}
			return buffer.toString();
		}
		finally {
			if(myReader != null) {
				myReader.close();
			}
			eMsStringBuffer.recycleInstance(buffer);
		}
	}
}
