/*
 * eMsPreparedStatement.java
 *
 * Created on 2004년 2월 6일 (금), 오후 1:54
 */

package pluto.db;

import java.io.StringReader;
import java.sql.ResultSet;
import java.util.Iterator;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;
import pluto.util.PlutoLinkedList;

/**
 * 
 * @author t??
 */
@Slf4j
public class eMsPreparedStatement implements eMsStatementInterface {

	public static final char			STRING_PARAM_INDEX		= '\'';

	public static final short			STRING_TYPE_PARAMETER	= 1;

	public static final short			NUMERIC_TYPE_PARAMETER	= 2;

	private java.sql.PreparedStatement	ps						= null;

	private String						ORIGINAL_QUERY			= null;

	private String						CONVERT_QUERY			= null;

	private PlutoLinkedList				PARAMETER_INFOS			= null;

	private eMsConnection				EMS_CONNECTION			= null;

	/** Creates a new instance of eMsPreparedStatement */
	public eMsPreparedStatement(String src, String START_INDEX, String END_INDEX) throws Exception {
		ORIGINAL_QUERY = src;
		StringBuffer buffer = null;
		try {
			buffer = new StringBuffer(1024);
			buffer.setLength(0);
			convertSql(buffer, ORIGINAL_QUERY, START_INDEX, END_INDEX);
			CONVERT_QUERY = buffer.toString();
		}
		catch(Exception e) {
			throw e;
		}
		finally {
			buffer = null;
		}
	}

	public void connectTo(eMsConnection conn) throws java.sql.SQLException {
		if( ps != null ) {
			try {
				ps.close();
			}
			catch(Exception e) {
				// empty
			}
		}
		EMS_CONNECTION = conn;
		ps = EMS_CONNECTION.prepareStatement(conn.encode(CONVERT_QUERY));

	}

	public void close() {
		if( ps != null ) {
			try {
				ps.close();
			}
			catch(Exception e) {
				// empty
			}
		}
	}

	protected void convertSql(StringBuffer myBuffer, String source, String START_INDEX, String END_INDEX) throws Exception {
		if( source == null ) {
			throw new RuntimeException("TARGET QUERY IS NULL");
		}

		PARAMETER_INFOS = new PlutoLinkedList();

		source = source.concat("             ");

		int idx1 = 0;
		int idx2 = 0;
		int idx3 = 0;

		while (true) {
			idx1 = source.indexOf(START_INDEX, idx3);

			if( idx1 < 0 )
				break;

			idx2 = source.indexOf(END_INDEX, idx1);

			if( idx2 < 0 ) {
				throw new RuntimeException("PARSE ERROR NEAR=>" + source.substring(idx1 - 3));
			}

			/**
			 * String Type & Numeric Type 구분
			 */
			if(  source.charAt(idx1 - 1) == STRING_PARAM_INDEX && source.charAt(idx2 + END_INDEX.length()) == STRING_PARAM_INDEX ) {
				/**
				 * 스트링 타입의 파라미터 추가
				 */
				PARAMETER_INFOS.addLast(new eMsPreparedStatement.PreparedParameter(STRING_TYPE_PARAMETER, source.substring(idx1 + 2, idx2)));
				if (log.isDebugEnabled()) 
					log.debug("STRING_TYPE_PARAMETER:" + source.substring(idx1 + START_INDEX.length(), idx2) + " ADDED");
				
				myBuffer.append(source.substring(idx3, idx1 - 1));
				myBuffer.append("?");
				idx3 = idx2 + END_INDEX.length() + 1;
			}
			else if( source.charAt(idx1 - 1) != STRING_PARAM_INDEX && source.charAt(idx2 + END_INDEX.length()) != STRING_PARAM_INDEX ) {
				/**
				 * NUMERIC 타입의 파라미터 추가
				 */
				
				PARAMETER_INFOS.addLast(new eMsPreparedStatement.PreparedParameter(NUMERIC_TYPE_PARAMETER, source.substring(idx1 + 2, idx2)));
				if (log.isDebugEnabled()) 
					log.debug("NUMERIC_TYPE_PARAMETER:" + source.substring(idx1 + START_INDEX.length(), idx2) + " ADDED");

				myBuffer.append(source.substring(idx3, idx1));
				myBuffer.append("?");
				idx3 = idx2 + END_INDEX.length();
			}
			else {
				throw new RuntimeException("PARSE ERROR NEAR=>" + source.substring(idx1 - 3));
			}
		}

		myBuffer.append(source.substring(idx3));

		return;
	}

	protected synchronized void setParameter(Map map) throws java.sql.SQLException {
		if (log.isDebugEnabled()) 
			log.debug("INSERT PARAMETER:{}", map);
		int idx = 1;
		
		for (Iterator iter = PARAMETER_INFOS.iterator(); iter.hasNext();) {
			PreparedParameter param = (PreparedParameter) iter.next();
			
			Object value = map.get(param.getIndex());
			switch (param.getType()) {
				case STRING_TYPE_PARAMETER: {
					if (log.isDebugEnabled()) {
						if (value != null) {
							log.debug("STRING_TYPE_PARAMETER set:" + param.getIndex() + " => [" + value.toString() + "]");						
						} else {
							log.debug("STRING_TYPE_PARAMETER set:" + param.getIndex() + " => NULL");
						}
					}
					ps.setString(idx, value == null ? "" : EMS_CONNECTION.encode(value.toString()));
					break;
				}

				case NUMERIC_TYPE_PARAMETER: {
					if( value == null )
						throw new RuntimeException("CAN'T SET NUMERIC TO NULL:" + param.getIndex());
					if (log.isDebugEnabled()) 
						log.debug("NUMERIC_TYPE_PARAMETER set:" + param.getIndex() + " => [" + value.toString() + "]");
					ps.setInt(idx, Integer.parseInt(value.toString()));
					break;
				}

				default: {
					throw new RuntimeException("UNKNOWN TYPE:" + String.valueOf(param.getType()));
				}
			}
			idx++;
		}
	}
	
	protected synchronized void setParameterLong(Map map) throws java.sql.SQLException {
		if (log.isDebugEnabled()) 
			log.debug("INSERT PARAMETER:" + map.toString());
		int idx = 1;
		for (Iterator iter = PARAMETER_INFOS.iterator(); iter.hasNext();) {
			PreparedParameter param = (PreparedParameter) iter.next();
			Object value = map.get(param.getIndex());
			switch (param.getType()) {
				case STRING_TYPE_PARAMETER: {
					if (log.isDebugEnabled()) 
						log.debug("STRING_TYPE_PARAMETER set:" + param.getIndex() + " => [" + value.toString() + "]");
					ps.setString(idx, value == null ? "" : EMS_CONNECTION.encode(value.toString()));
					break;
				}
				case NUMERIC_TYPE_PARAMETER: {
					if( value == null )
						throw new RuntimeException("CAN'T SET NUMERIC TO NULL:" + param.getIndex());
					if (log.isDebugEnabled()) 
						log.debug("NUMERIC_TYPE_PARAMETER set:" + param.getIndex() + " => [" + value.toString() + "]");
					ps.setLong(idx, Long.parseLong(value.toString()));
					break;
				}
				default: {
					throw new RuntimeException("UNKNOWN TYPE:" + String.valueOf(param.getType()));
				}
			}
			idx++;
		}
	}
	
	protected synchronized void setParameterMMS(Map map) throws java.sql.SQLException {
		if (log.isDebugEnabled()) 
			log.debug("INSERT  MMS PARAMETER:" + map.toString());
		int idx = 1;
		for (Iterator iter = PARAMETER_INFOS.iterator(); iter.hasNext();) {
			PreparedParameter param = (PreparedParameter) iter.next();
			Object value = map.get(param.getIndex());
			switch (param.getType()) {
				case STRING_TYPE_PARAMETER: {
					if (log.isDebugEnabled()) 
						log.debug("STRING_TYPE_PARAMETER set:" + param.getIndex() + " => [" + value.toString() + "]");
					if(value.toString().getBytes().length <= 1024)
						ps.setString(idx, value == null ? "" : EMS_CONNECTION.encode(value.toString()));
					else
						ps.setCharacterStream(idx, value == null ? new StringReader("") : new StringReader(EMS_CONNECTION.encode(value.toString())), value.toString().length());
					break;
				}

				case NUMERIC_TYPE_PARAMETER: {
					if( value == null )
						throw new RuntimeException("CAN'T SET NUMERIC TO NULL:" + param.getIndex());
					if (log.isDebugEnabled()) 
						log.debug("NUMERIC_TYPE_PARAMETER set:" + param.getIndex() + " => [" + value.toString() + "]");
					ps.setInt(idx, Integer.parseInt(value.toString()));
					break;
				}

				default: {
					throw new RuntimeException("UNKNOWN TYPE:" + String.valueOf(param.getType()));
				}
			}
			idx++;
		}
	}

	public int executeUpdate(Map map) throws java.sql.SQLException {
		try {
			EMS_CONNECTION.setIdleLimitTime(600);
			setParameter(map);
			return ps.executeUpdate();
		}
		catch(java.sql.SQLException e) {
			throw e;
		}
		finally {
			EMS_CONNECTION.releaseIdleLimitTime();
		}
	}
	
	public int executeUpdateLong(Map map) throws java.sql.SQLException {
		try {
			EMS_CONNECTION.setIdleLimitTime(600);
			setParameterLong(map);
			return ps.executeUpdate();
		}
		catch(java.sql.SQLException e) {
			throw e;
		}
		finally {
			EMS_CONNECTION.releaseIdleLimitTime();
		}
	}
	
	public void addBatch(Map map) throws java.sql.SQLException {
		try {
			EMS_CONNECTION.setIdleLimitTime(600);
			setParameter(map);
			ps.addBatch();
		} catch (java.sql.SQLException e) {
			throw e;
		} finally {
			EMS_CONNECTION.releaseIdleLimitTime();
		}
	}
	
	public void addBatchLong(Map map) throws java.sql.SQLException {
		try {
			EMS_CONNECTION.setIdleLimitTime(600);
			setParameterLong(map);
			ps.addBatch();
		} catch (java.sql.SQLException e) {
			throw e;
		} finally {
			EMS_CONNECTION.releaseIdleLimitTime();
		}
	}
	public int[] executeBatch() throws java.sql.SQLException {
		try {
			return ps.executeBatch();
		} catch (java.sql.SQLException e) {
			throw e;
		} finally {
			EMS_CONNECTION.releaseIdleLimitTime();
		}
	}
	
	public int executeUpdateMMS(Map map) throws java.sql.SQLException {
		try {
			EMS_CONNECTION.setIdleLimitTime(600);
			setParameterMMS(map);
			return ps.executeUpdate();
		}
		catch(java.sql.SQLException e) {
			throw e;
		}
		finally {
			EMS_CONNECTION.releaseIdleLimitTime();
		}
	}

	public eMsResultSet executeQuery(Map map) throws java.sql.SQLException {
		try {
			EMS_CONNECTION.setIdleLimitTime(600);
			setParameter(map);
			return new eMsResultSet(EMS_CONNECTION, ps.executeQuery());
		}
		catch(java.sql.SQLException e) {
			throw e;
		}
		finally {
			EMS_CONNECTION.releaseIdleLimitTime();
		}
	}
	
	public eMsResultSet executeQuery(Map map, String dbType) throws java.sql.SQLException {
		try {
			EMS_CONNECTION.setIdleLimitTime(600);
			setParameter(map);
			return new eMsResultSet(EMS_CONNECTION, ps.executeQuery(), dbType);
		}
		catch(java.sql.SQLException e) {
			throw e;
		}
		finally {
			EMS_CONNECTION.releaseIdleLimitTime();
		}
	}

	public String toString() {
		return "ORIGINAL=>" + ORIGINAL_QUERY + "\r\n" + "CONVERT_QUERY=>" + CONVERT_QUERY + "\r\n" + PARAMETER_INFOS.toString();
	}

	public boolean check(String query) {
		/**
		 * PS는 점검하지 않기 때문에 ... 근냥 true를 반환
		 */
		return true;
	}

	public boolean checkResultSet(Map map) throws java.sql.SQLException {
		ResultSet rs = null;
		try {
			EMS_CONNECTION.setIdleLimitTime(600);
			setParameter(map);
			rs = ps.executeQuery();
			if( rs.next() ) {
				return true;
			}
			
			return false;
		}
		catch(java.sql.SQLException e) {
			throw e;
		}
		finally {
			if( rs != null ) {
				try {
					rs.close();
				}
				catch(Exception e) {
					// empty
				}
			}
			EMS_CONNECTION.releaseIdleLimitTime();
		}
	}

	static class PreparedParameter {
		short	type;

		String	index;

		PreparedParameter(short t, String i) {
			type = t;
			index = i;
		}

		public short getType() {
			return type;
		}

		public String getIndex() {
			return index;
		}

		public String toString() {
			switch (type) {
				case STRING_TYPE_PARAMETER: {
					return "STRING_TYPE_PARAMETER:".concat(index);
				}
				case NUMERIC_TYPE_PARAMETER: {
					return "NUMERIC_TYPE_PARAMETER:".concat(index);
				}
				default: {
					return "UNKNOWN TYPE";
				}
			}
		}
	}
	public void setFetchSize(int rows) throws java.sql.SQLException {
		ps.setFetchSize(rows);
	}
}
