/*
 * eMsConnection.java
 *
 * Created on 2003년 3월 11일 화, 오후 1:22
 */

package pluto.db;

import java.io.File;
import java.io.FileInputStream;
import java.lang.reflect.Method;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Vector;

import lombok.extern.slf4j.Slf4j;
import pluto.lang.Name;
import pluto.lang.eMsLocale;
import pluto.util.Cal;
import pluto.util.KeyValueEntry;
import pluto.util.StringConvertUtil;
//import amail.ems.batch.target.model.LogPreparedStatement;

/**
 * 데이터 베이스에 접근하는 일반적인 방법을 정의한다. <br>
 * 1. Statement의 생성 <br>
 * 2. close 시에 생성되었던 Statement정리 <br>
 * 3. recycling <br>
 * 단 실제로 연결하는 부분을 추상화하여 연결방법을 다양하게 구현할수 있다.
 * 
 * @author 이상근
 * @version 3.0
 */
@Slf4j
public abstract class eMsConnection extends Name {

	/**
	 * 기본적인 구현 Class를 지정한다.
	 */
	public static final String					DEFAULT_CLASS_NAME				= "pluto.common.db.NotPooledConnection";

	/**
	 * 기본적으로 주어지는 쿼리 타임아웃이다. (30분)
	 */
	public static final int						__DEFAULT_QUERY_TIMEOUT__		= 30 * 60 * 1000;

	public static final int						__CONNECTION_INDEX_TYPE__		= 1;

	public static final int						__CONNECTION_INFORMATION_TYPE__	= 2;

	public static final int						__MAXIMUM_RECYCLE_STATEMENT__	= 5;

	public static int							GET_CONNECTION_TYPE				= __CONNECTION_INFORMATION_TYPE__;

	public static String						GET_CONNECTION_INDEX			= "CONNECTION_INDEX";

	// 기존 InfoConvertor관련 변수들
	/**
	 * ConnectInfo 리스트
	 */
	protected static Hashtable					info							= new Hashtable();

	private static String						TargetFileName					= null;

	private static File							TargetFile						= null;

	private static Class						SUB_IMPLEMENTS_CLASS			= null;

	private static Class						BASIC_IMPLEMENTS_CLASS			= null;

	private static Vector						DRIVER_NAMES					= new Vector();

	protected static boolean					STATEMENT_RECYCLING_FLAG		= true;

	protected static pluto.util.AmailHashtable	MONITORED_TARGET_CONNECTIONS	= new pluto.util.AmailHashtable();

	protected static ConnectionMonitor			INNER_CONNECTION_MONITOR		= null;

	protected static boolean					CONNECTION_MONITOR_PRESENT		= false;

	private static final String  oracle_class = "class oracle.jdbc.driver.OraclePreparedStatementWrapper";
	//oracle jdbc 상위버전에서 오류사항 제거하기 위한 변수 선언.
	
	static {
		try {
			SUB_IMPLEMENTS_CLASS = Class.forName(DEFAULT_CLASS_NAME);
			BASIC_IMPLEMENTS_CLASS = Class.forName(DEFAULT_CLASS_NAME);
		} catch(Exception ignore) {
			log.error("query init errror", ignore);
		}
	}

	/**
	 * 초기화 한다. <br>
	 * 
	 * <PRE>
	 * 
	 * 설정사항 <br>
	 * <br>
	 * &lt;TARGET name="eMsConnection"&gt; &nbsp;&nbsp;&nbsp;&nbsp;&lt;class name="pluto.db.eMsConnection"/&gt; &nbsp;&nbsp;&nbsp;&nbsp;&lt;var name="info.file" value="[데이터 베이스 설정 파일 경로]"/&gt;
	 * &lt;/TARGET&gt;
	 * 
	 * </PRE>
	 * 
	 * 추가적은 name / value 는 Property로 변환하여 구현 eMsConnection의 init( Object ) method를 invoke한다.
	 * 
	 * @param tmp
	 *        초기화 파라미터
	 * @throws Exception
	 *         파일이 없거나 기타 에러시
	 */
	public static void init(Object tmp) throws Exception {
		Properties prop = (Properties) tmp;

		/**
		 * out_db.properties에서 정의된 설정 index로 외부 데이터 베이스를 가져올 것인가? 아니면 002 테이블에 모든 정보를 정의할 것인가를 지정한다.
		 */
		String getConnectionType = prop.getProperty("get.connection.type", "info");

		if( getConnectionType.equalsIgnoreCase("index") ) {
			GET_CONNECTION_TYPE = __CONNECTION_INDEX_TYPE__;
			GET_CONNECTION_INDEX = prop.getProperty("get.connection.index", "CONNECTION_INDEX");
		}
		else {
			GET_CONNECTION_TYPE = __CONNECTION_INFORMATION_TYPE__;
		}

		/**
		 * 커넥션 정보파일을 읽어들인다.
		 */
		TargetFileName = prop.getProperty("info.file");

		if( TargetFileName == null )
			return;

		TargetFile = new File(TargetFileName);

		info.clear();

		Properties tmpinfo = new Properties();

		tmpinfo.load(new FileInputStream(TargetFile));

		// 각 키를 읽어 ConnectInfo를 생성한다.
		for (Enumeration eNum = tmpinfo.keys(); eNum.hasMoreElements();) {
			String key = (String) eNum.nextElement();

			if( key.endsWith(".name") ) {
				String id = key.substring(0, key.lastIndexOf("."));

				/**
				 * 드라이버를 로드한다. 한번만 로드하기 위해서 names에 등록한다.
				 */
				String driver = tmpinfo.getProperty(id + ".driver");

				registDriver(driver);

				ConnectInfo tmpInfo = new ConnectInfo();

				tmpInfo.setDB_ID(id);
				tmpInfo.setDRIVER(driver);
				tmpInfo.setDB_NAME(tmpinfo.getProperty(key));
				tmpInfo.setDB_BASE_CHARSET(tmpinfo.getProperty(id + ".base.charset", eMsLocale.CHAR_SET));
				tmpInfo.setDB_IN_CHARSET(tmpinfo.getProperty(id + ".in.charset", eMsLocale.CHAR_SET));
				tmpInfo.setDB_OUT_CHARSET(tmpinfo.getProperty(id + ".out.charset", eMsLocale.CHAR_SET));
				tmpInfo.setDB_PASS(tmpinfo.getProperty(id + ".pass"));
				tmpInfo.setDB_UID(tmpinfo.getProperty(id + ".id"));
				tmpInfo.setDB_URL(tmpinfo.getProperty(id + ".url"));
				tmpInfo.setDB_INIT_QUERY(tmpinfo.getProperty(id + ".default"));

				if (log.isDebugEnabled()) {
					log.debug(id + ":base:" + tmpInfo.getDB_BASE_CHARSET());
					log.debug(id + ":in:" + tmpInfo.getDB_IN_CHARSET());
					log.debug(id + ":out:" + tmpInfo.getDB_OUT_CHARSET());
				}

				/**
				 * 케릭터 셋을 검증한다.
				 */
				key.getBytes(tmpInfo.getDB_BASE_CHARSET());
				key.getBytes(tmpInfo.getDB_IN_CHARSET());
				key.getBytes(tmpInfo.getDB_OUT_CHARSET());

				info.put(id, tmpInfo);
			}
		}

		if( !info.containsKey("ems") ) {
			throw new RuntimeException("EMS Connection info is not set!!!!");
		}

		if( prop.getProperty("statement.recycle.flag", "false").equalsIgnoreCase("true") ) {
			STATEMENT_RECYCLING_FLAG = true;
		}
		else {
			STATEMENT_RECYCLING_FLAG = false;
		}

		String sub_class = prop.getProperty("connection.class", DEFAULT_CLASS_NAME);

		/**
		 * 구현 글래스를 지정하고 init를 실행한다.
		 */
		SUB_IMPLEMENTS_CLASS = Class.forName(sub_class);

		Method TargetMethod = null;
		Class[] method_args = new Class[1];
		method_args[0] = Object.class;

		Object[] args = new Object[1];
		args[0] = tmp;

		TargetMethod = SUB_IMPLEMENTS_CLASS.getDeclaredMethod("init", method_args);
		TargetMethod.invoke(null, args);

		/**
		 * 커넥션 모니터링 하는 쓰레드를 생성한다.
		 */
		//if (log.isDebugEnabled())  log.debug( prop.getProperty(
		// "connection.monitor.init" , "false" ) );
		if( prop.getProperty("connection.monitor.init", "false").equalsIgnoreCase("true") ) {
			INNER_CONNECTION_MONITOR = new eMsConnection.ConnectionMonitor();
			INNER_CONNECTION_MONITOR.start();
			CONNECTION_MONITOR_PRESENT = true;
			if (log.isDebugEnabled()) 
				log.debug("INIT ConnectionMonitor");
		}
	}

	public synchronized static void unload() throws Exception {
		INNER_CONNECTION_MONITOR.close();
	}

	public static boolean isMonitorPresent() {
		//if (log.isDebugEnabled())  log.debug( "INIT ConnectionMonitor=>".concat(
		// CONNECTION_MONITOR_PRESENT ? "true":"false" ) );
		return CONNECTION_MONITOR_PRESENT;
	}

	// 기존 Connection 관련 변수들
	/**
	 * 실제 DB Connection
	 */
	protected Connection			conn					= null;

	/**
	 * Statement를 등록
	 */
	protected pluto.util.FIFOBuffer	RECYCLE_STATMENTS		= new pluto.util.FIFOBuffer(__MAXIMUM_RECYCLE_STATEMENT__);

	/**
	 * Statement를 등록
	 */
	protected pluto.util.FIFOBuffer	INSTANT_STATMENTS		= new pluto.util.FIFOBuffer(-1);

	protected String				name					= null;

	protected Object				lock					= null;

	protected static Object			__MONITOR_LOCK__		= new Object();

	/**
	 * 연결정보
	 */
	protected ConnectInfo			CONNECT_INFO			= null;

	// 한글변환 관련 변수들
	private boolean					db_in_convert			= false;

	private boolean					db_out_convert			= false;

	private String					db_base_charset			= "KSC5601";

	private String					db_in_charset			= "KSC5601";

	private String					db_out_charset			= "KSC5601";

	private String					BASIC_CONNECTION_NAME	= null;

	/**
	 * Create Connection Instance
	 */
	protected eMsConnection() {
		// empty
	}
	
	public Connection getPureConnection() {
		return this.conn;
	}

	public void setName(String name) {
		if( BASIC_CONNECTION_NAME == null ) {
			BASIC_CONNECTION_NAME = name.toString();
			super.setName(BASIC_CONNECTION_NAME);
		}
		else {
			super.setName(BASIC_CONNECTION_NAME.concat(":").concat(name));
		}
	}

	static class ConnectionMonitor extends pluto.schedule.ScheduledMonitor {
		Object[]	__TARGET_CONNECTIONS__	= null;

		ConnectionMonitor() {
			super("eMsConnection inner Monitor");
			super.setName("eMsConnection inner Monitor");
			__TARGET_CONNECTIONS__ = new Object[10];
		}

		/**
		 * 상속받은 Class에서 실제적으로 모니터링하는 로직
		 */
		protected void check() throws Exception {
			if (log.isDebugEnabled()) 
				log.debug("CHECK IDLE CONNECTION");
			for (Iterator iter = MONITORED_TARGET_CONNECTIONS.iterator(); iter.hasNext();) {
				KeyValueEntry value = (KeyValueEntry) iter.next();
				eMsConnection conn = (eMsConnection) value.getKey();
				String name = (String) value.getValue();
				if (log.isDebugEnabled()) 
					log.debug("CHECK " + name);
				if( conn.isIdleConnection() ) {
					log.error("FOUND IDLE DB CONNECTION {}", conn.getName() + name);
					try {
						conn.interruptConnection();
					} catch(Exception e) {
						log.error("interrupt connection error", e);
					}
					log.error("CLOSE IDLE DB CONNECTION DONE... :{}", conn.getName() + name);
				}
				else {
					if (log.isDebugEnabled()) 
						log.debug("CHECK " + name + " OK");
				}
			}
		}
	}

	public static final void registDriver(String driver) throws SQLException {
		if( !DRIVER_NAMES.contains(driver) ) {
			DRIVER_NAMES.addElement(driver);
			try {
				Class.forName(driver);
			} catch(Exception e) {
				throw new SQLException("Driver Class Not Fount in ClassPath");
			}
		}
	}

	/**
	 * 등록된 ConnectInfo Clone Instance를 얻는다.
	 * 
	 * @return ConnectInfo Instance
	 * @param index
	 *        정보파일에 명시된 Connection Index
	 * @throws Exception
	 *         생성에러
	 */
	public static ConnectInfo getConnectInfo(String index) throws Exception {
		ConnectInfo returnValue = (ConnectInfo) info.get(index);

		if( returnValue == null )
			throw new Exception("Not Regist DB Index : " + index);

		return returnValue.getClone();
	}

	/**
	 * ConnectionPool에서만 호출이 가능하다.
	 * 
	 * @throws Exception
	 *         생성에러
	 * @return eMsConnection Instance
	 */
	public static eMsConnection getInstance() throws Exception {
		eMsConnection returnValue = (eMsConnection) SUB_IMPLEMENTS_CLASS.newInstance();

		//MONITORED_TARGET_CONNECTIONS.put(returnValue, " CREATE AT ".concat(Cal.getDate()));

		return returnValue;
	}

	/**
	 * ConnectionPool에서만 호출이 가능하다.
	 * 
	 * @throws Exception
	 *         생성에러
	 * @return eMsConnection Instance
	 */
	public static eMsConnection getBasicInstance() throws Exception {
		return (eMsConnection) BASIC_IMPLEMENTS_CLASS.newInstance();
	}

	// 5.0에 없어서 추가함. kckim 2005Sep01
	public static final void releaseConnection(eMsConnection con) {
		if( con != null ) {
			con.close();
		}
	}

	/**
	 * EMS내부 ConnectInfo로 execute( ConnectInfo ) 를 invoke한다.
	 * 
	 * @throws SQLException
	 *         생성시 에러
	 * @see pluto.db.eMsConnection#execute( ConnectInfo )
	 */
	public void execute() throws SQLException {
		execute((ConnectInfo) info.get("ems"));
	}

	/**
	 * 지정된 ConnectInfo와 Default 이름으로 execute( ConnectInfo , String ) 를 invoke
	 * 
	 * @param tmpInfo
	 *        연결정보
	 * @throws SQLException
	 *         연결에러
	 * @see pluto.db.eMsConnection#execute( ConnectInfo , String )
	 */
	public void execute(ConnectInfo info) throws SQLException {
		this.CONNECT_INFO = info;

		this.lock = new Object();

		/**
		 * Driver 등록
		 */
		registDriver(this.CONNECT_INFO.getDRIVER());

		/**
		 * 케릭터 셋 변환을 세팅한다.
		 */
		this.db_base_charset = this.CONNECT_INFO.getDB_BASE_CHARSET();
		this.db_in_charset = this.CONNECT_INFO.getDB_IN_CHARSET();
		this.db_out_charset = this.CONNECT_INFO.getDB_OUT_CHARSET();

		if( this.db_base_charset.equals(this.db_in_charset) ) {
			this.db_in_convert = false;
		}
		else {
			this.db_in_convert = true;
		}

		if( this.db_base_charset.equals(this.db_out_charset) ) {
			this.db_out_convert = false;
		}
		else {
			this.db_out_convert = true;
		}

		ensureOpen();
	}

	/**
	 * 데이터 베이스로 String 전송할때 케릭터셋 변환
	 * 
	 * @param src
	 *        변환할 String
	 * @return 변환된 String
	 */
	public String encode(String src) {
		if( src == null ) {
			return "";
		}

		if( !this.db_in_convert ) {
			return src;
		}

		try {
			return new String(src.getBytes(this.db_in_charset), this.db_base_charset);
		} catch(Exception e) {
			return src;
		}
	}

	/**
	 * 데이터 베이스로부터 String 전송 받을때 케릭터셋 변환
	 * 
	 * @param src
	 *        변환할 String
	 * @return 변환된 String
	 */
	public String decode(String src) {
		if( src == null )
			return "";

		if( !this.db_out_convert )
			return src;

		try {
			return new String(src.getBytes(this.db_base_charset), this.db_out_charset);
		} catch(Exception e) {
			return src;
		}
	}

	/**
	 * 데이터 베이스로부터 String 전송 받을때 케릭터셋 변환
	 * 
	 * @param src
	 *        변환할 String
	 * @return 변환된 String
	 */
	private static final byte[]	__NULL_RETURN_BYTE_ARRAY__	= " ".getBytes();

	public byte[] decodeByteArray(String src) {
		if( src == null )
			return __NULL_RETURN_BYTE_ARRAY__;

		try {
			return src.getBytes(this.db_out_charset);
		} catch(Exception e) {
			return src.getBytes();
		}
	}

	public final String getIN_CHARSET() {
		return this.db_in_charset;
	}

	public final String getOUT_CHARSET() {
		return this.db_out_charset;
	}

	public final String getBASE_CHARSET() {
		return this.db_base_charset;
	}

	/**
	 * Connection이름을 반환한다.
	 * 
	 * @return Connection이름
	 */
	public String toString() {
		return getName();
	}

	/**
	 * 연결에 오류가 있는가에 대한 것을 체크할수 있도록 추가
	 * 
	 * @return true : 연결종료 <br>
	 *         false : 연결유지
	 */
	public synchronized boolean isClosed() {
		try {
			return this.conn.isClosed();
		} catch(Exception e) {
			/**
			 * 확실하게 죽인다
			 */
			clean();
			try {
				if(this.conn != null)
					this.conn.close();
			} catch(Exception ex) { }

																					return true;
		}
	}

	/**
	 * 커넥션을 정리한다.
	 */
	public synchronized void releaseConnection() {
		if (log.isDebugEnabled()) 
			log.debug("releaseConnection");
		close();
	}

	public PreparedStatement prepareStatement(String sql) throws SQLException {
		PreparedStatement returnValue = this.conn.prepareStatement(sql);
		this.INSTANT_STATMENTS.push(returnValue);
		return returnValue;
	}

	/*public LogPreparedStatement prepareStatementViewLog(String sql) throws SQLException {
		LogPreparedStatement returnValue = new LogPreparedStatement(this.conn, sql);
		this.INSTANT_STATMENTS.push(returnValue);
		return returnValue;
	}*/
	/**
	 * Statement와는 달리 PreparedStatement는 sql에 오류가 있을 경우에도 Exception을 발생하기 때문에 생성시 Excepion이 발생한다면 일단 Statement를 생성해보고 그래도 Exception이 발생을 한다면 Connection 오류로 판정 reconnect를 한다.
	 * 
	 * @param sql
	 *        생성할 SQL
	 * @throws SQLException
	 *         생성시 에러
	 * @return 생성된 PreparedStatement
	 */
	public eMsPreparedStatement prepareStatement(String sql, String start, String end) throws Exception {
		eMsPreparedStatement returnValue = null;

		try {
			returnValue = new eMsPreparedStatement(sql, start, end);
			returnValue.connectTo(this);
		} catch(Exception e) {
			/* Statement를 생성해본다. */
			try {
				Statement stmt = this.conn.createStatement();
				stmt.close();

				/**
				 * 여기까지 진행이 된다면 커넥션에는 문제가 없으므로 초기 발생한 Exception을 위로 던진다.
				 */
				throw new SQLException(e.toString());
			} catch(SQLException ex) {
				/**
				 * Statement도 생성이 도지 않는다. 그러므로 reconnect를 한다.
				 */
				reconnect();

				returnValue = new eMsPreparedStatement(sql, start, end);
				returnValue.connectTo(this);
			}
		}


		this.INSTANT_STATMENTS.push(returnValue);

		return returnValue;
	}

	/**
	 * Statement와는 달리 CallableStatement는 sql에 오류가 있을 경우에도 Exception을 발생하기 때문에 생성시 Excepion이 발생한다면 일단 Statement를 생성해보고 그래도 Exception이 발생을 한다면 Connection 오류로 판정 reconnect를 한다.
	 * 
	 * @param sql
	 *        생성할 SQL
	 * @throws SQLException
	 *         생성시 에러
	 * @return 생성된 CallableStatement
	 */
	public CallableStatement prepareCall(String sql) throws SQLException {
		CallableStatement returnValue = null;

		try {
			returnValue = this.conn.prepareCall(encode(sql));
		}
		catch(Exception e) {
			/* Statement를 생성해본다. */
			try {
				Statement stmt = this.conn.createStatement();
				stmt.close();

				/*
				 * 여기까지 진행이 된다면 커넥션에는 문제가 없으므로 초기 발생한 Exception을 위로 던진다.
				 */
				throw new SQLException(e.toString());
			}
			catch(Exception ex) {
				/*
				 * Statement도 생성이 도지 않는다. 그러므로 reconnect를 한다.
				 */
				reconnect();

				returnValue = this.conn.prepareCall(sql);
			}
		}

		this.INSTANT_STATMENTS.push(returnValue);

		return returnValue;
	}

	protected synchronized Statement createInnerStatement() throws SQLException {
		return this.conn.createStatement();
	}

	/**
	 * 만일 스테이트 먼트를 생성하면서 에러가 발생을 한다면 커넥션에 문제가 있는것으로 간주를 하고 reconnect를 호출 재접속한다.
	 * 
	 * @throws SQLException
	 *         생성시 에러
	 * @return 생성된 Statement
	 */
	public synchronized eMsStatement createStatement() throws SQLException {

		Object tmp = this.RECYCLE_STATMENTS.pop();
		if( tmp != null ) {
			return (eMsStatement) tmp;
		}

		eMsStatement returnValue = new eMsStatement();
		returnValue.connectTo(this);
		return returnValue;
	}

	/**
	 * 생성된 Statement를 정리한다.
	 */
	public void clean() {
		if (log.isDebugEnabled()) 
			log.debug(" ==> cleaning Opened Statement [START]");

		cleanInstantStatements();

		if (log.isDebugEnabled()) 
			log.debug(" ==> cleaning Opened Statement [END]");
	}

	protected synchronized void cleanInstantStatements() {
		Object tmp = null;
		while ((tmp = this.INSTANT_STATMENTS.pop()) != null) {
			//	kckim 2005Jun13 다양한 xxxStatmetnt 인스턴스를 받아들여 close() 하기 위해서 아래와 같이
			// 수정함.
			//	(원래 SMS Receiver모듈에서 Casting 문제발생 해서 수정)
			Class tmpClass = tmp.getClass();

			Method tmpMethod = null;

			try {//jdbc driver patch.
				if( oracle_class.equals(tmpClass.toString()) ){
					
					PreparedStatement returnValue  = (PreparedStatement) tmp;
					returnValue.close();
					
				}else{
					tmpMethod = tmpClass.getMethod("close", null);
					tmpMethod.invoke(tmp, null);
				}
			}
			catch(Exception e) {
				log.error("clean instance error", e);
			}
		}
	}

	/*
	 * kckim 2005Jun13 혹시 만일을 위해서 기존에 있던 cleanInstantStatements()함수를 남겨둠... protected synchronized void cleanInstantStatements(){ Object tmp = null; while( ( tmp = this.INSTANT_STATMENTS.pop() ) !=
	 * null ) { eMsStatementInterface target = (eMsStatementInterface)tmp;
	 * 
	 * try{ target.close(); }catch( Exception e ){}
	 * 
	 */
	protected synchronized void cleanRecycleStatements() {
		Object tmp = null;
		while ((tmp = this.RECYCLE_STATMENTS.pop()) != null) {
			eMsStatementInterface target = (eMsStatementInterface) tmp;

			try {
				target.close();
			}
			catch(Exception e) {
				log.error("clean recycle error", e);
			}

			if (log.isDebugEnabled()) 
				log.debug(" ==> cleaning -> " + target.toString());
		}
	}

	public synchronized void recycleStatement(eMsStatementInterface stmt) {
		/**
		 * null은 재활용할수 없다.
		 */
		if( stmt == null )
			return;

		stmt.close();
	}

	/**
	 * 연결을 재설정한다.
	 * 
	 * @throws SQLException
	 *         재설정시 에러
	 */
	public synchronized void reconnect() throws SQLException {
		if (log.isDebugEnabled()) 
			log.debug("reconnect");
		releaseConnection();

		ensureOpen();
	}

	/**
	 * 재활용 되지 않고 삭제 될때 자원 정리
	 */
	public void destroy() {
		if (log.isDebugEnabled()) {
			log.debug("INTO DESTROY");
		}
		this.lock = null;
		close();
	}

	/**
	 * 사용을 마친다. 재활용 일경우에는 이 method를 override하면 된다.
	 */
	public synchronized void recycle() {
		if (log.isDebugEnabled()) {
			log.debug("INTO RECYCLE");
		}
		close();
	}

	/**
	 * 사용을 마친다.
	 */
	public void close() {
		if (log.isDebugEnabled()) {
			log.debug("INTO CLOSE");
		}

		this.cleanInstantStatements();

		this.cleanRecycleStatements();

		try {
			if(this.conn != null){
				this.conn.close();
				this.conn = null;
			}
		}
		catch(Exception e) { }

		/**
		 * 닫으면서
		 */
		if( CONNECTION_MONITOR_PRESENT ) {
			releaseIdleMonitor();
		}
	}

	/**
	 * 데이터베이스를 사용하는 녀석들은 리스팅 쿼리를 일괄적으로 업데이트 하도록 한다.
	 */
	public boolean executeUpdateList(List __QUERY_LIST__, Map __INFO__, String __START__, String __END__, boolean __CHECK_FAIL__) throws Exception {
		eMsStatement __UPDATE_STATEMENT__ = null;
		List innerList = __QUERY_LIST__;
		// 반복처리이므로 eMsStringBuffer를 활용하여 하는 걸로 수정
		// 이 자체를 반복 루틴에서 사용하는 녀석이 있다면 로직을 주정해야한다.
		StringBuffer buffer = null;
		try {
			__UPDATE_STATEMENT__ = createStatement();
			buffer = new StringBuffer(1024);
			synchronized (innerList) {
				for (Iterator iter = innerList.iterator(); iter.hasNext();) {
					String query = (String) iter.next();
					buffer.setLength(0);
					StringConvertUtil.ConvertString(buffer, query, __INFO__, __START__, __END__, true, false);
					if (log.isDebugEnabled()) 
						log.debug(buffer.toString());
					if( __UPDATE_STATEMENT__.executeUpdate(buffer.toString()) < 1 && __CHECK_FAIL__ )
						return false;
				}
			}
		}
		finally {
			if( __UPDATE_STATEMENT__ != null )
				recycleStatement(__UPDATE_STATEMENT__);

			buffer = null;
		}

		return true;
	}

	public int executeUpdate(String query) throws SQLException {
		eMsStatement __UPDATE_STATEMENT__ = null;

		try {
			__UPDATE_STATEMENT__ = createStatement();
			return __UPDATE_STATEMENT__.executeUpdate(query);
		}
		finally {
			if( __UPDATE_STATEMENT__ != null )
				recycleStatement(__UPDATE_STATEMENT__);
		}
	}

	/**
	 * 모니터링 할때 기준이 되는 시간
	 */
	private long	IDLE_CHOICE_TIME	= -1;

	//private boolean IS_IN_THE_MONITOR = false;

	protected final boolean isIdleConnection() {
		if (log.isDebugEnabled()) 
			log.debug("TARGET DATE" + Cal.getDate(this.IDLE_CHOICE_TIME));
		if (log.isDebugEnabled()) 
			log.debug("NOW DATE" + Cal.getDate());
		return this.IDLE_CHOICE_TIME > 0 && this.IDLE_CHOICE_TIME < System.currentTimeMillis();
	}

	protected final void releaseIdleLimitTime() {
		this.IDLE_CHOICE_TIME = -1L;
	}

	protected final void setIdleLimitTime(int sec) {
		this.IDLE_CHOICE_TIME = (sec > 0) ? (System.currentTimeMillis() + sec * 1000L) : -1;
	}

	protected final void releaseIdleMonitor() {
		this.IDLE_CHOICE_TIME = -1;

		synchronized (MONITORED_TARGET_CONNECTIONS) {
			if (log.isDebugEnabled()) 
				log.debug("EXIT MONITOR");

			MONITORED_TARGET_CONNECTIONS.remove(this);
		}
	}

	public final void interruptConnection() {
		try {
			if(this.conn != null)
				this.conn.close();
		}
		catch(Exception e) {
			log.error("INTERRUPT CONNECTION ERROR", e);
		}
	}

	protected void log(String logStr) {
		log.info("log :{}", logStr);
	}

	/**
	 * 세팅된 설정으로 실제 Connection을 하거나 / 점검한다.
	 * 
	 * @throws SQLException
	 *         점검시 에러
	 */
	public abstract void ensureOpen() throws SQLException;

	/**
	 * 연결의 유효성을 검사한다.
	 * 
	 * @return true : 파기대상 <br>
	 *         false : 파기하면 안됨
	 */
	public abstract boolean expire();

	public void setAutoCommit(boolean autoCommit) throws SQLException {
		this.conn.setAutoCommit(autoCommit);
	}

	public void commit() throws SQLException {
		this.conn.commit();
	}
	
	public void rollback() throws SQLException {
		this.conn.rollback();
	}

	/**
	 * 데이터베이스를 사용하는 녀석들은 리스팅 쿼리를 일괄적으로 업데이트 하도록 한다.
	 */
	public boolean executeUpdate(List __QUERY_LIST__, Map __INFO__, String __START__, String __END__, boolean __CHECK_FAIL__) throws Exception {
		eMsStatement __UPDATE_STATEMENT__ = null;

		try {
			__UPDATE_STATEMENT__ = createStatement();

			for (Iterator iter = __QUERY_LIST__.iterator(); iter.hasNext();) {
				String query = (String) iter.next();
				if( __UPDATE_STATEMENT__.executeUpdate(query, __INFO__, __START__, __END__) < 1 && __CHECK_FAIL__ )
					return false;
			}
		}
		catch(Exception e) {
			throw e;
		}
		finally {
			if( __UPDATE_STATEMENT__ != null )
				recycleStatement(__UPDATE_STATEMENT__);
		}

		return true;
	}

	/**
	 * 리스트에 담긴 모든 쿼리를 쭉! 업데이트한다.
	 */
	public void execTargetQueryUpdate(List __QUERY_LIST__) throws Exception {
		executeUpdate(__QUERY_LIST__, null, null, null, false);
	}

	public int execTargetQueryUpdate(String query, Map map, String start, String end) throws Exception {
		eMsStatement stmt = null;
		try {
			stmt = createStatement();
			return stmt.executeUpdate(query, map, start, end);
		}
		catch(Exception e) {
			recycleStatement(stmt);
			throw e;
		}
		finally {
			recycleStatement(stmt);
		}
	}

	public int execTargetQueryUpdate(String query) throws Exception {
		return execTargetQueryUpdate(query, null, null, null);
	}

	public boolean putSingleResultToMap(String query, Map map) throws Exception {
		eMsStatement stmt = null;
		eMsResultSet rs = null;
		try {
			stmt = createStatement();

			rs = stmt.executeQuery(query);
			if( rs.next() ) {
				rs.putToMap(map, false);
				return true;
			}

			return false;
		}
		catch(Exception e) {
			throw e;
		}
		finally {
			if( rs != null )
				rs.close();
			recycleStatement(stmt);
		}
	}
	
	
	/**
	 * 단건의 데이터를 가져올때 Properties로 전달할 수 있는 메소드
	 * @param query
	 * @param map
	 * @param start
	 * @param end
	 * @return
	 * @throws Exception
	 */
	public Properties putSingleResultToMapConvert(String query, Map map, String start, String end ) throws Exception {
		StringBuffer buffer = null;
		eMsStatement stmt = null;
		eMsResultSet rs = null;
		Properties prop = null;
		
		try {
			prop = new Properties();
			buffer = new StringBuffer(512);
			StringConvertUtil.ConvertString(buffer, query, map, start, end, true, false);
			
			stmt = createStatement();

			rs = stmt.executeQuery(buffer.toString());
			if( rs.next() ) {
				rs.putToMap(prop, false);
			}
		}
		catch(Exception e) {
			throw e;
		}
		finally {
			if( rs != null )
				rs.close();
			recycleStatement(stmt);
		}
		
		return prop;
	}
	
	/**
	 * 작은 건수의 리스트 데이터를 가져올때 사용한다.
	 * 
	 * (보통 리스트 데이터는 메모리에 전체를 담아둘 수 없기 때문에 흘러가 듯이 처리한다. 
	 *  하지만 특정 코드데이터나 메모리상에 저장해야할 리스트 데이터는 아래 메소드를 사용해서 
	 *  ArrayList에 담아 사용할 수 있다. )
	 * 
	 * [주의] 대량의 데이터를 가져올때는 사용하지 말아야한다.
	 *  
	 * @param query
	 * @param map
	 * @param start
	 * @param end
	 * @return
	 * @throws Exception
	 */
	public ArrayList<Properties> putResultToList(String query, Map map, String start, String end) throws Exception {
		StringBuffer buffer = null;

		eMsStatement stmt = null;
		eMsResultSet rs = null;
		Properties prop = null;
		ArrayList<Properties> list = new ArrayList<Properties>();;
		
		try {
			buffer = new StringBuffer(512);
			StringConvertUtil.ConvertString(buffer, query, map, start, end, true, false);
			
			stmt = createStatement();

			rs = stmt.executeQuery(buffer.toString());
			
			while( rs.next() ) {
				prop = new Properties();
				rs.putToMap(prop, false);
				list.add(prop);
			}
		}
		catch(Exception e) {
			throw e;
		}
		finally {
			if( rs != null )
				rs.close();
			recycleStatement(stmt);
		}
		
		return list;
	}
	

	public static boolean checkConnectionAboutException(Exception e) {
		if( e == null )
			return false;

		if( e instanceof SQLException ) {
			SQLException se = (SQLException) e;

			if( se.getErrorCode() == 0 || se.getSQLState() == null ) {
				return true;
			}
		}
		return false;
	}
}
