/*
 * @(#)ZipFileManager.java            2004. 12. 7.
 *
 * Copyright (c) 1998-2004 Amail, Inc.
 * 708-8 Global Building 10th floor, YeokSamdong, Gangnamgu, Seoul, 
 * Korea republic of. All rights reserved.
 * 
 * This software is the confidential and proprietary information of Amail,
 * Inc. ("Confidential Information"). You shall not disclose such 
 * Confidential Information and shall use it only in accordance with
 * the terms of the license agreement you entered into Amail.
 * 
 */
 
 
package pluto.compress;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import lombok.extern.slf4j.Slf4j;
import pluto.io.FileElement;

/**
 * Zip File Controller
 * 
 * @version		
 * @author 		lena
 *
 */
@Slf4j
public class ZipFileManager {
	
	
	/**
	 * 압축 파일인지 여부를 검사한다.
	 * @return
	 */
	public static boolean isCompressed(String file_name) {
		
		if( file_name == null ) return false;
		
		String temp = file_name.toLowerCase();
		
		return ( temp.endsWith(".zip") )?true:false;
	}
	
	/**
	 * Compresses a file. Keeps an original file by default.
	 * @param base
	 * @param fileName
	 * @return 
	 * @throws Exception
	 */
	public static String makeZip(String base, String fileName) throws Exception {
		
		return ZipFileManager.makeZip(base,fileName,true);
	}
	
	/**
	 * @param base
	 * @param fileName
	 * @param keepOriginal keeps original file if true, deletes otherwise
	 * @return Abstract path of result zip file
	 * @throws Exception
	 */
	public static String makeZip(String base, String fileName, boolean keepOriginal) 
			throws Exception {
		
		File f = new File(base + File.separator + fileName);
		String zipFileName =  fileName + ".zip";
		
		try(
			FileInputStream fis = new FileInputStream(f);
			ZipOutputStream zos = new ZipOutputStream( 
					new FileOutputStream( base + File.separator + zipFileName));
		){	
		 
			ZipEntry ze = new ZipEntry(fileName);
			zos.putNextEntry(ze);
			
			byte[] b = new byte[2<<6];
			
			int i;
			
			try {
				while( true ) {
					i = fis.read(b);
					if( i <= 0) break;
					zos.write(b,0,i);
					zos.flush();
				}
			} catch(Exception ex) {
				log.error("Exception", ex);
			}
		}catch(Exception e) {
			log.error(e.getMessage());
		}
		
		if( !keepOriginal ) {
			boolean fileDel = f.delete();
			if(!fileDel) log.error("File deletion failed");
		}
		
		return base + "/" + zipFileName;
	}
	
	/**
	 * 새로운 파일 이름을 특별히 지정하지 않고 zip 파일에 있는 정보를 그대로
	 * 유지한 채 zip 파일의 압축을 푼다. 
	 * @param zipFile
	 * @param destBase
	 * @param keepZipFile
	 * @return
	 * @throws Exception
	 * @see unZip(String zipFile, String destBase, String newFile, boolean keepZipFile)
	 */
	public static String unZip(
			String zipFile, String destBase, 
			boolean keepZipFile) 
			throws Exception {
		
		return ZipFileManager.unZip(zipFile,destBase,null,keepZipFile);
	}
	
	/**
	 * zip file 압축을 푼다. zip file에 element가 한 개 있는 경우 쓸 수 있다. 
	 * @param zipFile 압축을 풀 zip File
	 * @param destBase 압축이 풀릴 디렉토리 
	 * @param newFile 압축을 풀 파일 이름. null이면 zip 파일에 들어가 있는 정보를 그대로 이용한다.
	 * @param keepZipFile true이면 zip file을 놔두고, false이면 삭제한다. 
	 * @return
	 * @throws Exception
	 */
	public static String unZip(
			String zipFile, String destBase, 
			String newFile, boolean keepZipFile) 
			throws Exception {
		
		if( zipFile == null || !zipFile.toLowerCase().endsWith(".zip") ) {
			throw new Exception(" Not a zip file");
		}
		
		InputStream is = null;
		boolean fromHttp = false;
		
		if( zipFile.toLowerCase().startsWith("http://") ) {
			FileElement fe = FileElement.getFileElement(zipFile);
			is = fe.getStream();
			fromHttp = true;
		} else {
			is = new FileInputStream(new File(zipFile)); 
		}
		
		try(
			ZipInputStream zis = new ZipInputStream(is);
			FileOutputStream fos = new FileOutputStream(destBase + File.separator + newFile);
			){
			
			ZipEntry ze = zis.getNextEntry();
			if( newFile == null || newFile.trim().equals("") ) {
				newFile = ze.getName();
			}
				
			byte[] b = new byte[2<<6];
			
			int i;
			
			try {
				while( true ) {
					i = zis.read(b);
					if( i <= 0) break;
					fos.write(b,0,i);
					fos.flush();
				}
			} catch(Exception e) {
				log.error(e.getMessage());
			}
		} catch(Exception ex) {
			log.error("Exception", ex);
		}
		
		if( !keepZipFile && !fromHttp ) {
			File f = new File(zipFile);
			boolean fileDel = f.delete();
			if(!fileDel) log.error("File deletion failed");
		}
		return newFile;
	}
}
