/*
 * @(#)SystemCheckTask.java            2008. 8. 23.
 *
 * Copyright (c) 1998-2008 Amail, Inc.
 * 708-8 Global Building 10th floor, YeokSamdong, Gangnamgu, Seoul, 
 * Korea republic of. All rights reserved.
 * 
 * This software is the confidential and proprietary information of Amail,
 * Inc. ("Confidential Information"). You shall not disclose such 
 * Confidential Information and shall use it only in accordance with
 * the terms of the license agreement you entered into Amail.
 * 
 */

package pluto.common.task;

import java.io.File;
import java.io.FileInputStream;
import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Properties;

import lombok.extern.slf4j.Slf4j;
import pluto.config.SqlManager;
import pluto.db.ConnectionPool;
import pluto.db.eMsConnection;
import pluto.db.eMsStatement;
import pluto.util.Cal;
import pluto.util.StringConvertUtil;

/**
 * Class description :
 * 
 * @version
 * @author jini
 *  
 */
@Slf4j
public class SystemCheckTask extends pluto.schedule.Task {

	
	
	/**
	 * 시스템(HDD,DB)상태를 체크하여 DB에 정보를 UPDATE
	 */
	public static String		QUERY_UPDATE_SYSTEM_SERVER_INFO			= null;
	public static String		QUERY_UPSERT_SYSTEM_SERVER_INFO			= null;
	public static String		QUERY_UPDATE_SYSTEM_DAEMON_INFO			= null;
	public static String		QUERY_UPSERT_SYSTEM_DAEMON_INFO			= null;
	
	public static String		server_id		= "";	
	public static String 		daemon_id		= "";
	
	public static String		daemon_name		= "daemon";
	
	public static String		analysis_type	= "daemon";
	public static String		hdd_path		= "";
	public static String 		db_type			= "";
	public static String 		db_url			= "";
	public static String		db_id			= "";
	public static String		db_pw			= "";
	public static String		db_schema		= "";
	public static String 		db_info			= "";
	
	//단위 : MB
	public static double		max_db			= 1048576;
	
	public static double		warning_db_percent		= 80;
	public static double		warning_hdd_percent		= 80;
	

	/**
	 * 구동에 필요한 기본 static 변수의 초기화
	 */
	static {
		try {
			QUERY_UPSERT_SYSTEM_SERVER_INFO = SqlManager.getQuery("SYSTEM_INFO_CHECK", "QUERY_UPSERT_SYSTEM_SERVER_INFO");
			QUERY_UPDATE_SYSTEM_SERVER_INFO = SqlManager.getQuery("SYSTEM_INFO_CHECK", "QUERY_UPDATE_SYSTEM_SERVER_INFO");
			QUERY_UPSERT_SYSTEM_DAEMON_INFO = SqlManager.getQuery("SYSTEM_INFO_CHECK", "QUERY_UPSERT_SYSTEM_DAEMON_INFO");
			QUERY_UPDATE_SYSTEM_DAEMON_INFO = SqlManager.getQuery("SYSTEM_INFO_CHECK", "QUERY_UPDATE_SYSTEM_DAEMON_INFO");			
		}
		catch(Exception e) {
			log.error(e.getMessage());
			System.exit(1);
		}
	}

	protected eMsConnection		EMS_CONNECTION						= null;

	protected eMsStatement		__SELECT_STATEMENT__				= null;

	/** Creates a new instance of SystemCheckTask */
	public SystemCheckTask() {
		super(TYPE_INTERVAL);
		this.setName("SystemCheckTask_at_".concat(Cal.getSerialDate()));
		this.setTaskID("SystemCheckTask");
	}
	
	// 설정에서 넘기값 세팅
	public static void init(Object tmp) throws Exception {

		Properties prop = (Properties) tmp;

		// server id 구분
		server_id = prop.getProperty("tms.server.id");
		daemon_id = prop.getProperty("tms.daemon.id");
		daemon_name = prop.getProperty("tms.daemon.name");
		
		analysis_type = prop.getProperty("analysis.type", "daemon");
		
		// 분석 타입에 따라서 불피요한 업데이트를 제외한다. 
		if ( canUpdateServerInfo() ) {
			
			db_type = prop.getProperty("db.type");
			hdd_path = prop.getProperty("hdd.path");
			db_info = prop.getProperty("db.info");
			
			File dbInfo = new File(db_info);
			Properties tmpinfo = new Properties();
			tmpinfo.load(new FileInputStream(dbInfo));
			db_url = tmpinfo.getProperty("ems.url");
			db_id = tmpinfo.getProperty("ems.id");
			db_pw = tmpinfo.getProperty("ems.pass");
			db_schema = tmpinfo.getProperty("ems.db.schema");
			
			warning_hdd_percent	= Double.valueOf( prop.getProperty("warning.hdd.percent") );
			
			max_db = Double.valueOf( prop.getProperty("max.db") );
			warning_db_percent	= Double.valueOf( prop.getProperty("warning.db.percent") );
			
		}
		
		if (log.isDebugEnabled()) {
			log.debug("SYSTEM CHECK TASK INIT...");
		}
		
		initDaemonAndServerInfo();
	}
	
	public static boolean canUpdateServerInfo() {
		return "server".equals(analysis_type);
	}
	
	private static void initDaemonAndServerInfo() {
		
		eMsConnection connection = null;
		eMsStatement stmt = null;
		
		try {
			
			Properties prop = new Properties();
			prop.setProperty("SERVER_ID", server_id);
			prop.setProperty("DAEMON_ID", daemon_id);
			prop.setProperty("DAEMON_NAME", daemon_name);
			
			String execDaemonQuery = StringConvertUtil.ConvertString(QUERY_UPSERT_SYSTEM_DAEMON_INFO, prop, "${", "}", true, false);
			
			connection = ConnectionPool.getConnection();
			stmt = connection.createStatement();
			stmt.executeUpdate(execDaemonQuery);
			
			if ( canUpdateServerInfo() ) {
				
				prop.setProperty("SERVER_NAME", server_id);
				
				long totalHdd = GetMaxHddSpace(hdd_path);
				prop.setProperty("MAX_HDD", String.valueOf(totalHdd));
				prop.setProperty("WARNING_HDD", String.valueOf( totalHdd *  warning_hdd_percent / 100.0 ));
				
				prop.setProperty("MAX_DB", String.valueOf(max_db));
				prop.setProperty("WARNING_DB", String.valueOf( max_db * warning_db_percent / 100.0 ) );
				
				prop.setProperty("HDD_PATH", hdd_path);
				prop.setProperty("CURRENT_HDD", String.valueOf(GetFreeHDDSpace(hdd_path)));
				prop.setProperty("CURRENT_DB", String.valueOf(GetDBSpace(db_type, db_url, db_id, db_pw)));
				
				String execServerQuery = StringConvertUtil.ConvertString(QUERY_UPSERT_SYSTEM_SERVER_INFO, prop, "${", "}", true, false);
				stmt.execute(execServerQuery);
			}
			
		} catch (Exception e) {
			log.error("error", e);
		} finally {
			if (stmt != null) {
				stmt.close();
			}
			if ( connection != null ) {
				connection.close();
			}
		}
	}
	

	/**
	 * Task를 초기화하는 로직을 구현한다. Throwable이 발생하게 되면 execute_initiateError() 를 호출하도록
	 * 되어있다.
	 */
	public void execute_initiate() throws Exception {
		this.setName("SystemCheck_at_".concat(Cal.getSerialDate()));
		EMS_CONNECTION = ConnectionPool.getConnection();
		__SELECT_STATEMENT__ = EMS_CONNECTION.createStatement();
	}

	/**
	 * 초기화할때 Throwable이 뛰쳐나왔을때 처리하는 로직을 구현한다. Exception을 절대로 반환하면 안된다. 그러면 ㅠㅠ
	 * 에러처리는 구현안에 모두 포함하여 하는 것을 권장한다.
	 */
	public void execute_initiateError(Throwable thw) {
		log.error(" Request Connection Error");
		log.error(getName(), thw);

		//REPORT 데이터베이스 커넥션을 가져오는 과정에서 에러가 발생하였습니다.
		/*if( !(thw instanceof SQLException) ) {
			Reporter.report("common", getName(), "systemchecktask.init.err", thw);
		}*/
	}

	/**
	 * 에러가 발생하거나 정상적으로 끝나거나 언제나 실행이된다. 할당받은 자원을 free 시키는 로직을 구현한다.
	 * execute_initiateError()와 마찬가지로 Exception을 반환하면 안되고 처리로직을 전체 포함하여야 한다.
	 */
	public void release_Resource() {
		if( EMS_CONNECTION != null ) {
			EMS_CONNECTION.recycleStatement(__SELECT_STATEMENT__);
			EMS_CONNECTION.recycle();
		}
	}

	/**
	 * 실제적인 일을 처리하는 비지니스로직
	 * 
	 * @return true : 성공적인 수행 <br>
	 *         false : 수행 실패
	 */
	public void execute() throws Exception {
		// system check(hdd,db)
		check_system();	
	}
	
	/**
	 * 시스템(DISK,DB)용량을 체크해서 DB에 INSERT한다.
	 *
	 */
	private synchronized void check_system() throws Exception{
		
		Properties SYSTEM_INFO = new Properties();
		SYSTEM_INFO.setProperty("SERVER_ID", server_id);
		SYSTEM_INFO.setProperty("DAEMON_ID", daemon_id);

		// 분석 타입에 따라서 불피요한 업데이틀 제외한다. 
		if ( canUpdateServerInfo() ) {
			log.debug(" is GetFreeHDDSpace method  START...");
			long freeHdd = GetFreeHDDSpace(hdd_path);
			log.debug(" is GetFreeHDDSpace method  END...");
			log.debug(" is GetDBSpace method  START...");
			long freeDb =  GetDBSpace(db_type, db_url, db_id, db_pw);
			log.debug(" is GetDBSpace method  END...");

			SYSTEM_INFO.setProperty("CURRENT_HDD", String.valueOf(freeHdd));
			SYSTEM_INFO.setProperty("CURRENT_DB", String.valueOf(freeDb));

			String execServerQuery = StringConvertUtil.ConvertString(QUERY_UPDATE_SYSTEM_SERVER_INFO, SYSTEM_INFO, "${", "}", true, false);
			log.debug(" is UPDATE SERVER QUERY ..."+ execServerQuery);
			__SELECT_STATEMENT__.executeUpdate(execServerQuery);
		}
		
		String execDaemonQuery = StringConvertUtil.ConvertString(QUERY_UPDATE_SYSTEM_DAEMON_INFO, SYSTEM_INFO, "${", "}", true, false);
		log.debug(" is UPDATE DAEMON QUERY ..."+ execDaemonQuery);
		__SELECT_STATEMENT__.executeUpdate(execDaemonQuery);			
	}
	
	/**
	 * 디스크 남은 크기를 체크한다.
	 * 
	 * 수정. jdk1.6이상에서 메소드 호출로 남은 디스크 사이즈 체크 가능
	 * 
	 * @param path	체크할 경로
	 * @return		남은 디스크 사이즈
	 */
	public static long GetFreeHDDSpace(String path) {

		long freeSpace = 0;
		
		try {
			freeSpace = new File(path).getFreeSpace()/(1024*1024);
		} catch ( Exception e ) {
			log.error("hdd free space error", e);
		}
		
		return freeSpace;
	}
	
	public static long GetMaxHddSpace(String path) {
		
		long freeSpace = 0;
		
		try {
			freeSpace = new File(path).getTotalSpace()/(1024*1024);
		} catch ( Exception e ) {
			log.error("hdd max space error", e);
		}
		
		return freeSpace;
	}
	
	/**
	 * DB 남은 사이즈를 체크한다. (단위 : MB)
	 * @param DB
	 * @param url
	 * @param id
	 * @param pw	
	 * @return		남은 사이즈
	 */
	public static long GetDBSpace(String DB, String url, String id, String pw) {

		long rtn = 0;
	    Connection con = null;
	    CallableStatement cstmt = null;
	    Statement stmt = null;
	    ResultSet rs = null;
	    String query = "";
	    try {
	
	      con = DriverManager.getConnection(url, id, pw);
	
	      if(DB.equalsIgnoreCase("mssql")) {
	
	        cstmt = con.prepareCall("{call sp_spaceused}");
	        rs = cstmt.executeQuery();
	
	        if (rs.next()) {
	          rtn = (int) Float.parseFloat(rs.getString(2).replaceAll("MB", "").replaceAll(" ", "").replaceAll(",", ""));
	        }
	
	      } else if(DB.equalsIgnoreCase("oracle")) {
	
//	    	rtn = GetOracleDBSize(con);
	
	    	/* 수정 2006.11.14 */
	      } else if(DB.equalsIgnoreCase("mysql")) {
	
	    	stmt = con.createStatement();
	
//	    	rs = stmt.executeQuery("show table status");
	    	query += "SELECT SUM(data_length+index_length) used_MB, SUM(data_free) free_MB"
	    		   + "       ,(SUM(data_length+index_length)+SUM(data_free)) total_MB"
	    		   + "  FROM information_schema.tables"
	    		   + " WHERE table_schema = '"+db_schema+"' ";
	    	
	    	rs = stmt.executeQuery(query);
	
	        while(rs.next()) {
	        	//2016.08.29 수정 index 크기도 추가
//	        	rtn += ( rs.getLong("Data_length") + rs.getLong("Index_length") ) ;
	        	rtn += rs.getLong("free_MB");
	        }
	
	        rtn = rtn/1024/1024;
	      }
	    } catch(Exception e) {
	      log.error("DB space error", e);
	    } finally {
	    	if (rs != null) {
	    		try {
	    			rs.close();
	    		} catch (Exception ignore){
	    			
	    		}
	    	}
	    	if (cstmt != null) {
	    		try{
	    			cstmt.close();
	    		} catch (Exception ignore){
	    			
	    		}
	    	}
	    	if (stmt != null) {
	    		try{
	    			stmt.close();
	    		} catch (Exception ignore){
	    		}
	    	}
	    	if (con != null) {
	    		try {
	    			con.close();
	    		} catch (Exception ignore){
	    			
	    		}
	    	}
	    }
	    return rtn;
	}
	
	/**
	 * ORACLE CHECK
	 * @param con
	 * @return
	 */
	public static long GetOracleDBSize(Connection con) {
		if (con == null) {
			return 0;
		}
		long rtn = 0;
		Statement stmt = null;
		ResultSet rs = null;

		try {

			stmt = con.createStatement();
			rs = stmt.executeQuery("select sum(bytes) from user_free_space");
			while(rs.next()) {
				rtn += rs.getLong(1);
			}
			rtn = rtn / 1024 / 1024;

		} catch(Exception e) {
			log.error(e.getMessage());
		} finally {
			if (rs != null) {
				try {
					rs.close();
				}catch(Exception e){}
			}
			if (stmt != null) {
				try {
					stmt.close();
				}catch(Exception e){}
			}
		}
		return rtn;
	}
}