// Copyright (c) 1999-2004 Brian Wellington (bwelling@xbill.org)

package pluto.DNS;

import java.io.IOException;

/**
 * A6 Record - maps a domain name to an IPv6 address (experimental)
 * 
 * @author Brian Wellington
 */

public class A6Record extends Record {

	private int prefixBits;

	private Inet6Address suffix;

	private Name prefix;

	A6Record() {
		// empty
	}

	Record getObject() {
		return new A6Record();
	}

	/**
	 * Creates an A6 Record from the given data
	 * 
	 * @param prefixBits
	 *            The number of bits in the address prefix
	 * @param suffix
	 *            The address suffix
	 * @param prefix
	 *            The name of the prefix
	 */
	public A6Record(Name name, int dclass, long ttl, int prefixBits, Inet6Address suffix, Name prefix) {
		super(name, Type.A6, dclass, ttl);
		this.prefixBits = checkU8("prefixBits", prefixBits);
		this.suffix = suffix;
		if (prefix != null)
			this.prefix = checkName("prefix", prefix);
	}

	void rrFromWire(DNSInput in) throws IOException {
		prefixBits = in.readU8();
		int suffixbits = 128 - prefixBits;
		int suffixbytes = (suffixbits + 7) / 8;
		if (prefixBits < 128) {
			suffix = new Inet6Address(128 - prefixBits, in.readByteArray(suffixbytes));
		}
		if (prefixBits > 0)
			prefix = new Name(in);
	}

	void rdataFromString(Tokenizer st, Name origin) throws IOException {
		prefixBits = st.getUInt8();
		if (prefixBits > 128) {
			throw st.exception("prefix bits must be [0..128]");
		}
		else if (prefixBits < 128) {
			try {
				suffix = new Inet6Address(st.getString());
			}
			catch (TextParseException e) {
				throw st.exception(e.getMessage());
			}
		}
		if (prefixBits > 0)
			prefix = st.getName(origin);
	}

	/** Converts rdata to a String */
	String rrToString() {
		StringBuffer sb = new StringBuffer();
		sb.append(prefixBits);
		if (suffix != null) {
			sb.append(" ");
			sb.append(suffix);
		}
		if (prefix != null) {
			sb.append(" ");
			sb.append(prefix);
		}
		return sb.toString();
	}

	/** Returns the number of bits in the prefix */
	public int getPrefixBits() {
		return prefixBits;
	}

	/** Returns the address suffix */
	public Inet6Address getSuffix() {
		return suffix;
	}

	/** Returns the address prefix */
	public Name getPrefix() {
		return prefix;
	}

	void rrToWire(DNSOutput out, Compression c, boolean canonical) {
		out.writeU8(prefixBits);
		if (suffix != null) {
			int suffixbits = 128 - prefixBits;
			int suffixbytes = (suffixbits + 7) / 8;
			byte[] data = suffix.toBytes();
			out.writeByteArray(data, 16 - suffixbytes, suffixbytes);
		}
		if (prefix != null)
			prefix.toWire(out, null, canonical);
	}

}
