/*
 * @(#)SimpleMessage.java            2004. 11. 24.
 *
 * Copyright (c) 1998-2004 Amail, Inc.
 * 708-8 Global Building 10th floor, YeokSamdong, Gangnamgu, Seoul, 
 * Korea republic of. All rights reserved.
 * 
 * This software is the confidential and proprietary information of Amail,
 * Inc. ("Confidential Information"). You shall not disclose such 
 * Confidential Information and shall use it only in accordance with
 * the terms of the license agreement you entered into Amail.
 * 
 */

package mercury.contents.common.message;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;
import java.util.Random;

import lombok.extern.slf4j.Slf4j;
import mercury.contents.common.parser.BodyParser;
import pluto.lang.eMsLocale;
import pluto.lang.eMsTypes;
import pluto.util.convert.BASE64;
import pluto.util.convert.MimeConvertor;

/**
 * 기본적인 패턴의 Message를 정의한다. <br>
 * 기본 Content Type은 BASE_HTML_MESSAGE 이고 <BR>
 * 기본 인코딩 Type은 8BIT 이다.
 * 
 * @version
 * @author dragon
 *  
 */
@Slf4j
public class SimpleMessage implements Message {

	public static String				MESSAGE_ID_TAIL		= "@eMsSMTPV6.0(PLUTO)";

	protected Short						CONTENT_TYPE		= new Short(eMsTypes.BASE_HTML_MESSAGE);

	protected Short						CONTENT_ENCODING	= new Short(eMsTypes.ENC_8BIT);

	protected String					CHAR_SET			= null;

	protected String					MESSAGE_ID			= null;

	protected String					CONTENT_NAME		= null;

	protected BodyParser				PARSED_CONTENT		= null;

	protected String					MESSAGE_HEADER		= null;
	
	private Date myDate = new Date();
	private int MAX_RANDOM_VALUE = 8999;
	private int MIN_RANDOM_VALUE = 1000;
	private Random rnd = new Random();
	/** Creates new SimpleMessage */
	public SimpleMessage() {
		
		String localHost = eMsLocale.SMTP_LOCAL_HOST[0];
		if( localHost==null ) {
			localHost = "TMS_SMTPV3.0";
		}
		int idx1 = localHost.indexOf("[");
		if( idx1 > -1 ) {
			int idx2 = localHost.indexOf("]");
			localHost = localHost.substring(idx1+1, idx2>-1?idx2:localHost.length() );
		}
		MESSAGE_ID_TAIL = "@"+localHost;
		MESSAGE_ID = String.valueOf(System.currentTimeMillis()).concat(MESSAGE_ID_TAIL);
	}

	/**
	 * Content 타입을 정의한다. <br>[ TEXT , HTML ]
	 * 
	 * @param type
	 *            Content 타입
	 * @see kr.co.amail.lang.eMsTypes#TEXT_PLAIN
	 * @see kr.co.amail.lang.eMsTypes#TEXT_HTML
	 */
	public void setContentType(Short type) {
		this.CONTENT_TYPE = type;
	}

	/**
	 * Content type을 반환한다.
	 * 
	 * @return Content type
	 */
	public Short getContentType() {
		return this.CONTENT_TYPE;
	}

	/**
	 * 메세지의 케릭터 셋을 지정
	 * 
	 * @param charset
	 *            케릭터 셋
	 */
	public void setCharSet(String charset) {
		this.CHAR_SET = charset;
	}

	/**
	 * 케릭터 셋을 반환한다.
	 * 
	 * @return 케릭터 셋
	 */
	public String getCharSet() {
		return this.CHAR_SET;
	}

	/**
	 * 메세지의 인코딩 방식을 지정한다.
	 * 
	 * @param enc_type
	 *            메세지의 인코딩 방식
	 * @see kr.co.amail.lang.eMsTypes#ENC_BASE64
	 * @see kr.co.amail.lang.eMsTypes#ENC_QP
	 * @see kr.co.amail.lang.eMsTypes#ENC_UUENCODE
	 * @see kr.co.amail.lang.eMsTypes#ENC_8BIT
	 * @see kr.co.amail.lang.eMsTypes#ENC_7BIT
	 */
	public void setContentEncoding(Short enc_type) {
		this.CONTENT_ENCODING = enc_type;
	}

	/**
	 * 컨텐트 인코딩 타입을 반환한다.
	 * 
	 * @return 컨텐트 인코딩 타입
	 * @see kr.co.amail.lang.eMsTypes#ENC_BASE64
	 * @see kr.co.amail.lang.eMsTypes#ENC_QP
	 * @see kr.co.amail.lang.eMsTypes#ENC_UUENCODE
	 * @see kr.co.amail.lang.eMsTypes#ENC_8BIT
	 * @see kr.co.amail.lang.eMsTypes#ENC_7BIT
	 */
	public Short getContentEncoding() {
		return this.CONTENT_ENCODING;
	}

	/**
	 * Content를 작성하는 BodyParser를 지정한다.
	 * 
	 * @param content
	 *            Content를 작성하는 BodyParser
	 */
	public void setContent(BodyParser content) {
		this.PARSED_CONTENT = content;
	}

	/**
	 * Content를 작성하는 BodyParser를 반환한다.
	 * 
	 * @return Content를 작성하는 BodyParser
	 */
	public BodyParser getContent() {
		return this.PARSED_CONTENT;
	}

	/**
	 * 메세지 본문을 반환한다.
	 * 
	 * @param info1
	 *            개인화 매핑 데이터
	 * @param info2
	 *            기본 매핑 데이터
	 * @throws Exception
	 *             작성시 에러
	 * @return 메세지 본문
	 * @deprecated 스트림으로 바로 쓰는 것으로 사용이 전환됨.
	 */
	public String getStringMessage(Object info1, Object info2, Properties prop) throws Exception {
		ByteArrayOutputStream out = null;
		try {
			out = new ByteArrayOutputStream(512);
			putStringMessageToStream(info1, info2, prop, out);
			return new String (out.toByteArray(), eMsLocale.FILE_SYSTEM_OUT_CHAR_SET );
		}
		catch(Exception e) {
			throw new RuntimeException(e.toString());
		}
		finally {
			try {
				if(out != null) {
					out.close(); 
				}
			} catch(Exception e) {}
			out = null;
		}
	}

	public void putStringMessageToStream(Object info1, Object info2, Properties prop, OutputStream out) throws Exception {

		String returnValue = this.PARSED_CONTENT.convert(info1, info2, prop);

		// String MAIL_BASE_CHAR_SET = prop.getProperty("MAIL_BASE_CHAR_SET",eMsLocale.MAIL_BASE_CHAR_SET);

		if (log.isDebugEnabled()) {
			log.debug("message original body", returnValue);
		}

		/*
		 * 바이너리 첨부 파일일 경우에는 이후 BodyParser 자체에서 Content를 만들기 때문에 이후의 변환 작업이 필요하지
		 * 않다. 그래서 그냥 BodyParser에서 생성한 Content를 반환한다.
		 */
		if( this.CONTENT_TYPE.shortValue() == eMsTypes.ATTACH_BINARY_MESSAGE ) {
			out.write(returnValue.getBytes( eMsLocale.CHAR_SET ));
			return;
		}

		short encType = this.CONTENT_ENCODING.shortValue();

		switch (encType) {
			case eMsTypes.ENC_QP:
			case eMsTypes.ENC_BASE64:
			case eMsTypes.ENC_UUENCODE:
			case eMsTypes.ENC_7BIT: {
				
				//MimeConvertor.putMimeToStream(out, returnValue,eMsLocale.MAIL_BASE_CHAR_SET, this.CHAR_SET, encType);
				MimeConvertor.putMimeToStream(out, returnValue, eMsLocale.MAIL_BASE_CHAR_SET, this.CHAR_SET, encType);				
				break;
			}

			default: {
				out.write(returnValue.getBytes( eMsLocale.CHAR_SET) );
				break;
			}
		}

		if (log.isDebugEnabled()) {
			log.debug("message convert body", out.toString());
		}
		return;
	}

	/**
	 * 
	 * @deprecated 스트림에다가 바로 쓰기 때문에 이것은 더이상 사용하면 안된다.
	 */
	public String getHeader() {
		ByteArrayOutputStream out = null;
		try {
			out = new ByteArrayOutputStream(512);
			putHeaderToStream(out);
			return out.toString();
		}
		catch(Exception e) {
			throw new RuntimeException(e.toString());
		}
		finally {
			try {
				if(out != null) {
					out.close(); 
				}
			} catch(Exception e) {}
			out = null;
		}
	}

	/**
	 * 메세지의 헤더를 반환한다.
	 * 
	 * @return 메세지 헤더
	 */
	public void putHeaderToStream(OutputStream out) throws Exception {
		// 메세지의 헤더는 기본적으로 변경되는 것이 아니기 때문에 한번 생성하고 재활용한다.
		/*if( this.MESSAGE_HEADER != null ) {
			out.write(this.MESSAGE_HEADER.getBytes(eMsLocale.CHAR_SET));
			return;
		}*/
		//Content Type 추가
		StringBuffer buffer = null;
		try {
			// TODO 메세지 아이디를 생성하는 것을 실제 사용가능한 아이디로 부여해야 할거 같다.
			// 예를 들면 지금 POST_ID 등을 X- 로 추가하는데 이것을 여기에서 넣도록 하는것도 방법일듯..
			
			// 메세지 아이디를 매번 발송시마다 유니크하게 하며
			// RFC 2822 규약에 맞춤 2006-11-14
			SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
			myDate.setTime( System.currentTimeMillis() );
			
			buffer = new StringBuffer(512);
			buffer.append("Message-ID: <");
			buffer.append(sdf.format( myDate ));
			buffer.append(rnd.nextInt(MAX_RANDOM_VALUE) + MIN_RANDOM_VALUE);
			buffer.append("_TMS");
			buffer.append(MESSAGE_ID_TAIL);
			buffer.append(">\r\n");
			buffer.append("Content-Type: ");
			switch (this.CONTENT_TYPE.shortValue()) {
				case eMsTypes.BASE_TEXT_MESSAGE:
				case eMsTypes.ATTACH_TEXT_MESSAGE: {
					buffer.append("text/plain;\r\n");
					break;
				}

				case eMsTypes.BASE_HTML_MESSAGE:
				case eMsTypes.ATTACH_HTML_MESSAGE: {
					buffer.append("text/html;\r\n");
					break;
				}

				case eMsTypes.ATTACH_BINARY_MESSAGE: {
					buffer.append("application/octet-stream;\r\n");
					break;
				}

				case eMsTypes.ATTACH_IMAGE_MESSAGE: {
					buffer.append("image/jpeg;\r\n");
					break;
				}

				default: {
					buffer.append("text/html;\r\n");
					break;
				}
			}

			switch (this.CONTENT_TYPE.shortValue()) {
				/**
				 * 텍스트일 경우에는 charset을 추가한다.
				 */
				case eMsTypes.BASE_TEXT_MESSAGE:
				case eMsTypes.ATTACH_TEXT_MESSAGE:
				case eMsTypes.BASE_HTML_MESSAGE:
				case eMsTypes.ATTACH_HTML_MESSAGE: {
					buffer.append("\tcharset=\"");
					buffer.append(this.CHAR_SET);
					buffer.append("\"\r\n");
					break;
				}

				/**
				 * BINARY일 경우에는 이름을 추가한다.
				 */
				case eMsTypes.ATTACH_BINARY_MESSAGE:
				case eMsTypes.ATTACH_IMAGE_MESSAGE: {
					/**
					 * 헤더 인코딩 방식에 따라 파일 이름을 작성한다. 첨부되는 파일의 경우의 모든 파일 이름은 base64로
					 * 인코딩한다.
					 */
					buffer.append("\tname=\"=?");
					buffer.append(this.CHAR_SET);
					buffer.append("?B?");
					buffer.append(BASE64.encode(this.getContentName()));
					buffer.append("?=\"\r\n");
					break;
				}
			}

			/**
			 * Content-Transfer-Encoding 헤더를 추가한다. 인코딩 방식은 첨부 바이너리나 이미지의 경우에는
			 * 무조건 base64이고 나머지는 세팅된 인코딩 방식을 표시한다.
			 */
			buffer.append("Content-Transfer-Encoding: ");

			if( this.CONTENT_TYPE.shortValue() == eMsTypes.ATTACH_BINARY_MESSAGE || this.CONTENT_TYPE.shortValue() == eMsTypes.ATTACH_IMAGE_MESSAGE ) {
				buffer.append("base64");
			}
			else {
				switch (this.CONTENT_ENCODING.shortValue()) {
					case eMsTypes.ENC_BASE64: {
						buffer.append("base64");
						break;
					}

					case eMsTypes.ENC_QP: {
						buffer.append("quoted-printable");
						break;
					}

					case eMsTypes.ENC_UUENCODE: {
						buffer.append("uuencode");
						break;
					}

					case eMsTypes.ENC_8BIT: {
						buffer.append("8bit");
						break;
					}

					case eMsTypes.ENC_7BIT: {
						buffer.append("7bit");
						break;
					}

					default: {
						buffer.append("8bit");
						break;
					}
				}
			}

			buffer.append("\r\n");

			/**
			 * 첨부파일일 경우에는 Content-Disposition 헤더를 추가한다.
			 */
			if( this.CONTENT_TYPE.shortValue() == eMsTypes.ATTACH_BINARY_MESSAGE || this.CONTENT_TYPE.shortValue() == eMsTypes.ATTACH_HTML_MESSAGE ) {
				buffer.append("Content-Disposition: attachment;\r\n");
				buffer.append("\tfilename=\"=?");
				buffer.append(this.CHAR_SET);
				buffer.append("?B?");
				buffer.append(BASE64.encode(this.getContentName()));
				buffer.append("?=\"\r\n");
			}

			/**
			 * 첨부이미지일 경우에는 Content-ID 헤더를 추가한다.
			 */
			if( this.CONTENT_TYPE.shortValue() == eMsTypes.ATTACH_IMAGE_MESSAGE ) {
				buffer.append("Content-ID: <");
				buffer.append(this.getMessageID());
				buffer.append(">\r\n");
			}
			this.MESSAGE_HEADER = buffer.toString();

			out.write(this.MESSAGE_HEADER.getBytes(eMsLocale.CHAR_SET));
		}
		finally {
			buffer = null;
		}
	}

	/**
	 * 메세지의 아이디를 지정한다.
	 * 
	 * @param id
	 *            메세지의 아이디
	 */
	public void setMessageID(String id) {
		if( id.endsWith(SimpleMessage.MESSAGE_ID_TAIL) ) {
			this.MESSAGE_ID = id;
		}
		else {
			this.MESSAGE_ID = id.concat(SimpleMessage.MESSAGE_ID_TAIL);
		}
	}

	/**
	 * 메세지의 아이디를 반환한다.
	 * 
	 * @return 메세지의 아이디
	 */
	public String getMessageID() {
		return this.MESSAGE_ID;
	}

	public String getContentName() {
		return this.CONTENT_NAME == null ? this.PARSED_CONTENT.getName() : this.CONTENT_NAME;
	}

	public void setContentName(String name) {
		this.CONTENT_NAME = name;

	}

	protected void finalize() {
	}
}
