package jupiter.mass.log.updator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import lombok.extern.slf4j.Slf4j;
import pluto.config.SqlManager;
import pluto.config.eMsSystem;
import pluto.db.ConnectInfo;
import pluto.db.ConnectionPool;
import pluto.db.eMsConnection;
import pluto.db.eMsPreparedStatement;
import pluto.db.eMsResultSet;
import pluto.db.eMsStatement;
import pluto.schedule.Task;
import pluto.util.Cal;
import pluto.util.StringConvertUtil;

import jupiter.mass.log.updator.KakaoSwitchUpdator;

@Slf4j
public class KakaoMsgUpdator extends Task {
	
	private static ConnectInfo otherConnectInfo = new ConnectInfo();
	
	private static String querySelectTargetList = null;
	private static String queryAutoUpdateScheduleInfo = null;
	private static String queryCampUpdateScheduleInfo = null;
	
	private static final String _MASS_ = "MASS";
	private static final String _AUTO_ = "AUTO";
	private static final String _TEST_ = "TEST";
	
	private eMsConnection oriCon = null;
	private eMsConnection moduleCon = null;
	
	private eMsResultSet rsSelect = null;
	private eMsPreparedStatement psSelect = null;
	
	private eMsStatement psSchdUpdate = null;
	
	private boolean isOtherDb = false;
	private Properties taskProp = null;
	
	private static int commitInterval = 1000;
	
	private Map<String, List<Properties>> autoListTarget = new HashMap<String, List<Properties>>();
	private Map<String, List<Properties>> campListTarget = new HashMap<String, List<Properties>>();
	
	private List<Properties> updateModuleTarget = new ArrayList<Properties>();
	
	private Map<String,ScheduleTblBean> autoSchdSummary = new HashMap<String,ScheduleTblBean>();
	private Map<String,ScheduleTblBean> campSchdSummary = new HashMap<String,ScheduleTblBean>();
	
	static {
		querySelectTargetList = SqlManager.getQuery("SWITCHED_UPDATE", "QUERY_SELECT_TARGET_LIST");
		
		queryAutoUpdateScheduleInfo = SqlManager.getQuery("SWITCHED_UPDATE", "QUERY_UPDATE_AUTO_SCHEDULE_INFO");
		queryCampUpdateScheduleInfo = SqlManager.getQuery("SWITCHED_UPDATE", "QUERY_UPDATE_CAMP_SCHEDULE_INFO");
		
		otherConnectInfo.setDRIVER(eMsSystem.getProperty("kakao.db.driver"));
		otherConnectInfo.setDB_URL(eMsSystem.getProperty("kakao.db.url"));
		otherConnectInfo.setDB_UID(eMsSystem.getProperty("kakao.db.id"));
		otherConnectInfo.setDB_PASS(eMsSystem.getProperty("kakao.db.pass"));
		otherConnectInfo.setDB_INIT_QUERY(eMsSystem.getProperty("kakao.db.init"));
		otherConnectInfo.setDB_BASE_CHARSET(eMsSystem.getProperty("kakao.db.base.charset"));
		otherConnectInfo.setDB_OUT_CHARSET(eMsSystem.getProperty("kakao.db.out.charset"));
		otherConnectInfo.setDB_IN_CHARSET(eMsSystem.getProperty("kakao.db.in.charset"));
	}
	
	public KakaoMsgUpdator() {
		super();
	}
	
	@Override
	public void setTaskProperty(Properties prop) {
		taskProp = prop;
		
		String currentTime = Cal.getSerialDate();
		this.setName("KakaoMsgUpdator_at_" + currentTime);
		this.setTaskID("KakaoMsgUpdator");
		
		isOtherDb = Boolean.parseBoolean(taskProp.getProperty("use.other.db","false"));
		commitInterval = Integer.parseInt(taskProp.getProperty("commitInterval","1000"));
	}
	
	@Override
	public void execute_initiate() throws Exception {
		log.info("{}... START", getName());
		
		oriCon = ConnectionPool.getConnection();
		moduleCon = ConnectionPool.getConnection(otherConnectInfo);
		psSelect = moduleCon.prepareStatement(querySelectTargetList, "${", "}");
		psSchdUpdate = oriCon.createStatement();
	}
	/**
	 * CAMP / AUTO / TEST
	 * 테스트의 경우 모듈사 테이블에 완료 업데이트 하며, send_list / schd 테이블 업데이트하지 않는다.
	 * 잘못해서 테스트가 들어올 수 있어 다음과 같이 처리
	 */
	@Override
	public void execute() throws Exception {
		
		rsSelect = psSelect.executeQuery(taskProp);
		
		while(rsSelect.next()) {
			
			Properties result = new Properties();
			eMsPreparedStatement psListUpdate = null;
			
			try {
				
				rsSelect.putToMap(result, false);
				log.debug("Target List Info: {}", result);
				String sendType = result.getProperty("SEND_TYPE","").trim();
				String postId = result.getProperty("POST_ID","");
				String memberId = result.getProperty("MEMBER_ID");
				String listTable = result.getProperty("LIST_TABLE","");
				if (postId.isEmpty()) continue;
				if (memberId.isEmpty()) continue;
				if (sendType.isEmpty()) continue;
				if (listTable.isEmpty()) continue;
				
				if(sendType.equals(_AUTO_)){
					if (postId.lastIndexOf("_")>0) {
						result.setProperty("WORKDAY", postId.split("_")[0]);
						result.setProperty("SEQNO", postId.split("_")[1]);
					}else{
						log.error("[ERROR] The postId format is incorrect : {}" , postId);
						continue;
					}
						
					if (memberId.lastIndexOf("_")>0){
						result.setProperty("MEMBER_ID", memberId.split("_")[0]);
						result.setProperty("MEMBER_ID_SEQ", memberId.split("_")[1]);
					}else{
						log.error("[ERROR] The memberId format is incorrect. : {}" , postId);
						continue;
					}
					
					List<Properties> autoTarget = autoListTarget.get(listTable);
					if(autoTarget == null){
						autoTarget = new ArrayList<Properties>();
						autoListTarget.put(listTable, autoTarget);
					}
					autoTarget.add(result);
					
					ScheduleTblBean autoBean = autoSchdSummary.get(postId);
					if (autoBean == null) {
						autoBean = new ScheduleTblBean(postId);
						autoSchdSummary.put(postId, autoBean);
					}
					autoBean.addSwitchedCnt();
				}else if(sendType.equals(_MASS_)){
					
					List<Properties> campTarget = campListTarget.get(listTable);
					if(campTarget == null){
						campTarget = new ArrayList<Properties>();
						campListTarget.put(listTable, campTarget);
					}
					campTarget.add(result);
					
					ScheduleTblBean campBean = campSchdSummary.get(postId);
					if (campBean == null) {
						campBean = new ScheduleTblBean(postId);
						campSchdSummary.put(postId, campBean);
					}
					campBean.addSwitchedCnt();
				}else if(sendType.equals(_TEST_)){
					
				}else{
					log.info("[PASS] Msg Result is not updated. {}", result);
					continue;
				}
				// 업데이트 할 테이블 이름이 다를 수 있으므로
			} catch(Exception e) {
				log.error("[ERROR] Msg Result process.", e);
			} finally {
				if (psListUpdate != null) psListUpdate.close();
			}
		}
		
		if (!autoListTarget.isEmpty()) {
			listUpdate(_AUTO_, autoListTarget);
		}
		if (!campListTarget.isEmpty()) {
			listUpdate(_MASS_ ,campListTarget);
		}
		if (!autoSchdSummary.isEmpty()) {
			updateSummary(_AUTO_,autoSchdSummary);
		}
		if (!campSchdSummary.isEmpty()) {
			updateSummary(_MASS_,campSchdSummary);
		}
	}
	
	@Override
	public void execute_initiateError(Throwable thw) {
		log.error("execute_initiateError()...", thw);
	}

	@Override
	public void release_Resource() {
		if (rsSelect != null) rsSelect.close();
		if (psSelect != null) psSelect.close();
		
		if (psSchdUpdate != null) psSchdUpdate.close();
		
		if (moduleCon != null) moduleCon.close();
		if (oriCon != null) oriCon.close();
		
		log.info("{}... END", getName());
	}
	
	// 스케쥴별 summary를 업데이트 함
	private void updateSummary(String type, Map<String,ScheduleTblBean> schdMap) {
		
		StringBuffer buff = new StringBuffer(512);
		
		try {
			
			for (Map.Entry<String, ScheduleTblBean> target : schdMap.entrySet()) {
				
				buff.setLength(0);
				
				ScheduleTblBean stBean = target.getValue();
				
				log.debug("{} summary stBean data: {}", getName(), stBean.toString());
				
				String targetQuery;
				if (type.equals(_AUTO_)) {
					targetQuery = queryAutoUpdateScheduleInfo;
				} else {
					targetQuery = queryCampUpdateScheduleInfo;
				}
				
				StringConvertUtil.ConvertString(buff, targetQuery, stBean.getElement(type), "${" , "}", true, false);
				
				if (this.psSchdUpdate.executeUpdate(buff.toString()) < 1) {
					log.debug("{} schedule summary fail: {}", getName(), stBean.getName());
				}
			}
		} catch(Exception e) {
			log.error("ERROR schdSummary... ", e);
		}
	}
	
	private void listUpdate(String type, Map<String,List<Properties>> targetList) {
		
		StringBuffer buff = new StringBuffer(512);
		log.info("Taget Type is {}", type);
		try {
			
			for (Map.Entry<String, List<Properties>> target : targetList.entrySet()) {
				buff.setLength(0);
				String listTable = target.getKey();
				List<Properties> targetListData = target.getValue();
				log.debug("{} updateTarget List table data: {}, {}", getName(), targetListData, listTable);
				KakaoSwitchUpdator switchUpdator = new KakaoSwitchUpdator();
				switchUpdator.init(commitInterval);
				switchUpdator.update(targetListData, listTable, type);
				switchUpdator.moduleUpdate(targetListData);
				switchUpdator.release();
			}
		} catch(Exception e) {
			log.error("ERROR UpdateError... ", e);
		} 
	}
	
	protected String ppsParseSQL(String source, String mapping, String value,String start, String end) {

		if( mapping == null || source == null )
			return source;

		return replace_target(source, start + mapping + end, value);
	}
	
	public static String replace_target(String target, String from, String to) {
		if( target == null || from == null || to == null )
			return null;

		int idx1 = 0;
		int idx2;

		while ((idx2 = target.indexOf(from, idx1)) != -1) {
			target = target.substring(0, idx2) + to + target.substring(idx2 + from.length());
			idx1 = idx2 + to.length();
		}
		return target;
	}
	
	
}
