/*
 * @(#)SmtpCommunicator.java            2004. 12. 6.
 *
 * Copyright (c) 1998-2004 Amail, Inc.
 * 708-8 Global Building 10th floor, YeokSamdong, Gangnamgu, Seoul, 
 * Korea republic of. All rights reserved.
 * 
 * This software is the confidential and proprietary information of Amail,
 * Inc. ("Confidential Information"). You shall not disclose such 
 * Confidential Information and shall use it only in accordance with
 * the terms of the license agreement you entered into Amail.
 * 
 */

package jupiter.common.communicator;

import java.io.EOFException;
import java.io.IOException;

import freemarker20.template.SimpleHash;
import pluto.config.CodeAnalizer;
import lombok.extern.slf4j.Slf4j;
import pluto.lang.eMsLocale;
import pluto.log.Composer;
import pluto.log.FailedMailLogger;
import pluto.log.Log;
import pluto.log.SmtpLogger;
import pluto.mail.SendState;
import pluto.mail.mx.DNS;
import pluto.mail.mx.LookupCacheManager;
import pluto.mail.mx.LookupUtil;
import pluto.mail.mx.MXSearchResult;
import pluto.mail.mx.exception.NameNotKnownException;
import pluto.mgs.util.TimeBasedSequenceIdFactory;
import pluto.net.SocketReaderAgent;
import pluto.util.Cal;
import pluto.util.convert.DelimConvertor;
import venus.spool.common.basic.SpoolingManager;
 
/**
 * <br>
 * 메일 발송 통신의 단계를 객체로 구현했다. <br>
 * 서버와 연결하는 방법을 정의하고 단계별 Time Out을 셋팅한다.
 * 
 * @version
 * @author dragon
 *  
 */
@Slf4j
public abstract class SmtpCommunicator extends SocketReaderAgent implements Log {

	/**
	 * 메일 발송 각단계의 상태를 저장한다.
	 */
	protected SendState				sendState					= new SendState();

	/**
	 * 대상도메인
	 */
	protected String				domain						= null;

	/**
	 * 대상 리스트
	 */
	protected Object				RCPT_TO						= null;

	/**
	 * 발송단계를 저장한다.
	 * 
	 * @see pluto.mail.SendState#WAIT
	 * @see pluto.mail.SendState#START
	 * @see pluto.mail.SendState#CONNECT
	 * @see pluto.mail.SendState#INIT
	 * @see pluto.mail.SendState#HELO
	 * @see pluto.mail.SendState#MAIL_FROM
	 * @see pluto.mail.SendState#RCPT_TO
	 * @see pluto.mail.SendState#DATA
	 * @see pluto.mail.SendState#DATA_BODY
	 * @see pluto.mail.SendState#QUIT
	 * @see pluto.mail.SendState#CLOSE
	 * @see pluto.mail.SendState#RSET
	 */
	protected short					CURR_STEP					= SendState.WAIT;

	/**
	 * 로그 작성
	 */
	protected Composer				composer					= null;

	/**
	 * 일하는 녀석의 인덱스를 지정
	 */
	protected int					AGENT_IDX					= 0;

	public int						INSTANCE_TIMEOUT_CONNECT	= eMsLocale.TIMEOUT_CONNECT;

	public int						INSTANCE_TIMEOUT_INIT		= eMsLocale.TIMEOUT_INIT;

	public int						INSTANCE_TIMEOUT_HELO		= eMsLocale.TIMEOUT_HELO;

	public int						INSTANCE_TIMEOUT_MAILFROM	= eMsLocale.TIMEOUT_MAILFROM;

	public int						INSTANCE_TIMEOUT_RCPT		= eMsLocale.TIMEOUT_RCPT;

	public int						INSTANCE_TIMEOUT_DATAINIT	= eMsLocale.TIMEOUT_DATAINIT;

	public int						INSTANCE_TIMEOUT_DATABLOCK	= eMsLocale.TIMEOUT_DATABLOCK;

	public int						INSTANCE_TIMEOUT_RSET		= eMsLocale.TIMEOUT_RSET;

	// 회원아이디 구분자 인코딩을 위해서 사용됨
	protected StringBuffer			TMP_BUFFER					= null;

	protected static final boolean	STEP_DEBUG					= false;

	protected SimpleHash			_MEMBER_HASH_				= new SimpleHash();

	/**
	 * 기본 생성자 모니터링을 하지않는 인스턴스를 생성한다.
	 */
	public SmtpCommunicator() throws Exception {
		this(false);
	}

	/**
	 * 기본 생성자
	 * 
	 * @param regist
	 *            true : 모니터에 등록 <br>
	 *            false : 모니터에 등록하지 않음
	 *  
	 */
	public SmtpCommunicator(boolean regist) throws Exception {
		super(regist);
		// 회원아이디 구분자 인코딩을 위한 버퍼 초기화
		this.TMP_BUFFER = new StringBuffer(256);
		this.composer = Composer.getComposerInstance();
	}

	public void setTIMEOUT_CONNECT(int timeout) {
		this.INSTANCE_TIMEOUT_CONNECT = timeout;
	}

	public void setTIMEOUT_INIT(int timeout) {
		this.INSTANCE_TIMEOUT_INIT = timeout;
	}

	public void setTIMEOUT_HELO(int timeout) {
		this.INSTANCE_TIMEOUT_HELO = timeout;
	}

	public void setTIMEOUT_MAILFROM(int timeout) {
		this.INSTANCE_TIMEOUT_MAILFROM = timeout;
	}

	public void setTIMEOUT_RCPT(int timeout) {
		this.INSTANCE_TIMEOUT_RCPT = timeout;
	}

	public void setTIMEOUT_DATAINIT(int timeout) {
		this.INSTANCE_TIMEOUT_DATAINIT = timeout;
	}

	public void setTIMEOUT_DATABLOCK(int timeout) {
		this.INSTANCE_TIMEOUT_DATABLOCK = timeout;
	}

	public void setTIMEOUT_RSET(int timeout) {
		this.INSTANCE_TIMEOUT_RSET = timeout;
	}

	/**
	 * SMTP Protocol 각 단계를 수행한다.
	 * 
	 * @param step
	 *            SMTP 단계
	 * @param extra
	 *            추가 전달 String
	 * @param domain
	 *            접속한 도메인
	 * @param que_step
	 *            발송단계
	 */
	protected void step(short step, String extra, String domain, String que_step) {
		this.step(step, extra, domain, Integer.parseInt(que_step));
	}

	/**
	 * SMTP Protocol 각 단계를 수행한다.
	 * 
	 * @param step
	 *            SMTP 단계
	 * @param extra
	 *            추가 전달 String
	 * @param domain
	 *            접속한 도메인
	 * @param que_step
	 *            발송단계
	 */
	protected void step(short step, String extra, String domain, int que_step) {
		// 각 단계를 세팅한다.
		this.CURR_STEP = step;

		// 발송 상태를 초기화
		this.sendState.reset();

		String send_string = null;

		String initial = null;
		
		int amountTimeout;
		
		switch (step) {
			
			case SendState.RELAY: {
				if( STEP_DEBUG )
					log("STEP RELAY");
				initial = "INIT";
				amountTimeout = this.INSTANCE_TIMEOUT_INIT;
				send_string = domain + ":" + 25;
				break;
			}
		
			case SendState.INIT: {
				if( STEP_DEBUG )
					log("STEP INIT");
				initial = "INIT";
				amountTimeout = this.INSTANCE_TIMEOUT_INIT;
				break;
			}
			case SendState.HELO: {
				if( STEP_DEBUG )
					log("STEP HELO");
				initial = "HELO";
				amountTimeout = this.INSTANCE_TIMEOUT_HELO;
				send_string = "HELO " + extra;
				break;
			}
			case SendState.SEQ: {
				if( STEP_DEBUG )
					log("STEP SEQ");
				//seq				
				String seq = TimeBasedSequenceIdFactory.seq(_MEMBER_HASH_.getAsString("POST_ID"));		
				this.sendState.setMGS_SEQ(seq);
				initial = "SEQ";
				amountTimeout = this.INSTANCE_TIMEOUT_HELO;
				send_string = eMsLocale.MGS_SEQ_HEADER + seq;
				break;
				
			}
			case SendState.MAIL_FROM: {
				if( STEP_DEBUG )
					log("STEP MAILFROM");
				initial = "MAILFROM";
				amountTimeout = this.INSTANCE_TIMEOUT_MAILFROM;
				send_string = "MAIL FROM: <".concat(extra).concat(">");
				break;
			}
			case SendState.RCPT_TO: {
				if( STEP_DEBUG )
					log("STEP RCPT");
				initial = "RCPT";
				amountTimeout = this.INSTANCE_TIMEOUT_RCPT;
				send_string = "RCPT TO: <".concat(extra).concat(">");
				break;
			}
			case SendState.DATA: {
				if( STEP_DEBUG )
					log("STEP DATAINIT");
				initial = "DATAINIT";
				amountTimeout = this.INSTANCE_TIMEOUT_DATAINIT;
				send_string = "DATA";
				break;
			}
			case SendState.DATA_BODY: {
				if( STEP_DEBUG )
					log("STEP DATA BODY");
				initial = "DATABODY";
				amountTimeout = this.INSTANCE_TIMEOUT_DATABLOCK;
				send_string = extra.concat("\r\n.");
				break;
			}
			case SendState.RSET: {
				if( STEP_DEBUG )
					log("STEP RSET");
				initial = "RSET";
				send_string = "RSET";
				amountTimeout = this.INSTANCE_TIMEOUT_RSET;
				break;
			}
			case SendState.QUIT: {
				if( STEP_DEBUG )
					log("STEP QUIT");
				initial = "QUIT";
				send_string = "QUIT";
				amountTimeout = this.INSTANCE_TIMEOUT_RSET;
				break;
			}
			default: {
				if( STEP_DEBUG )
					log("UNKNOWN COMMAND");
				this.sendState.set("UNKNOWN COMMAND", "99");
				return;

			}
		}

		String returnCode = null;
		String receiveString = null;

		try {
			// 전달할 메세지를 전달한다.
			if( step != SendState.INIT) {
//				send(send_string.getBytes(eMsLocale.FILE_SYSTEM_OUT_CHAR_SET));
				// 빠르게 구현하기 위해서 _MEMBER_HASH 변수를 옮겼다.
				send(send_string.getBytes(_MEMBER_HASH_.getAsString("MAIL_BASE_CHAR_SET",eMsLocale.MAIL_BASE_CHAR_SET)));
			}
			
			if (step == SendState.SEQ) {
				return ;
			}

			// 타임아웃을 결정한다. 2008.06.28
			this.CONNECTED_SOCKET.setSoTimeout(amountTimeout);
			
			// 상대 서버 Response를 받는다.
			do {
				receiveString = receive();
				if( receiveString == null ) {
					receiveString = "000 SEND NULL";
				}
				
				returnCode = CodeAnalizer.searchReturnCode(initial.toLowerCase(), receiveString, domain);

			} while (returnCode != null && returnCode.equals("continue"));

			if( returnCode == null ) {
				this.sendState.set(initial + " Receive Unknown response => " + receiveString, "95");
				if( STEP_DEBUG )
					log(initial + "=>" + receiveString + " Receive Unknown response");
				return;
			}

			if( !returnCode.equals("00") ) {
				this.sendState.set(initial + " receive[" + returnCode + "]" + "=>" + receiveString, returnCode);
				if( STEP_DEBUG )
					log(initial + " receive[" + returnCode + "]" + "=>" + receiveString);
				return;
			}
		}
		catch(Exception ex) {
			// ex.printStackTrace();
			FailedMailLogger.put(Thread.currentThread().getName() + ", " + this.getConnectHost() + ", step : " + step + ", time : " + getElapsedTime());
			
			if( ex instanceof IOException ){
				this.sendState.set(initial + " receive[" + ex.toString() + "]=>550  NetworkError", "90");
			}
			else{
				this.sendState.set(initial + " receive[" + ex.toString() + "]=>550  SystemError", "90");
			}
			
			if( STEP_DEBUG ){
				log(initial + " receive[" + ex.toString() + "]");
			}
			return;
		}
		this.sendState.set(receiveString, "00");
		if( STEP_DEBUG ){
			log(initial + " [OK]");
		}
		return;
	}

	/**
	 * 해당 도메인의 MailServer로 Socket을 연결한다.
	 * 
	 * @param domain
	 *            연결 도메인
	 */
	protected void connect(String domain, String step) {
		this.domain = domain;

		// 각 단계를 세팅한다.
		this.CURR_STEP = SendState.CONNECT;

		this.sendState.reset();

		/* DNS 찾아오기 */
		MXSearchResult __TARGET_DNS_LIST__ = LookupCacheManager.getMXSearchResult(domain);
		/* MGS Server 정보 */
		
		MXSearchResult RELAY_SERVER_LIST = null;
		
		if(eMsLocale.MGS_FLAG){
			RELAY_SERVER_LIST = LookupCacheManager.getMXrelayInfo();
		}
		
		// __TARGET_DNS_LIST__이 설정이 안되었거나, 유효시간이 지났으면 
		// 결국에는 유효시간안에 서버가 재부팅이 되었을 경우에는 첫번째 도메인에 대해서만 캐쉬로부터 읽어 들인다.
		synchronized (__TARGET_DNS_LIST__) {
			
			if(log.isDebugEnabled()) {
				log.debug("[__TARGET_DNS_LIST__]"+__TARGET_DNS_LIST__.toString()+"/"+__TARGET_DNS_LIST__.isError()+"/"+(__TARGET_DNS_LIST__.mxDomain == null));
			}
			//[JOO] touch 하는 시간을 구체적으로 구현했다.
			if(log.isDebugEnabled()) {
				log.debug("[CountErrorRaise]"+__TARGET_DNS_LIST__.getCountErrorRaise());
			}
			

			//******* domain을 서칭할 것인지를 판단하는 부분 ****************//
			//엔진 구동 후 해당 도메인으로 최초 발송
			boolean isFirst         = (__TARGET_DNS_LIST__.getError_type() == DNS.ERROR_INITIALIZE) && (__TARGET_DNS_LIST__.mxDomain == null);
			//error and is not over error_count
			boolean isNotOverErrCnt    =  __TARGET_DNS_LIST__.isError() && (__TARGET_DNS_LIST__.getCountErrorRaise() < LookupCacheManager.SEARCH_ERROR_DIRECT_SEARCH_COUNT) ;
			//is reflash time
			boolean isOverValidTime = !__TARGET_DNS_LIST__.isValidTime();
			
			//domain research
			if(  isFirst || isNotOverErrCnt || isOverValidTime){
				if(log.isDebugEnabled()) {
					log.debug("[touch]"+__TARGET_DNS_LIST__.toString());
				}
				__TARGET_DNS_LIST__.touch();
			}
			
		}
		
		if( __TARGET_DNS_LIST__.isError() ) {
			Throwable e = __TARGET_DNS_LIST__.getError();
			if( e instanceof NameNotKnownException || e instanceof  EOFException) {
				this.CONNECT_HOST = domain;
				this.sendState.set("550 HostUnknown:" + e.toString(), "20");
				return;
			}

			this.CONNECT_HOST = domain;
			this.sendState.set("550 DNS Search Error:" + e.toString(), "90");
			return;
		}

		if(log.isDebugEnabled()){
			log("try connect");
		}

		int TRY_CONNECT_COUNT = 0;
		int TRY_INIT_COUNT = 0;
		int iConnectIndex = -1;
		String __CONNECT_HOST__ = null;
		int __CONNECT_HOST_IP__ = -1;

		// 여러개의 MX레코드에서 에러가 발생할 경우 각각을 기록하기 위해서 사용 2004.11.15
		this.TMP_BUFFER.setLength(0);
		this.TMP_BUFFER.append("Connect Fail => ");

		// 설정된 connect와 init의 재시도 횟수만큼 반복한다. 
		while(TRY_CONNECT_COUNT <= eMsLocale.SMTP_CONNECT_TRY_LIMIT && TRY_INIT_COUNT <= eMsLocale.SMTP_INIT_TRY_LIMIT) {
			iConnectIndex = __TARGET_DNS_LIST__.getTargetIndex();
			
			// 에러가 발생했으면,
			if(iConnectIndex == -1){
				break;
			}

			__CONNECT_HOST__ = __TARGET_DNS_LIST__.getDomain(iConnectIndex);
			__CONNECT_HOST_IP__ = __TARGET_DNS_LIST__.getIP(iConnectIndex);

			
			/*
			 * DNS Lookup 이후에 강제 Relay 하기 위한 부분 
			 */
			if( LookupCacheManager.RELAY_HOST_AFTER_DNS != null ) {
			
				__CONNECT_HOST__	= LookupCacheManager.RELAY_HOST_AFTER_DNS;  
				__CONNECT_HOST_IP__ = LookupCacheManager.RELAY_HOST_AFTER_DNS_IP;  
			}
			
			if( LookupCacheManager.RELAY_DNS_HOSTS_ACTIVE ) {
				
				String sDomain = ""; 
				String sIP = "";
				
				sIP = LookupCacheManager.RELAY_DNS_HOSTS.getProperty(__CONNECT_HOST__,"");
				if( !"".equals(sIP)) {
					try {
						__CONNECT_HOST_IP__ = Integer.parseInt(sIP); 
					} catch(Exception e) {};
				} else {
					
					// 1차 도메인 체크 ( yahoo.com )
					int idx = __CONNECT_HOST__.lastIndexOf('.');

					try {
						sDomain = __CONNECT_HOST__.substring( __CONNECT_HOST__.lastIndexOf('.',idx-1)+1 );
						sIP = LookupCacheManager.RELAY_DNS_HOSTS.getProperty(sDomain,"");
					} catch(Exception e) {
						sIP = "";
					}
						
					if (!"".equals(sIP)) {
						try {
							__CONNECT_HOST_IP__ = Integer.parseInt(sIP); 
						} catch(Exception e) {};
					} else {
					
						// 2차 도메인 체크 ( yahoo.co.kr )
						try {
							idx = __CONNECT_HOST__.lastIndexOf('.');
							idx = __CONNECT_HOST__.lastIndexOf('.', idx-1);
							sDomain = __CONNECT_HOST__.substring( __CONNECT_HOST__.lastIndexOf('.', idx-1)+1 );
							
							sIP = LookupCacheManager.RELAY_DNS_HOSTS.getProperty(sDomain,"");
						} catch(Exception e) {
							sIP = "";
						}
						
						if( !"".equals(sIP)) {
							try {
								__CONNECT_HOST_IP__ = Integer.parseInt(sIP); 
							} catch(Exception e) {};
						}
					}					
				}
			} // End if
			
			// 연결오류났을때 delay체크를 위한 시간
			long longConnectStartTime = System.currentTimeMillis();

			try {
				// 타임아웃을 설정한다.
				if(eMsLocale.MGS_FLAG){
					connect(__CONNECT_HOST__,RELAY_SERVER_LIST.getIP(0), Integer.parseInt(eMsLocale.MGS_PORT), eMsLocale.TIMEOUT_CONNECT);
					
				}else{
					connect(__CONNECT_HOST__,__CONNECT_HOST_IP__, 25, eMsLocale.TIMEOUT_CONNECT);
				}
			}
			catch(Exception e) {
				
				// 에러가 발생한 시각을 해당 MX 리스트에 설정한다.
				__TARGET_DNS_LIST__.setMXError(iConnectIndex, System.currentTimeMillis());
				
				// 에러를 버퍼에 기록한다.
				this.TMP_BUFFER.append(__CONNECT_HOST__);
				this.TMP_BUFFER.append(":");
				this.TMP_BUFFER.append(e.toString());
				this.TMP_BUFFER.append(",");

				// 에러난 사항을 AgentLog에 남긴다.
				StringBuilder buffer = new StringBuilder(256);
				

				buffer.setLength(0);
				buffer.append("550 Error:[");
				buffer.append(__CONNECT_HOST__);
				buffer.append("#");
				buffer.append(String.valueOf(TRY_CONNECT_COUNT));
				buffer.append("] [");
				buffer.append(String.valueOf(System.currentTimeMillis() - longConnectStartTime));
				buffer.append("] ");
				buffer.append(e.toString());

				log(buffer.toString());

				FailedMailLogger.put(Thread.currentThread().getName() + ", " + this.getConnectHost() + ", step : " + step + ", time : " + getElapsedTime());
				
				if( e instanceof java.net.ConnectException ) {
					processBlanListHost(__CONNECT_HOST__);
				}
				
				TRY_CONNECT_COUNT++;
				continue;
			} // End catch
			
			if(eMsLocale.MGS_FLAG)
				step(SendState.RELAY, null, LookupUtil.getIntIPToString(__CONNECT_HOST_IP__), step);
			else
				step(SendState.INIT, null, domain, step);
			//Welcome Message를 받으러 갑니다. (INIT)
			if( this.sendState.isError() ) {
				// INIT 에러가 발생을 할 경우에는 connection을 닫아 버린다.
				__TARGET_DNS_LIST__.setMXError(iConnectIndex, System.currentTimeMillis());
				TRY_INIT_COUNT++;
				continue;
			}
			
			// Connect되고, INIT까지 성공했으면, 반복문을 빠져 나온다.
			break;				
		} // 

		// 연결이 되지 않았다면 모든 MX 레코드에 대해서 Exception이 발생한 것이므로 Connection 에러 처리를 하며
		// 각각의 에러를 기록한 버퍼를 로그로 작성한다.
		if( !isConnect() ) {
			this.TMP_BUFFER.setLength(this.TMP_BUFFER.length() - 1);
			this.sendState.set(this.TMP_BUFFER.toString(), "80");
			if( STEP_DEBUG )
				log("CONNECT FINAL FAIL");
			return;
		}

		if( this.sendState.isError() ) {
			return;
		}

		// 성공을 하면 도메인의 설정을 초기화를 시킨다.
		__TARGET_DNS_LIST__.resetMXError(iConnectIndex);
		this.sendState.set(null, "00");
	}

	/**
	 * ConnectException이 발생한 도메인에 대한 처리를 담당한다.
	 * 
	 * @param hostname
	 */
	protected void processBlanListHost(String hostname) {
		// do nothing
	}

	protected void closeConnection() {
		// BODY까지 진행이 된건지 아니면 중간에 Exception이 발생한건지 판단하여 뒷처리를 한다.
		// HELO 단계를 지났다면 QUIT를 실행한다.
		
		// INIT 단계를 지났다면 QUIT를 실행해도 된다.
		if( this.CURR_STEP > SendState.INIT ) {
			step(SendState.QUIT, null, this.domain, 0);
		}

		// 이제 모든 통신을 종료한다.
		if( STEP_DEBUG ){
			log("step CLOSE");
		}
		this.close();
	}

	public short getErrorType() {
		return this.sendState.getLogLevel();
	}

	/**
	 * 
	 * @param _AGENT_ARRAY_
	 * @param type
	 * @param code
	 * @param msg
	 * @param __LOG_LEVEL__
	 */
	protected void resultLog(String[] agentArray, String type, String code, String msg, short logLevel) {
		resultLog(agentArray, type, code, msg, logLevel, null);
	}
	
	/**
	 * 
	 * @param agentArray
	 * @param type
	 * @param code
	 * @param msg
	 * @param logLevel
	 * @param mgsSeq
	 */
	protected void resultLog(String[] agentArray, String type, String code, String msg, short logLevel, String mgsSeq) {
		synchronized (this.composer.getLock()) {
			this.composer.setProperty(LOG_SEND_TYPE, agentArray[INDEX_OF_SEND_KIND]);
			this.composer.setProperty(LOG_T_TYPE, type);
			this.composer.setProperty(LOG_T_CODE, code);
			this.composer.setProperty(LOG_MAIL_ID, agentArray[INDEX_OF_POST_ID]);

			// 회원아이디는 구분자 인코딩을 하여 추가한다.
			this.TMP_BUFFER.setLength(0);
			DelimConvertor.encodeToBuffer(this.TMP_BUFFER, agentArray[INDEX_OF_MEMBER_ID]);
			this.composer.setProperty(LOG_MEMBER_ID, this.TMP_BUFFER.toString());
			
			this.composer.setProperty(LOG_ROWID, agentArray[INDEX_OF_ROWID]);	// add TMS3.0 ROWID
			this.composer.setProperty(LOG_OS, agentArray[INDEX_OF_OS]);			// add TMS3.1 OS
			this.composer.setProperty(LOG_NOTI_FLAG, agentArray[INDEX_OF_NOTI_FLAG]);			// add TMS3.1 NOTI_FLAG
			
			this.composer.setProperty(LOG_STEP, agentArray[INDEX_OF_STEP]);
			this.composer.setProperty(LOG_T_DATE, Cal.getDate());
			this.composer.setProperty(LOG_MX_RECORD, getConnectHost());
			this.composer.setProperty(LOG_DELAY, String.valueOf(super.getElapsedTime()));
			this.composer.setProperty(LOG_TOKEN_ID, agentArray[INDEX_OF_EMAIL_ENCRYPT]);
			this.composer.setProperty(LOG_DOMAIN, agentArray[INDEX_OF_DOMAIN]);
			this.composer.setProperty(LOG_WORKER, getName());
			this.composer.setProperty(LOG_LIST_TABLE, agentArray[INDEX_OF_LIST_TABLE]);

			if( this.CURR_STEP < SendState.DATA_BODY && logLevel == LOG_LEVEL_N_ERROR && agentArray[INDEX_OF_SPOOL] != null
					&& agentArray[INDEX_OF_SPOOL].length() > 1 ) {
				try {
					SpoolingManager.registSpool(agentArray[INDEX_OF_SPOOL]);
					this.composer.setProperty(LOG_ETC_LOG, msg.concat("=> Que"));
				}
				catch(Exception e) {
					log.error(getName(), "SpoolRegistError:" + e.toString());
					if( STEP_DEBUG )
						log("SpoolRegistError:" + e.toString());
					this.composer.setProperty(LOG_ETC_LOG, msg.concat("=> Que err:").concat(e.toString()));
				}
			}
			else {
				this.composer.setProperty(LOG_ETC_LOG, msg);
			}
			if (eMsLocale.MGS_FLAG) {
				//MGS SEQ 로그 추가
				this.composer.setProperty(LOG_MGS_SEQ, mgsSeq);
			}
			SmtpLogger.put(this.composer);
		}

		if( logLevel == LOG_LEVEL_X_ERROR ) {
			// 규정하지 않은 에러가 발생하면 리포팅을 해야한다.
			// TODO - 리포트를 이메일로 하지말고 어딘가 저장을 해야할 듯하다.
			// Reporter.report(_AGENT_ARRAY_[INDEX_OF_DOMAIN], getName(), "unknown.smtp.response", msg);
		}
	}

	@Override
	public abstract void log(String log);
}
