package com.humuson.tms.common.util;

import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 개인정보(이메일, 휴대폰번호) 마스킹 처리 유틸
 *
 * @author 황집중
 *
 */
public class MaskingUtil {

	private static final String EMAIL_PATTERN	= "^[a-z0-9A-Z._-]*@[a-z0-9A-Z]*.[a-zA-Z.]*$";
	private static final String PHONE_PATTERN	= "(01[016789])(\\d{3,4})(\\d{4})";

	private static final String IP_ADDRESS_V4_PATTERN = "^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$";
	private static final String IP_ADDRESS_V6_PATTERN = "^(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]).){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]).){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))$";
	private static final String IP_ADDRESS_PATTERN ="^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)|(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]).){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]).){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))$";
	private static final int EMAIL_MASKING_MAX	= 3; // 최대 마스킹 개수
	private static final int PHONE_MASKING_GROUP= 2; // 마스킹할 번호 그룹 (ex. 010-1111-2222, 세번째 그룹은 2222)
	private static final int IP_ADDRESS_V4_GROUP = 3;
	private static final int IP_ADDRESS_V6_GROUP = 8;

	/**
	 * 이메일, 휴대폰번호 포맷 상관없이
	 * 마스킹 처리하여 리턴.
	 * 포맷에 맞지 않으면 들어온 문자열 그대로 리턴.
	 *
	 * @param str
	 * @return maskedStr
	 */
	public static String getMaskedStr(String str) {
		if(isEmail(str)) {
			return maskingEamil(str);
		} else if(isPhoneNum(str)) {
			return maskingPhoneNum(str);
		} else if(isIpAddress(str)) {
			return maskingIpAddress(str); // IP
		} else {
			return maskingName(str);
		}
	}

    public static String getMaskedTarget(String str) {
        if(isEmail(str)) {
            return maskingEamil(str);
        } else if(isPhoneNum(str)) {
            return maskingPhoneNum(str);
        } else if(isIpAddress(str)) {
            return maskingIpAddress(str); // IP
        }

        return str;
    }

	/**
	 * 이메일 포맷
	 * 마스킹 처리해서 리턴
	 *
	 * @param str
	 * @return maskedEmailString
	 */
	public static String getMaskedEmailStr(String str) {
		if(isEmail(str)) {
			return maskingEamil(str);
		}
		return str;
	}

	/**
	 * 휴대폰 번호 포맷
	 * 마스킹 처리해서 리턴
	 *
	 * @param str
	 * @return maskedPhoneNumString
	 */
	public static String getMaskedPhoneNumStr(String str) {
		if(isPhoneNum(str)) {
			return maskingPhoneNum(str);
		}
		return str;
	}

	/**
	 * 이메일 포맷 검사
	 *
	 * @param str
	 * @return boolean
	 */
	public static boolean isEmail(String str) {
		return isValid(EMAIL_PATTERN, str);
	}

	/**
	 * 휴대폰 번호 포맷 검사
	 *
	 * @param str
	 * @return boolean
	 */
	public static boolean isPhoneNum(String str) {
		return isValid(PHONE_PATTERN, str);
	}

	/**
	 * 문자열이 정규식에 맞는 포맷인지 검사
	 *
	 * @param regex
	 * @param str
	 * @return boolean
	 */
	private static boolean isValid(String regex, String str) {
		Matcher matcher = Pattern.compile(regex).matcher(str);

		return matcher.matches();
	}

	/**
	 * 이메일 문자열 마스킹 처리
	 *
	 * @param str
	 * @return maskedEmailString
	 */
	private static String maskingEamil(String str) {

		/*
		 * 이메일 주소 포맷
		 * ex. (userId)@(domain)
		 */
		String regex = "\\b(\\S+)+@(\\S+.\\S+)";

		Matcher matcher = Pattern.compile(regex).matcher(str);
		if(matcher.find()) {

			String notAt	= "";
			String asterisk	= "";

			String id = matcher.group(1);

			/*
			 * id를
			 * EMAIL_MASKING_MAX 값 기준으로
			 * 마스킹 처리
			 */
			int length = id.length();
			if(length < EMAIL_MASKING_MAX) {

				char[] c = new char[length];
				Arrays.fill(c, '*');
				return str.replace(id, String.valueOf(c));

			} else if(length == EMAIL_MASKING_MAX) {

				for(int i=0; i < EMAIL_MASKING_MAX-1; i++) {
					notAt		+= "[^@]";
					asterisk	+= "*";
				}
				return str.replaceAll("\\b(\\S+)"+notAt+"+@(\\S+)", "$1"+asterisk+"@$2");

			} else {

				for(int i=0; i < EMAIL_MASKING_MAX; i++) {
					notAt		+= "[^@]";
					asterisk	+= "*";
				}
				return str.replaceAll("\\b(\\S+)"+notAt+"+@(\\S+)", "$1"+asterisk+"@$2");

			}
		}
		return str;
	}

	/**
	 * 휴대폰 번호 문자열 마스킹 처리
	 *
	 * @param str
	 * @return maskedPhoneNumString
	 */
	private static String maskingPhoneNum(String str) {

		/*
		 * 휴대폰 번호 포맷
		 * ex. 010+(3,4,자리)+(4자리)
		 */
		String regex = "(01[016789])(\\d{3,4})(\\d{4})$";

		Matcher matcher = Pattern.compile(regex).matcher(str);
		if(matcher.find()) {

			String replaceGroup = matcher.group(PHONE_MASKING_GROUP);

			char[] c = new char[replaceGroup.length()];

			Arrays.fill(c, '*');
			return str.replace(replaceGroup, String.valueOf(c));
		}

		return str;
	}

	private static String maskingName(String str) {
		char[] c = new char[str.length() - 2];
		Arrays.fill(c, '*');
		StringBuilder sb = new StringBuilder();
		sb.append(str.charAt(0));
		sb.append("*");
		if(str.length() > 2) {
			sb.append(str.substring(2, str.length()));
		}
		return sb.toString();
	}

	public static boolean isIpAddress(String str) {
		return isValid(IP_ADDRESS_PATTERN, str);
	}

	private static String maskingIpAddress(String str) {
		String type = isValid(IP_ADDRESS_V4_PATTERN, str) ? "4" : isValid(IP_ADDRESS_V6_PATTERN, str) ? "6" : null;
		if(type == null) {
			return str;
		}
		String splitPattern = "4".equals(type) ? "." : ":";
		String mask = "4".equals(type) ? "***" : "****";
		int encGroup = "4".equals(type) ? IP_ADDRESS_V4_GROUP - 1 : IP_ADDRESS_V6_GROUP - 1;

		String[] strs = str.split("\\" + splitPattern);
		StringBuilder sb = new StringBuilder();
		for(int i = 0 ; i < strs.length ; i++) {
			if(i != 0) {
				sb.append(splitPattern);
			}
			if(i == encGroup) {
				sb.append(mask);
			} else {
				sb.append(strs[i]);
			}
		}
		return sb.toString();
	}

	public static String getMaskedIpAddressStr(String str) {
		if(isIpAddress(str)) {
			return maskingIpAddress(str);
		}
		return str;
	}

	public static String getMaskedNameStr(String str) {
		return maskingName(str);
	}



	public static void main(String[] args) {

		String email	= "hwang1018kr@humuson.com";
		String phoneNum	= "01012345678";
		String name1 = "이철규3";
		String name2 = "홍길";
		String name3 = "홍길삼동";
		String ip4 = "202.30.64.22";
		String ip6 = "2001:0230:abcd:ffff:0000:0000:ffff:1111";

		String maskedEmail		= MaskingUtil.getMaskedEmailStr(email);
		String maskedPhoneNum	= MaskingUtil.getMaskedPhoneNumStr(phoneNum);
		String maskedName1 = MaskingUtil.maskingName(name1);
		String maskedName2 = MaskingUtil.maskingName(name2);
		String maskedName3 = MaskingUtil.maskingName(name3);
		String maskedIp4 = MaskingUtil.maskingIpAddress(ip4);
		String maskedIp6 = MaskingUtil.maskingIpAddress(ip6);

		System.out.println("masked email		: " + maskedEmail);
		System.out.println("masked phone number	: " + maskedPhoneNum);
		System.out.println("masked name 1	: " + maskedName1);
		System.out.println("masked name 2	: " + maskedName2);
		System.out.println("masked name 3	: " + maskedName3);
		System.out.println("masked ip v4	: " + maskedIp4);
		System.out.println("masked ip v6	: " + maskedIp6);

	}

}
