package com.humuson.tms.common.util;

import java.math.BigDecimal;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.humuson.tms.common.security.HumusonEncryptor;

public class DateConverter {
	
	private static final Logger logger = LoggerFactory.getLogger(HumusonEncryptor.class);
	
	/**
	 * Default 일자 포맷 문자열
	 */
	public static final String DEFAULT_DATE_FORMAT = "yyyyMMdd";
	
	/**
	 * Default Locale
	 */
	public static final Locale DEFAULT_LOCALE = Locale.KOREA;
	
	/**
	 * 입력된 날짜 포맷으로 현재일자 추출
	 * @param format 일자 포맷 문자열
	 * @return 현재일자 문자열
	 */
	public static String getCurrentDate(String format) {
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		DateFormat df = new SimpleDateFormat(format);
		Date date = new Date();

		return getFormattedDate(date, df);
	}
	
	public static String getCurrentDate(String format, String timezone) {
		
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		DateFormat df = new SimpleDateFormat(format);
		df.setTimeZone(TimeZone.getTimeZone(timezone));
		Date date = new Date();
		
		return getFormattedDate(date, df);
	}

	/**
	 * 입력된 일자 객체를 원하는 포맷으로 변환
	 * @param date 일자
	 * @param format 일자 포맷 문자열
	 * @return 포맷 변환된 일자 문자열
	 */ 
	public static String getFormattedDate(String date, String format) {
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		DateFormat df = new SimpleDateFormat(format);
		
		return df.format(date);
	}

	/**
	 * 입력된 일자 객체를 원하는 포맷으로 변환
	 * @param date 일자
	 * @param format 일자 포맷 문자열
	 * @return 포맷 변환된 일자 문자열
	 */
	public static String getFormattedDate(String date, String format, String format2) {
		if (date == null || date.length() == 0)
			return null;
		
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		try {
			return getFormattedDate(convertString2Date(date, format), format2);
		} catch (ParseException e) {
			e.printStackTrace();
			return null;
		}
//		return df.format(date);
	}
	
	public static String getFormattedDate(String date, String format, String format2, String timezone) {
		if (date == null || date.length() == 0)
			return null;
		
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		DateFormat df = new SimpleDateFormat(format);
		
		df.setTimeZone(TimeZone.getTimeZone("UTC"));
		
		try {
			Date d = df.parse(date);
			return getFormattedDate(d, format2, timezone);
		} catch (ParseException e) {
			e.printStackTrace();
			return null;
		}
	}
	
	public static Date convertString2Date(String date, String format) throws ParseException {
		if (date == null || date.length() == 0)
			return null;
		
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		DateFormat df = new SimpleDateFormat(format);
		
		return df.parse(date);
	}
	
	/**
	 * 입력된 일자 객체를 원하는 포맷으로 변환
	 * @param date 일자
	 * @param format 일자 포맷 문자열
	 * @return 포맷 변환된 일자 문자열
	 */
	public static String getFormattedDate(Date date, String format) {
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		DateFormat df = new SimpleDateFormat(format);
		
		return df.format(date);
	}
	
	public static String getFormattedDate(Date date, String format, String timezone) {
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		DateFormat df = new SimpleDateFormat(format);
		
		if (timezone != null && !timezone.equals("")) {
			timezone = "GMT" + timezone;
		}

		df.setTimeZone(TimeZone.getTimeZone(timezone));
		
		return df.format(date);
	}

	/**
	 * 입력된 일자 객체를 원하는 포맷으로 변환
	 * @param date 일자
	 * @param df 일자 포맷 객체
	 * @return 포맷 변환된 일자 문자열
	 */
	public static String getFormattedDate(Date date, DateFormat df) {
		if (logger.isDebugEnabled()) {
			logger.debug("getFormattedDate : {}", df.format(date));
		}
		return df.format(date);
	}
	
	/**
	 * 현재 일자에 입력된 일수를 더함(Plus or minus)
	 * @param days 더하고 싶은 일수
	 * @return 현재 일자에 더해진 일자 문자열
	 */
	public static String changeCurrentDay(int days, String format) {
		
		SimpleDateFormat df = new SimpleDateFormat("yyyyMMdd");
		try {
			Date date = df.parse(format);
			
			Calendar c = Calendar.getInstance();
			c.setTime(date);
			c.add(Calendar.DAY_OF_MONTH, days);
			
			Date d = c.getTime();

			return getFormattedDate(d, df);
		} catch (ParseException e) {
			e.printStackTrace();
			return null;
		}
		
	}
	

	/**
	 * 현재 일자에 입력된 일수를 더함(Plus)
	 * @param days 더하고 싶은 일수
	 * @return 현재 일자에 더해진 일자 문자열
	 */
	public static String addCurrentDays(int days) {
		return addDays(new Date(), days, DEFAULT_DATE_FORMAT);
	}

	/**
	 * 현재 일자에 입력된 일수를 더함(Plus)
	 * @param days 더하고 싶은 일수
	 * @param format 일자 포맷 문자열
	 * @return 현재 일자에 더해진 일자 문자열
	 */
	public static String addCurrentDays(int days, String format) {
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		return addDays(new Date(), days, format);
	}

	/**
	 * 일자에 입력된 일수를 더함(Plus)
	 * @param date 기준 일자 객체
	 * @param days 더하고 싶은 일수
	 * @param format 일자 포맷 문자열
	 * @return 더해진 일자 문자열
	 */
	public static String addDays(Date date, int days, String format) {
		DateFormat df = new SimpleDateFormat(format);
		
		return addDays(date, days, df);
	}
	
	/**
	 * 일자에 입력된 일수를 더함(Plus)
	 * @param date 기준 일자 객체
	 * @param days 더하고 싶은 일수
	 * @param df 일자 포맷 객체
	 * @return 더해진 일자 문자열
	 */
	public static String addDays(Date date, int days, DateFormat df) {
		Calendar c = Calendar.getInstance();
		c.setTime(date);
		c.add(Calendar.DAY_OF_MONTH, days);
		
		Date d = c.getTime();

		return getFormattedDate(d, df);
	}

	public static String addCurrentMonth(int month, String format) {
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		return addMonth(new Date(), month, format);
	}
	
	public static String addMonth(String date, String format, int month, String target) {
		DateFormat df = new SimpleDateFormat(target);
		Date d;
		
        try {
	        d = convertString2Date(date, format);
        } catch (ParseException e) {
	        e.printStackTrace();
	        return null;
        }
		
		return addMonth(d, month, df);
	}
	
	public static String addMonth(Date date, int month, String format) {
		DateFormat df = new SimpleDateFormat(format);
		
		return addMonth(date, month, df);
	}
	
	public static String addMonth(Date date, int month, DateFormat df) {
		Calendar c = Calendar.getInstance();
		c.setTime(date);
		c.add(Calendar.MONTH, month);
		
		return getFormattedDate(c.getTime(), df);
	}
	
	public static String calcDaysFromAddMonth(String fromDate, String dateFormat, int month) {

        
        TimeZone tz;
		DateFormat sdf = new SimpleDateFormat(dateFormat);
		tz = TimeZone.getTimeZone("UTC");
		sdf.setTimeZone(tz);
        String presentStr = sdf.format(new Date().getTime());
        
		DateFormat df = new SimpleDateFormat(dateFormat);
		
        try {
        	
        	Date pDate = df.parse(presentStr);
        	
        	Date fDate = df.parse(fromDate);
        	
	        String toDate = addMonth(fDate, month, df);
	        
	        Date tDate = df.parse(toDate);
	        
	        long diff = tDate.getTime() - pDate.getTime();
	        if (logger.isDebugEnabled())
	        	logger.debug("calc days diff : {}", diff);
	        boolean isMinus = false;
	        if(diff < 0) isMinus = true;
	        
	        long diffDays = diff / (24 * 60 * 60 * 1000);
	        
	        if(isMinus) return "-" + Long.toString(diffDays);
	        else return Long.toString(diffDays);
	        
        } catch (ParseException e) {
	        e.printStackTrace();
	        return null;
        }
		
	}
	
	public static String calcDaysFromDate(String fromDate, String toDate, String dateFormat) {
		
		try {
			
			if(fromDate.length() > 0 && toDate.length() > 0 && dateFormat.length() > 0) {
				DateFormat df = new SimpleDateFormat(dateFormat);
				
				Date fDate = df.parse(fromDate);
		    	Date tDate = df.parse(toDate);
		    	
		    	long diff = tDate.getTime() - fDate.getTime();
		        
		    	boolean isMinus = false;
		        if(diff < 0) isMinus = true;
		        
		        long diffDays = diff / (24 * 60 * 60 * 1000);
		        
		        if(isMinus) return  Long.toString(diffDays);
		        else return Long.toString(diffDays);
		        
			} else {
				return "-";
			}
			
		} catch (ParseException e) {
	        e.printStackTrace();
	        return null;
        }
		
	}
	
	public static String getLastPayedDate(String fromDate, String dateFormat, int month) {
 
		DateFormat df = new SimpleDateFormat(dateFormat);
		
        try {
        	
        	Date fDate = df.parse(fromDate);
	        
	        return addMonth(fDate, month, df);
	        
        } catch (ParseException e) {
	        e.printStackTrace();
	        return null;
        }
		
	}
	
	public static String getRegistApprovalDate(String registDate) {
		
		try {
			
			String initDate = registDate.substring(0,6) + "01";
			if (logger.isDebugEnabled()) logger.debug("getRegistApprovalDate initDate : {}", initDate);
			
			
			String addTwoMonthDate = addMonth(initDate, "yyyyMMdd", 2, "yyyyMMdd");
			if (logger.isDebugEnabled()) logger.debug("getRegistApprovalDate addTwoMonthDate : {}", addTwoMonthDate);
			
			DateFormat df = new SimpleDateFormat("yyyyMMdd");
			Date approvalDate = df.parse(addTwoMonthDate);
			 
			String resultStr = addDays(approvalDate, -1, "yyyyMMdd"); 
			if (logger.isDebugEnabled()) logger.debug("getRegistApprovalDate resultStr : {}", resultStr);
			return resultStr;

			
		} catch (ParseException e) {
	        e.printStackTrace();
	        return null;
        }
		
	}
	
	/**
	 * 유효한 일자인지 체크
	 * @param date 일자 문자열
	 * @param format 일자 포맷 문자열
	 * @return 유효한 일자 여부
	 */
	public static boolean valid(String date, String format) {
		if (date == null || date.length() == 0)
			return false;

		DateFormat df = new SimpleDateFormat(format, DEFAULT_LOCALE);
		return valid(date, df);
	}
	
	/**
	 * 유효한 일자인지 체크
	 * @param date 일자 문자열
	 * @param format 일자 포맷 문자열
	 * @param locale Locale
	 * @return 유효한 일자 여부
	 */
	public static boolean valid(String date, String format, Locale locale) {
		if (date == null || date.length() == 0)
			return false;

		if (locale == null)
			locale = DEFAULT_LOCALE;
		
		DateFormat df = new SimpleDateFormat(format, locale);
		return valid(date, df);
	}
	
	/**
	 * 유효한 일자인지 체크
	 * @param date 일자 문자열
	 * @param df 일자 포맷 객체
	 * @return 유효한 일자 여부
	 */
	public static boolean valid(String date, DateFormat df) {
		Date d = null;

		if (date == null || date.length() == 0)
			return false;

		try {
			d = df.parse(date);
		} catch (ParseException e) {
			e.printStackTrace();
			return false;
		}

		if (!df.format(d).equals(date)) {
			return false;
		}

		return true;
	}

	/**
	 * 첫번째 일자가 두번째 일자보다 앞선 일자인지 체크 
	 * @param date1 일자1
	 * @param date2 일자2
	 * @param format 일자 포맷 문자열
	 * @return 첫번째 일자가 두번째 일자보다 앞선 일자인지 여부
	 */
	public static boolean before(String date1, String date2, String format) {
		if (date1 == null || date1.length() == 0)
			return false;

		if (date2 == null || date2.length() == 0)
			return false;

		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		DateFormat df = new SimpleDateFormat(format, Locale.KOREA);
		return before(date1, date2, df);
	}

	/**
	 * 첫번째 일자가 두번째 일자보다 앞선 일자인지 체크 
	 * @param date1 일자1
	 * @param date2 일자2
	 * @param locale Locale
	 * @return 첫번째 일자가 두번째 일자보다 앞선 일자인지 여부
	 */
	public static boolean before(String date1, String date2, Locale locale) {
		if (date1 == null || date1.length() == 0)
			return false;

		if (date2 == null || date2.length() == 0)
			return false;

		if (locale == null)
			locale = DEFAULT_LOCALE;

		DateFormat df = new SimpleDateFormat(DEFAULT_DATE_FORMAT, locale);
		return before(date1, date2, df);
	}

	/**
	 * 첫번째 일자가 두번째 일자보다 앞선 일자인지 체크 
	 * @param date1 일자1
	 * @param date2 일자2
	 * @param format 일자 포맷 문자열
	 * @param locale Locale
	 * @return 첫번째 일자가 두번째 일자보다 앞선 일자인지 여부
	 */
	public static boolean before(String date1, String date2, String format, Locale locale) {
		if (date1 == null || date1.length() == 0)
			return false;

		if (date2 == null || date2.length() == 0)
			return false;
		
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;

		if (locale == null)
			locale = DEFAULT_LOCALE;

		DateFormat df = new SimpleDateFormat(format, locale);
		return before(date1, date2, df);
	}

	/**
	 * 첫번째 일자가 두번째 일자보다 앞선 일자인지 체크 
	 * @param date1 일자1
	 * @param date2 일자2
	 * @param df 일자 포맷 객체
	 * @return 첫번째 일자가 두번째 일자보다 앞선 일자인지 여부
	 */
	public static boolean before(String date1, String date2, DateFormat df) {
		if (date1 == null || date1.length() == 0)
			return false;

		if (date2 == null || date2.length() == 0)
			return false;

		Date d1 = null;
		Date d2 = null;

		try {
			d1 = df.parse(date1);
			d2 = df.parse(date2);

			if ((!df.format(d1).equals(date1))
					|| (!df.format(d2).equals(date2))) {
				return false;
			}

			if (d1.after(d2))
				return true;
			else
				return false;
		} catch (ParseException e) {
			e.printStackTrace();
			return false;
		}
	}
	
	public static String addDot(String date1) {
		if (date1 == null || date1.length() != 8)
			return date1;
		
		return date1.substring(0,4) +"."+ date1.substring(4,6) +"."+ date1.substring(6,8);
	}
	
	/**
	 * 일자를 입력받아서 해당하는 요일을 return
	 *   : default "ko"
	 * @param date
	 * @return
	 */
	public static String getDayOfWeek(String date) {
		return getDayOfWeek(date, "ko");
	}
	
	/**
	 * 일자를 입력받아서 해당하는 요일을 return
	 * @param date
	 * @param locale
	 * @return
	 */
	public static String getDayOfWeek(String date, String locale) {
//		if (date == null || date.length() != 10) {
		if (date == null) {
			return null;
		}
		
		date = date.replace(".", "");
		date = date.replace("-", "");
		date = date.replace("_", "");
		
		if(date.length() < 8){
			return null;
		}
			
		int year = Integer.parseInt(date.substring(0, 4));
		int month = Integer.parseInt(date.substring(4, 6))-1;
		int dates = Integer.parseInt(date.substring(6, 8));
		
		Calendar c = new GregorianCalendar(year, month, dates);
		int dayOfWeek = c.get(Calendar.DAY_OF_WEEK);    // 6=Friday
		
		String[] week = getWeekArray(locale);
		
		return week[dayOfWeek-1];
	}
	
	
	public static String getDayOfWeek(String date, String format, String locale, String timezone) {
		if (date == null || date.length() == 0)
			return null;
		
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		DateFormat df = new SimpleDateFormat(format);
		
		df.setTimeZone(TimeZone.getTimeZone("UTC"));
		
		try {
			Date d = df.parse(date);
			
			if (timezone != null && !timezone.equals("")) {
				timezone = "GMT" + timezone;
			}

			df.setTimeZone(TimeZone.getTimeZone(timezone));
			
			return getDayOfWeek(df.format(d), locale);
			
		} catch (ParseException e) {
			e.printStackTrace();
			return null;
		}
	}
	
	/**
	 * 특정 locale에 맞는 요일의 문자열 array를 return
	 * @param locale
	 * @return
	 */
	public static String[] getWeekArray(String locale) {
		String[] week = {"sun","mon","tue","wed","thu","fri","sat"};
		return week;
	}
	
	/**
	 * 특정 locale에 맞는 월의 문자열 String을 return
	 * @param locale
	 * @return
	 */
	public static String getMonthChartString(String locale) {
		StringBuilder month = new StringBuilder();
		month.append("['1월', '2월', '3월', '4월', '5월', '6월', '7월', '8월', '9월', '10월', '11월', '12월']");
		return month.toString();
	}
	
	public static String convertMilli2String(long t) {
		return String.format("%d min, %d sec", 
						TimeUnit.MILLISECONDS.toMinutes(t),
						TimeUnit.MILLISECONDS.toSeconds(t) - 
						TimeUnit.MINUTES.toSeconds(TimeUnit.MILLISECONDS.toMinutes(t))
		);
	}
	
	/**
	 * 초(second)를 HH:mm:ss 포맷으로 변환
	 * @param second
	 * @return
	 */
	public static String convertSecond2Date(int second) {
		return convertSecond2Date((long) second);
	}
	
	/**
	 * 초(second)를 HH:mm:ss 포맷으로 변환
	 * @param second
	 * @return
	 */
	public static String convertSecond2Date(BigDecimal second) {
		return convertSecond2Date(second.longValue());
	}
	
	/**
	 * 초(second)를 HH:mm:ss 포맷으로 변환
	 * @param second
	 * @return
	 */
	public static String convertSecond2Date(long second) {
		long hh = second/3600;
		long mm = (second % 3600) / 60;
		long ss = ((second % 3600) % 60);
		
		return String.format("%02d:%02d:%02d", hh, mm, ss);
	}
	
	public static String convertSecond2DayTime(String secTime, String dayUnit) {

		long second = Long.parseLong(secTime);
		double day = 0;
	
		if(second >= 86400) {
			day = Math.floor(second / 86400);
			second = second % 86400;
		}
		
		long hh = second/3600;
		long mm = (second % 3600) / 60;
		long ss = ((second % 3600) % 60);
			
		if(day > 0) {
			return String.format("%d%s %02d:%02d:%02d", (int)day, dayUnit, hh, mm, ss);	
		} else {
			return String.format("%02d:%02d:%02d", hh, mm, ss);
		}
		
	}
	
	public static String convertUtcTime(String date, String format, String gmt) {
		if (date == null || date.length() == 0)
			return null;
		
		
		String gmtKey = "UTC";
		if (gmt != null && !gmt.equals("")) {
			gmtKey = "GMT" + gmt;
		}
		
		if (format == null || format.length() == 0)
			format = DEFAULT_DATE_FORMAT;
		
		DateFormat df = new SimpleDateFormat(format);
		df.setTimeZone(TimeZone.getTimeZone(gmtKey));
		
		Date d = null;
		try {
			d = df.parse(date);
			
			df.setTimeZone(TimeZone.getTimeZone("UTC"));
			return df.format(d);
		} catch (ParseException e) {
			logger.error("convertUtcTime error", e);
			return null;
		}
	}
	/*
	public static String getUtcCurrentTime() {
		
		 return getUtcBeforeHourTime(0);
	}
	*/
	public static String getUtcCurrentTime() {
		return getUtcCurrentTime("GMT+09:00");
	}
	
	public static String getUtcBeforeHourTime(int hour) {
		 TimeZone tz;
		 Date date = new Date();
		 DateFormat df = new SimpleDateFormat("yyyyMMddHH");
		    
		 tz = TimeZone.getTimeZone("UTC"); 
		 df.setTimeZone(tz);
		 return df.format(date.getTime() + (60 * 60 * hour * 1000));
		 
	}
	
	
	public static String getUtcCurrentTime(String timezone, String format) {
		Date date = new Date();
		 
		DateFormat df = new SimpleDateFormat(format);
		    
		df.setTimeZone(TimeZone.getTimeZone(timezone));
		return df.format(date);
	}
	
	public static String getUtcCurrentTime(String gmt) {
		
		String gmtKey = "UTC";
		if (gmt != null && !gmt.equals("")) {
			gmtKey = "GMT" + gmt;
		}
				
		SimpleDateFormat sdf = new SimpleDateFormat( );
        sdf.applyPattern("yyyyMMdd");
        sdf.setTimeZone(TimeZone.getTimeZone(gmtKey));
        return sdf.format( new Date().getTime() );
	}
	
	public static String getUtcCurrentTime(String gmt, int hour) {
		
		String gmtKey = "UTC";
		if (gmt != null && !gmt.equals("")) {
			gmtKey = "GMT" + gmt;
		}
				
		SimpleDateFormat sdf = new SimpleDateFormat( );
        sdf.applyPattern("yyyyMMddHH");
        sdf.setTimeZone(TimeZone.getTimeZone(gmtKey));
        
        Calendar c = Calendar.getInstance();
		c.add(Calendar.HOUR, hour);
		Date d = c.getTime();
        
        return sdf.format( d.getTime());
	}
	
	 public static String changeFileNameType(String format) {
		 
		 String changeTerm = "";
		 format = format.replace(".", "");
		 if(format.length() != 6) {
			 return null;
		 }
		 
		 changeTerm = format.substring(0, 4);
		 int month = Integer.parseInt(format.substring(5));
		 
		 switch (month) {
			case 1:
				changeTerm += "January";
				break;
			case 2:
				changeTerm += "February";
				break;
			case 3:
				changeTerm += "March";
				break;
			case 4:
				changeTerm += "April";
				break;
			case 5:
				changeTerm += "May";
				break;
			case 6:
				changeTerm += "June";
				break;
			case 7:
				changeTerm += "July";
				break;
			case 8:
				changeTerm += "August";
				break;
			case 9:
				changeTerm += "September";
				break;
			case 10:
				changeTerm += "October";
				break;
			case 11:
				changeTerm += "November";
				break;
			case 12:
				changeTerm += "December";
				break;
			default:
				break;
		}
		 
		 
		 return changeTerm;
	 }
	
	public static void main(String[] args) throws Exception {
		if (logger.isDebugEnabled()) logger.debug(DateConverter.getFormattedDate("20120131055010","yyyyMMddHHmmss", "yyyy.MM.dd HH:mm:ss", "+09:00"));
    }
}
