package com.humuson.tms.batch.writer;

import static com.humuson.tms.batch.job.constrants.JobExecutionContextConstrants.KEY_APP_GRP_ID;
import static com.humuson.tms.batch.job.constrants.JobParamConstrants.APP_GRP_KEY;
import static com.humuson.tms.constrants.PushResponseConstants.UNKNOWN_FAIL;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.core.launch.JobExecutionNotRunningException;
import org.springframework.batch.core.launch.NoSuchJobExecutionException;
import org.springframework.batch.item.ItemWriter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;

import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.domain.PushQueue;
import com.humuson.tms.batch.domain.PushResult;
import com.humuson.tms.batch.service.AppUserRemover;
import com.humuson.tms.batch.service.PushInfoService;
import com.humuson.tms.batch.service.PushResultService;
import com.humuson.tms.batch.service.PushSendService;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * 
 * @author hyogun
 *
 */
@Slf4j
public class PushSendWriter implements ItemWriter<PushQueue>, StepExecutionListener {
	
	private static final String MASS = "MASS";
	@Autowired protected PushInfoService<App, PushMessage> pushInfoService;
	@Autowired protected JdbcTemplate jdbcTemplate;
	@Autowired protected PushSendService<List<PushQueue>, List<PushResult>> mgsPushSendServiceImpl;
	@Autowired protected AppUserRemover appUserRemover;
	@Autowired protected PushResultService pushResultService;
	@Setter protected String appGrpKey;
	@Setter private String insertInAppMsg;

	protected int appGrpId;
	protected App appInfo;
	@Value("#{config['use.resend.gcm.for.private.failed']}")
	private boolean useResendGcm;
	@Value("#{config['tms.private.inactive.publish']}")
	private boolean privateInactivePublish;
	@Value("#{config['use.wakeup.gcm']}")
	private boolean useWakeupGcm;
	
	@Value("#{config['not.registed.android.immediately.update']}")
	private boolean immediatelyUpdate;
	
	@Value("#{config['use.mgs.public.push']}")
	protected boolean useMgsPublicPush;
	
	@Override
	public void write(List<? extends PushQueue> rawData) throws Exception {
		
		if (isSendStop()) {	
			log.info("PushSendWriter Stop!!");
		}
		
		long startTime = System.currentTimeMillis();
		
		if (useMgsPublicPush) {
			@SuppressWarnings("unchecked")
			List<PushQueue> rawData2 = (List<PushQueue>) rawData;
			pushResultService.batchUpdateSendingStatus(rawData2);
			log.info("batch push que update");
		}
		
		List<Object[]> inAppMsgParams = new ArrayList<Object[]>();
		Map<String, List<PushQueue>> msgUidMap = new HashMap<String, List<PushQueue>>();
		
		for (PushQueue pushQue : rawData) {
			List<PushQueue> pushQueList = msgUidMap.get(pushQue.getPushMessage().getMsgUid());
			
			if (pushQueList == null || pushQueList.isEmpty()) {
				pushQueList = new ArrayList<PushQueue>();
			}
			
			pushQueList.add(pushQue);
			msgUidMap.put(pushQue.getPushMessage().getMsgUid(), pushQueList);
			
			inAppMsgParams.add(new Object[]{
					pushQue.getPushMessage().getExpireDate(), pushQue.getAppGrpId(), pushQue.getDeviceId(), 
					MASS.equals(pushQue.getCustId()) ? "" : pushQue.getCustId(), pushQue.getReqUid(), pushQue.getPushMessage().getMsgUid(), pushQue.getPushType(), 
					pushQue.getPushMessage().getPushTitle(), pushQue.getPushMessage().getPushMsg(), pushQue.getPushMessage().getPushValue(), pushQue.getPushMessage().getPushKey(), 
					pushQue.getPushMessage().getMsgType(), pushQue.getPushMessage().getInAppContent(),
					pushQue.getPushMessage().getPushImg(), pushQue.getPushMessage().getEtc1(),
					pushQue.getPushMessage().getEtc2(), pushQue.getPushMessage().getEtc3(),
					pushQue.getPushMessage().getEtc4(), pushQue.getPushMessage().getEtc5(),
					pushQue.getPushMessage().getEtc6()
			});
		}
		
		List<PushResult> resList = null;
		try {
			// step1. 수신함 적재
			this.insertInAppMsg(inAppMsgParams);
			log.info("insertUserMsg End [count:{}, elapseTime:{}]", 
					inAppMsgParams.size(), System.currentTimeMillis() -startTime);
			startTime = System.currentTimeMillis();
			
			// step2. 푸시 전송
			resList = this.sendRequest(msgUidMap);
			log.info("push send End [count:{}, elapseTime:{}]", 
					resList.size(), System.currentTimeMillis() -startTime);
			
		} catch (Exception e) {
			log.error("Push Send Error ", e);
			
			if (resList == null) {
				resList = new ArrayList<PushResult>();
				StringBuilder sb = new StringBuilder();
				for (PushQueue pushQueue : rawData) {
					sb.setLength(0);
					sb.append(pushQueue.getPushId()).append("&&")
							.append(pushQueue.getDeviceId()).append("&&")
							.append(pushQueue.getReqUid()).append("&&")
							.append(pushQueue.getCustId()).toString();
					resList.add(new PushResult(appGrpId, UNKNOWN_FAIL, 
							sb.toString(), "U",pushQueue.getRowId()));
				}
			}
			
			throw e;
		} finally {
			// step3. 결과 처리
			startTime = System.currentTimeMillis();
			this.pushResultService.batchUpdatePushResult(resList);
			log.info("update push result End [count:{}, elapseTime:{}]", 
					resList.size(), System.currentTimeMillis() -startTime);
		}
	}

	private List<PushResult> sendRequest(Map<String, List<PushQueue>> msgUidMap) {
		List<PushResult> resList = new ArrayList<PushResult>();
		Set<String> msgUidSet = msgUidMap.keySet();
		
		List<PushResult> tmpResult = new ArrayList<PushResult>();
		List<PushQueue> oneToOneList = new ArrayList<PushQueue>();
		
		for (String msgUid : msgUidSet) {
			if (msgUidMap.get(msgUid).size() == 1) {
				oneToOneList.addAll(msgUidMap.get(msgUid));
			} else {
				List<PushQueue> msgUidQueList = msgUidMap.get(msgUid);
				try {
					tmpResult = this.mgsPushSendServiceImpl.request(msgUidQueList, 
							msgUidQueList.get(0).getPushMessage(), 
							useResendGcm, privateInactivePublish, useWakeupGcm);
					log.info("push send result : {} , msgUid :{}", tmpResult.size(), msgUid);
				} catch (Exception e) {
					log.error("Push Request failed", e);
					tmpResult.clear();
					StringBuilder sb = new StringBuilder();
					for (PushQueue pushQueue : msgUidQueList) {
						sb.setLength(0);
						sb.append(pushQueue.getPushId()).append("&&")
								.append(pushQueue.getDeviceId()).append("&&")
								.append(pushQueue.getReqUid()).append("&&")
								.append(pushQueue.getCustId()).toString();
						tmpResult.add(new PushResult(appGrpId, UNKNOWN_FAIL, sb.toString(), appInfo.getAppOsMap().get(pushQueue.getAppId()),pushQueue.getRowId()));
					}
				} 
				resList.addAll(tmpResult);
			}
		}
		
		if (!oneToOneList.isEmpty()) {
			log.info("oneToOneList size: {}", oneToOneList.size());
			tmpResult = mgsPushSendServiceImpl.request(oneToOneList, useResendGcm, 
					privateInactivePublish, useWakeupGcm);
			resList.addAll(tmpResult);
		}
		
		return resList;
	}
	
	protected boolean isSendStop() 
			throws NoSuchJobExecutionException, JobExecutionNotRunningException {
		return false;
	}
	
	protected void insertInAppMsg(List<Object[]> inAppMsgParams) {
		if (!inAppMsgParams.isEmpty()) {
			// insert user msg
			try {
				jdbcTemplate.batchUpdate(this.insertInAppMsg, inAppMsgParams);
			} catch (Exception e) {
				log.error("insert inAppMsg error", e);
			}
			
		}
	}
	
	@Override
	public void beforeStep(StepExecution stepExecution) {
		appGrpKey = stepExecution.getJobExecution().getJobParameters().getString(APP_GRP_KEY);
		appGrpId = (int)stepExecution.getJobExecution().getExecutionContext().getLong(KEY_APP_GRP_ID);
		appInfo = pushInfoService.getAppInfo(appGrpKey);
		mgsPushSendServiceImpl.init(appInfo);
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		mgsPushSendServiceImpl.close();
		return null;
	}
}
