package com.humuson.tms.batch.writer;

import static com.humuson.tms.batch.job.constrants.JobParamConstrants.APP_GRP_KEY;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemWriter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import com.google.android.gcm.ccs.server.CcsConnectionManager;
import com.google.android.gcm.server.Message;
import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.domain.PushQueue;
import com.humuson.tms.batch.domain.PushResult;
import com.humuson.tms.batch.service.AppUserRemover;
import com.humuson.tms.batch.service.GcmHttpService;
import com.humuson.tms.batch.service.MqProducer;
import com.humuson.tms.batch.service.PushInfoService;
import com.humuson.tms.batch.service.PushResultService;
import com.humuson.tms.batch.service.PushSendService;
import com.humuson.tms.batch.service.impl.GcmCcsService;
import com.humuson.tms.common.model.mq.PushSendType;
import com.humuson.tms.constrants.PushResponseConstants;
import com.humuson.tms.constrants.StatusType;
import com.humuson.tms.mq.model.MgsPush;
import com.humuson.tms.mq.model.MgsPush.Response.ResponsePayload;
import com.humuson.tms.mq.model.MgsPush.Response.ReturnCode;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * Private 전송 이후 public 재발송 및 만료 처리 담당
 * @author hyogun
 *
 */
@Slf4j
public class PushMqReSendWriter implements ItemWriter<PushQueue>, StepExecutionListener {
	
	@Autowired protected PushInfoService<App, PushMessage> pushInfoService;
	@Autowired protected PushSendService<List<PushQueue>,List<PushResult>> mgsPushSendServiceImpl;
	@Autowired protected PushResultService pushResultService;
	@Autowired protected AppUserRemover appUserRemover;
	@Setter protected String appGrpKey;
	protected App appInfo;
	
	@Value("#{config['use.resend.gcm.for.private.failed']}")
	private boolean useResendGcm;
	
	@Value("#{config['not.registed.android.immediately.update']}")
	private boolean immediatelyUpdate;
	
	@Value("#{config['send.gcm.type']}")
	private String gcmType;
	
	@Autowired
	GcmCcsService gcmCcsService;
	
	@Autowired
	@Qualifier("gcmHttpServiceImpl")
	GcmHttpService<PushResult,PushQueue> gcmHttpServiceImpl;
	
	@Autowired
	@Qualifier(value = "mqCampResProducer")
	private MqProducer mqCampResProducer;
	
	@Autowired
	@Qualifier(value = "mqAutoResProducer")
	private MqProducer mqAutoResProducer;
	
	@Autowired
	CcsConnectionManager ccsConnectionManager;
	
	@Value("#{config['sending.status.mq.response.insert']}")
	protected boolean isSendDataInsert;
	
	/**
	 * 
	 */
	@Override
	public void write(List<? extends PushQueue> rawData) throws Exception {
		long startTime = System.currentTimeMillis();
		Map<String, List<PushQueue>> reSendGcmMsgUidMap = new HashMap<String, List<PushQueue>>();
		
		MgsPush.Response.Builder mqAutoResponseBuilder = MgsPush.Response.newBuilder();
		MgsPush.Response.Builder mqCampResponseBuilder = MgsPush.Response.newBuilder();
		ResponsePayload.Builder  responsePayloadBuilder = ResponsePayload.newBuilder();		
		
		for (PushQueue pushQue : rawData) {
//			log.info("pushQue [s:{}, e:{}]", pushQue.getSendStatus(), pushQue.getErrorCode());
			
			if ( App.IOS.equals(pushQue.getAppOs()) ){
			
				responsePayloadBuilder.setId(pushQue.getPushId()+"&&"+pushQue.getDeviceId()+"&&"+pushQue.getReqUid()+"&&"+pushQue.getCustId());
				responsePayloadBuilder.setReturnCode(ReturnCode.TTL_EXPIRED);
				responsePayloadBuilder.setServerId(pushQue.getServerId());
				
				if(pushQue.getReqUid().startsWith("CC")){
					mqCampResponseBuilder.addResPayload(responsePayloadBuilder.build());
				}else if(pushQue.getReqUid().startsWith("AA")){
					mqAutoResponseBuilder.addResPayload(responsePayloadBuilder.build());
				}
			
			} else if (useResendGcm) {
				List<PushQueue> pushQueList = reSendGcmMsgUidMap.get(pushQue.getPushMessage().getMsgUid());
				if (pushQueList == null)
					pushQueList = new ArrayList<PushQueue>();
				pushQueList.add(pushQue);
				reSendGcmMsgUidMap.put(pushQue.getPushMessage().getMsgUid(), pushQueList);
			} else {
				
				responsePayloadBuilder.setId(pushQue.getPushId()+"&&"+pushQue.getDeviceId()+"&&"+pushQue.getReqUid()+"&&"+pushQue.getCustId());
				responsePayloadBuilder.setReturnCode(ReturnCode.TTL_EXPIRED);
				responsePayloadBuilder.setServerId(pushQue.getServerId());
				
				if(pushQue.getReqUid().startsWith("CC")){
					mqCampResponseBuilder.addResPayload(responsePayloadBuilder.build());
				}else if(pushQue.getReqUid().startsWith("AA")){
					mqAutoResponseBuilder.addResPayload(responsePayloadBuilder.build());
				}
			}
			
		}
		
		if(!mqCampResponseBuilder.getResPayloadList().isEmpty()){
			mqCampResProducer.send(mqCampResponseBuilder.build());
		}
		
		if(!mqAutoResponseBuilder.getResPayloadList().isEmpty()){
			mqAutoResProducer.send(mqAutoResponseBuilder.build());
		}
		
		resendGcm(startTime, reSendGcmMsgUidMap);
	}

	private void resendGcm(long startTime, Map<String, List<PushQueue>> reSendGcmMsgUidMap) {
		// GCM 재전송
		List<PushResult> resList = new ArrayList<PushResult>();
		
		
		
		if (useResendGcm
				&& !reSendGcmMsgUidMap.keySet().isEmpty()) {
			
			Set<String> keySet = reSendGcmMsgUidMap.keySet();
			
			for (String key : keySet) {
				List<PushQueue> pushQueList = reSendGcmMsgUidMap.get(key);
				if("http".equalsIgnoreCase(gcmType)){
					if (pushQueList.size() == 1) {
						resList.add(gcmHttpServiceImpl.sendGcmMessage(pushQueList.get(0),appInfo.getGcmApiKey()));
					} else {
						resList.addAll(gcmHttpServiceImpl.sendGcmOne2OneList(pushQueList, appInfo.getGcmApiKey()));
					}
				}else if ("xmpp".equalsIgnoreCase(gcmType)){
					StringBuilder sb = new StringBuilder();
					for (PushQueue que : pushQueList) {
						sb.setLength(0);
						String id = sb.append(que.getPushId()).append("&&")
								.append(que.getDeviceId()).append("&&")
								.append(que.getReqUid()).append("&&")
								.append(que.getCustId()).toString();
						String ccsMessage = gcmCcsService.makeCcsMessage(que,que.getPushMessage(),appInfo.getAppGrpId(),PushSendType.MQ);
						try {
							gcmCcsService.sendDownstreamMessage(ccsMessage);							
							
						} catch (Exception e) {
							// TODO Auto-generated catch block
							
							log.error("exception in resendGcm (XMPP) : {}", que.getPushId(), e);
							resList.add(new PushResult(appInfo.getAppGrpId(), PushResponseConstants.ERROR_UNAVAILABLE,sb.toString(),
									App.ANDROID, que.getRowId()));
							gcmCcsService.setHitCount(0);
							gcmCcsService.reset();
							continue;
						}
						resList.add(new PushResult(appInfo.getAppGrpId(), PushResponseConstants.SENDING_CCS,
								sb.toString(),App.ANDROID, que.getRowId()));
					}
				}
			}
		}
		
		if (!resList.isEmpty() && "http".equalsIgnoreCase(gcmType)) {
			try {
				ArrayList<Object[]> prvPushMqDeleteList = new ArrayList<Object[]>();
				for (PushResult result : resList) {
					try {
						prvPushMqDeleteList.add(new Object[]{result.getReqUid()});
					} catch (Exception e2) {
						log.error("push result update error {}, [{}]", e2, result.toString());
					}
				}
				
				pushResultService.deleteMqResendList(prvPushMqDeleteList);
			} catch (Exception e) {
				log.error("update push result ", e);
				for (PushResult result : resList) {
					try {
						pushResultService.deleteMqResend(result.getReqUid());
					} catch (Exception e2) {
						log.error("push result update error {}, [{}]", e2, result.toString());
					}
				}
			}
			
			log.info("push ReSend End [count:{}, elapseTime:{}]", 
					resList.size(), System.currentTimeMillis() -startTime);
		}else if(!resList.isEmpty() && "xmpp".equalsIgnoreCase(gcmType)  ){
			
			ArrayList<Object[]> prvPushMqUpdateList = new ArrayList<Object[]>();
			ArrayList<Object[]> prvPushMqDeleteList = new ArrayList<Object[]>();
			
			
			
			if(isSendDataInsert){
				MgsPush.Response.Builder mqAutoResponseBuilder = MgsPush.Response.newBuilder();
				MgsPush.Response.Builder mqCampResponseBuilder = MgsPush.Response.newBuilder();
				ResponsePayload.Builder  responsePayloadBuilder = ResponsePayload.newBuilder();		
				
				for (PushResult result : resList) {
					responsePayloadBuilder.clear();
					responsePayloadBuilder.setId(result.getPushId()+"&&"+result.getDeviceId()+"&&"+result.getReqUid()+"&&"+result.getCustId());
					responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.valueOf(Integer.parseInt(result.getErrorCode())));
					if(result.getReqUid().startsWith("CC")){
						mqCampResponseBuilder.addResPayload(responsePayloadBuilder.build());
						prvPushMqUpdateList.add(new Object[]{result.getReqUid()});
					}else if(result.getReqUid().startsWith("AA")){
						mqAutoResponseBuilder.addResPayload(responsePayloadBuilder.build());
						prvPushMqUpdateList.add(new Object[]{result.getReqUid()});
					}else {
						prvPushMqDeleteList.add(new Object[]{result.getReqUid()});
					}
					
				}
				
				pushResultService.updateMqResendList(prvPushMqUpdateList);
				pushResultService.deleteMqResendList(prvPushMqDeleteList);
				
				if(!mqCampResponseBuilder.getResPayloadList().isEmpty()){
					mqCampResProducer.send(mqCampResponseBuilder.build());
				}
				
				if(!mqAutoResponseBuilder.getResPayloadList().isEmpty()){
					mqAutoResProducer.send(mqAutoResponseBuilder.build());
				}
			}else{
				
				for (PushResult result : resList) {
					if(result.getReqUid().startsWith("CC")){
						prvPushMqUpdateList.add(new Object[]{result.getReqUid()});
					}else if(result.getReqUid().startsWith("AA")){
						prvPushMqUpdateList.add(new Object[]{result.getReqUid()});
					}else {
						prvPushMqDeleteList.add(new Object[]{result.getReqUid()});
					}
				}
				pushResultService.updateMqResendList(prvPushMqUpdateList);
				pushResultService.deleteMqResendList(prvPushMqDeleteList);
			}
		}
	}
	
	@Override
	public void beforeStep(StepExecution stepExecution) {
		appGrpKey = stepExecution.getJobExecution().getJobParameters().getString(APP_GRP_KEY);
		appInfo = pushInfoService.getAppInfo(appGrpKey);
		gcmHttpServiceImpl.init(appInfo);
		gcmCcsService.init(appInfo.getGcmProjectNum(), appInfo.getGcmApiKey());
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		return stepExecution.getExitStatus();
	}
	
	
}
