package com.humuson.tms.batch.writer;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.core.launch.JobOperator;
import org.springframework.batch.item.ItemWriter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSourceUtils;

import com.humuson.tms.batch.domain.TargetSchedule;
import com.humuson.tms.batch.job.constrants.JobExecutionContextConstrants;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;
import com.humuson.tms.batch.job.partition.KeyDivider;
import com.humuson.tms.batch.service.PostIdDevideService;
import com.humuson.tms.batch.service.ScheduleService;
import com.humuson.tms.batch.service.SendListService;
import com.humuson.tms.batch.service.TargetServerService;
import com.humuson.tms.common.model.target.TargetData;
import com.humuson.tms.constrants.TargetStatus;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * 타겟팅 대상(FILE/ DB) 정보를 TMS_CAMP_SEND_LIST_XX로 입력을 담당
 * @author hyogun
 *
 */
@Slf4j
public class JdbcTargetRawDataWriter implements ItemWriter<TargetData>, StepExecutionListener {
	
	@Autowired private JdbcTemplate jdbcTemplate;
	@Autowired private NamedParameterJdbcTemplate namedParameterJdbcTemplate;
	@Autowired PostIdDevideService postIdDevideService;
	@Autowired TargetServerService targetServerService;
	@Autowired SendListService sendListServiceImpl;
	@Autowired ScheduleService<TargetSchedule> targetScheduleService;
	@Autowired JobOperator jobOperator;
	
	@Setter private String updateScheduleInfo;
	@Setter private String updateDeduplicationInfo;
	@Setter private String insertSiteUser;
	@Setter private String insertTempSendList;
	@Setter private String updateStatusStopInfo;
	@Setter private String deleteTempSendListWriter;
	@Setter private String insertTargetErrorList;
	
	
	private String postId;
	private String msgId;
	private String addSiteUserYn;
	private String deduplicationYn;
	private List<Properties> targetServerList;
	private List<Properties> postIdList;
	
	private StepExecution stepExecution;
	
	@Override
	public void write(List<? extends TargetData> rawData) throws Exception {

		// 	TODO 타게팅 중지요청 확인
		if (targetScheduleService.isCanceled(msgId)) {
			log.info("stepExecution.getJobExecutionId() : {}",stepExecution.getJobExecutionId());
			boolean isStoped = false;
			try {
				isStoped = jobOperator.stop(stepExecution.getJobExecutionId());
				if(isStoped){
					
					targetScheduleService.updateScheduleStatus(msgId, TargetStatus.CANCEL_COMPLETE.getCode());
					jdbcTemplate.update(deleteTempSendListWriter, msgId);
					
				}
				log.info("[JdbcTargetRawDataWriter] Target Stop Success ...");
				
			} catch (Exception e) {
				log.error("[JdbcTargetRawDataWriter] updateScheduleStatus ERROR : "+e.getMessage());
			}
			
		} else {
		
			// TODO 중복 제거 로직
			List<TargetData> targetDataList = new ArrayList<TargetData>();
			targetDataList.addAll((List<TargetData>) rawData);
			
			int duplicateCount = 0;
			
			if("Y".equals(deduplicationYn)){
				SqlParameterSource[] dupInsertParam = SqlParameterSourceUtils.createBatch(targetDataList.toArray());
				
				try {
					sendListServiceImpl.insertSendTempList(dupInsertParam);
				} catch (Exception e) {
					log.error("[JdbcTargetRawDataWriter][Exception] : "+e.getMessage());
					int size = targetDataList.size();
					
					SqlParameterSource param;
					for (int i=size-1; i>=0; i--) {
						param = dupInsertParam[i];
						
						try {
							sendListServiceImpl.insertSendTempList(param);
						} catch (Exception ee) {
							// 타게팅 실패리스트 INSERT
							jdbcTemplate.update(insertTargetErrorList, param.getValue("channelType"), param.getValue("msgId"), param.getValue("seqNo"), param.getValue("memberId"), param.getValue("duplicateKey"),"DE");
							targetDataList.remove(i);
							duplicateCount++;
						}
					}
				}
			}
		
			log.info("[JdbcTargetRawDataWriter] targetDataList.size : {} duplicateCount : {}",targetDataList.size(), duplicateCount);
			
			if (!targetDataList.isEmpty()) {
				
				long startTime = System.currentTimeMillis();
				int targetSize = targetDataList.size();
				
				KeyDivider abTestPartitioner = new KeyDivider("POST_ID", "AB_TEST_RATE");
				abTestPartitioner.divisionId(targetDataList.size(), postIdList);
				
				KeyDivider serverIdPartitioner = new KeyDivider("SERVER_ID", "PARTICIPATION_RATE");
				// TODO Collections.shuffle의 thread safe 여부 확인 필요
				//Collections.shuffle(targetServerList);
				serverIdPartitioner.divisionId(targetDataList.size(), targetServerList);
				
				log.info("targetDataList size :{}, serverIdPartitioner : {}",
						targetSize, serverIdPartitioner.toString());
				
				Map<String, Integer> serverIdTargetCntMap = new HashMap<String, Integer>();
				String serverIdKey = null;
				Integer serverIdTargetCnt = null;
				int filterErrorCount = 0;
				List<TargetData> newTargetData = new ArrayList<TargetData>();
				
				TargetData targetData;
				for (int i=0; i<targetSize; i++) {
					targetData = targetDataList.get(i);
					targetData.setPostId(abTestPartitioner.getId(i));
					targetData.setServerId(serverIdPartitioner.getId(i));
					
					serverIdKey = targetData.getPostId()+"_"+targetData.getServerId();
					if (log.isDebugEnabled()) {
					//	log.info(serverIdKey);
					}
					serverIdTargetCnt = serverIdTargetCntMap.get(serverIdKey);
					if (serverIdTargetCnt == null) {
						serverIdTargetCnt = 0;
					}
					
					newTargetData.add(targetData);
					if (log.isDebugEnabled()) {
					//	log.info("postId:{}, serverId:{}", targetData.getPostId(), targetData.getServerId());
					}
					serverIdTargetCntMap.put(serverIdKey, ++serverIdTargetCnt);
				}
				
				log.info("newTargetData.size :{}", newTargetData.size());
				int insertCount = 0;
				SqlParameterSource[] params = SqlParameterSourceUtils.createBatch(newTargetData.toArray());
				
				try {

					insertCount = sendListServiceImpl.insertSendList(params);
					
					if ("Y".equals(addSiteUserYn)) {
						namedParameterJdbcTemplate.batchUpdate(insertSiteUser, params);
					}
					
				} catch (Exception e) {
					log.error("insert error", e);
					int size = newTargetData.size();
					SqlParameterSource param;
					for (int i = size -1; i >= 0 ; i--) {
						param = params[i];
						
						try {
							
							insertCount += sendListServiceImpl.insertSendList(param);

							if ("Y".equals(addSiteUserYn)) {
								namedParameterJdbcTemplate.update(insertSiteUser, param);
							}
							
						} catch (Exception e2) {
							log.error("duplicate skip [m_id:{}, m_name:{}, m_email:{}, m_phone:{}, deviceId:{}]", 
									param.getValue("memberId"), param.getValue("memberName"), param.getValue("memberEmail"),
									param.getValue("memberPhone"), param.getValue("deviceId"),
									e2);
							// 타게팅 실패리스트 INSERT
							jdbcTemplate.update(insertTargetErrorList, param.getValue("channelType"), param.getValue("msgId"), param.getValue("seqNo"),param.getValue("memberId"), param.getValue("channelType").equals("EM") ? param.getValue("memberEmail") : param.getValue("channelType").equals("SM") ? param.getValue("memberPhone"): param.getValue("deviceId"),"OE");
							
							filterErrorCount++;
							
							serverIdKey = abTestPartitioner.getId(i+1)+"_"+serverIdPartitioner.getId(i+1);
							serverIdTargetCnt = serverIdTargetCntMap.get(serverIdKey);
							serverIdTargetCntMap.put(serverIdKey, --serverIdTargetCnt);
						}
					}
				} finally {
					log.info("insert send list [size : {}, elapseTime:{}, duplicateCnt : {}]",
							insertCount,
							System.currentTimeMillis() - startTime,
							duplicateCount);
				}
				
				jdbcTemplate.update(updateScheduleInfo, insertCount, filterErrorCount, duplicateCount, msgId);
				
				Set<String> serverIdByPostId = serverIdTargetCntMap.keySet();
				String[] keys = null;
				
				for (String key : serverIdByPostId) {
					keys = key.split("_");
					targetServerService.updateTargetServerInfo(keys[0], keys[1], serverIdTargetCntMap.get(key));
				}
			} else if(duplicateCount > 0){
				jdbcTemplate.update(updateDeduplicationInfo, duplicateCount, msgId);
			}
		}
	}

	@Override
	public void beforeStep(StepExecution stepExecution) {
		this.stepExecution = stepExecution;
		postId = stepExecution.getJobParameters().getString(JobParamConstrants.POST_ID);
		msgId = stepExecution.getJobParameters().getString(JobParamConstrants.MSG_ID);
		addSiteUserYn = stepExecution.getJobExecution().getExecutionContext()
				.getString(JobExecutionContextConstrants.KEY_ADD_SITE_USER_YN, "N");
		deduplicationYn = stepExecution.getJobParameters().getString(JobParamConstrants.DEDUPLICATION_YN);
		targetServerList = this.targetServerService.getTargetServerList(postId);
		postIdList = this.postIdDevideService.getAbTestPostIds(msgId, postId);
		
		log.info("postIdList : {}, deduplicationYn : {} ", postIdList.toString(),deduplicationYn);
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		return stepExecution.getExitStatus();
	}
	
	public static void main(String[] args) {
		String a = "lBME72nnpBRKBnzm/oph9+5eo8opEDDZIat9OtjK4l4=";
		System.out.println(a.length());
	}
}
