package com.humuson.tms.batch.writer;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemWriter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSourceUtils;

import com.humuson.tms.batch.job.constrants.JobExecutionContextConstrants;
import com.humuson.tms.batch.job.constrants.JobParamConstrants;
import com.humuson.tms.batch.job.partition.KeyDivider;
import com.humuson.tms.batch.service.SendListService;
import com.humuson.tms.batch.service.TargetServerService;
import com.humuson.tms.common.model.target.TargetData;

@Slf4j
public class JdbcOtpmzTargetWriter implements ItemWriter<TargetData>, StepExecutionListener {
	
	@Autowired
	private JdbcTemplate jdbcTemplate;
	
	@Autowired TargetServerService targetServerService;
	@Autowired SendListService sendListServiceImpl;
	
	@Setter 
	private String insertSendList, updateScheduleInfo, updateDeduplicationInfo;
	
	private String postId, channelType, deduplicationYn, msgId;
	
	private List<Properties> targetServerList;
	
	@Override
	public void write(List<? extends TargetData> rawData) throws Exception {
		
		log.info("rawData size : {}", rawData.size());
		
		// TODO 중복 제거 로직
		List<TargetData> targetDataList = new ArrayList<TargetData>();
		targetDataList.addAll((List<TargetData>) rawData);
		
		int duplicateCount = 0;
		
		if("Y".equals(deduplicationYn)){
			
			SqlParameterSource[] dupInsertParam = SqlParameterSourceUtils.createBatch(targetDataList.toArray());

			try {
				
				sendListServiceImpl.insertSendTempList(dupInsertParam);
				
			} catch (Exception e) {
				log.error("[JdbcOtpmzTargetWriter][ERROR] : "+e.getMessage());
				int size = targetDataList.size();
				
				SqlParameterSource param;
				for (int i=size-1; i>=0; i--) {
					param = dupInsertParam[i];
					try {
						sendListServiceImpl.insertSendTempList(param);
					} catch (Exception ee) {
						targetDataList.remove(i);
						duplicateCount++;
					}
				}
			}
		}
		
		log.info("[JdbcTargetRawDataWriter] targetDataList.size : {} duplicateCount : {}",targetDataList.size(), duplicateCount);
		
		if (!targetDataList.isEmpty()) {
		
			long startTime = System.currentTimeMillis();
			List<Object[]> params = new ArrayList<Object[]>();
			
			KeyDivider serverIdPartitioner = new KeyDivider("SERVER_ID", "PARTICIPATION_RATE");
			//Collections.shuffle(targetServerList);
			serverIdPartitioner.divisionId(targetDataList.size(), targetServerList);
			
			Map<String, Integer> serverIdTargetCntMap = new HashMap<String, Integer>();
			TargetData obj = null;
			String serverId = null;
			Integer serverIdTargetCnt = null;
			int targetSize = targetDataList.size();
			
			for (int i=0; i<targetSize; i++) {
				serverId = serverIdPartitioner.getId(i);
				
				if (serverIdTargetCnt == null) serverIdTargetCnt = 0;
				serverIdTargetCntMap.put(serverId, ++serverIdTargetCnt);
				
				obj = targetDataList.get(i);
				params.add(new Object[]{
						channelType, postId, obj.getChunkId(), serverId,
						obj.getMemberId(), obj.getMemberName(), 
						obj.getMemberEmail(), obj.getDomain(), 
						obj.getMemberToken(), obj.getDeviceId(), 
						obj.getMemberPhone(), obj.getTelCode(),
						obj.getField1(), obj.getField2(), obj.getField3(), obj.getField4(), 
						obj.getField5(), obj.getField6(), obj.getField7(), obj.getField8()
				});
			}
			
			int[] rows = jdbcTemplate.batchUpdate(insertSendList, params);
			
			log.info("insert size : {}, elapseTime:{} ", rows.length, (System.currentTimeMillis() -startTime));
			
			jdbcTemplate.update(updateScheduleInfo, params.size(), postId);
			
			targetServerService.updateTargetServerInfo(postId, serverIdTargetCntMap);
		
		} else if(duplicateCount > 0){
			jdbcTemplate.update(updateDeduplicationInfo, duplicateCount, msgId);
		}
	}

	@Override
	public void beforeStep(StepExecution stepExecution) {
		postId = stepExecution.getJobParameters().getString(JobExecutionContextConstrants.KEY_POST_ID);
		channelType = stepExecution.getJobParameters().getString(JobParamConstrants.CHN_TYPE);
		deduplicationYn = stepExecution.getJobParameters().getString(JobParamConstrants.DEDUPLICATION_YN);
		targetServerList = this.targetServerService.getTargetServerList(postId);
		msgId = stepExecution.getJobParameters().getString(JobParamConstrants.MSG_ID);
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		return stepExecution.getExitStatus();
	}
}
