package com.humuson.tms.batch.writer;

import java.util.ArrayList;
import java.util.List;

import org.json.simple.JSONObject;
import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemWriter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.DirectQueue;
import com.humuson.tms.batch.domain.DirectQueueResult;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.domain.PushQueue;
import com.humuson.tms.batch.domain.PushResult;
import com.humuson.tms.batch.service.DirectPushResultService;
import com.humuson.tms.batch.service.MqProducer;
import com.humuson.tms.batch.service.PushInfoService;
import com.humuson.tms.batch.service.PushSendService;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.constrants.PushResponseConstants;
import com.humuson.tms.mq.model.MgsPush;
import com.humuson.tms.mq.model.MgsPush.PushChnType;
import com.humuson.tms.mq.model.MgsPush.Request.AckMode;
import com.humuson.tms.mq.model.MgsPush.Request.SendType;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * Private 전송 이후 public 재발송 및 만료 처리 담당
 * @author hyogun
 *
 */
@Slf4j
public class DirectPushSendWriter implements ItemWriter<DirectQueue>, StepExecutionListener {
	
	@Autowired protected PushSendService<List<PushQueue>,List<PushResult>> mgsPushSendServiceImpl;
	@Setter protected String appGrpKey;
	protected App appInfo;
	
	@Autowired
	@Qualifier("mqAutoReqProducer")
	private MqProducer mqAutoReqProducer;
	
	@Autowired
	@Qualifier("mqFastAutoReqProducer")
	private MqProducer mqFastAutoReqProducer;

	@Autowired protected DirectPushResultService directPushResultService;
	
	@Autowired protected PushInfoService<App, PushMessage> pushInfoService;
	
	@Override
	public void beforeStep(StepExecution stepExecution) {
		appInfo = pushInfoService.getAppInfo(appGrpKey);
	}
	
	@Override
	public void write(List<? extends DirectQueue> rawData) throws Exception {
		
		MgsPush.Request.Builder gcmRequestBuilder = MgsPush.Request.newBuilder()
				.setAppKey(appInfo.getAppKey(App.ANDROID))
				.setAckMode(AckMode.ACK)
				.setPushChnType(PushChnType.GCM)
				.setSendType(SendType.ONE2ONE)
				.setSeverId("01");;
		MgsPush.Request.Builder gcmFastRequestBuilder = MgsPush.Request.newBuilder()
				.setAppKey(appInfo.getAppKey(App.ANDROID))
				.setAckMode(AckMode.ACK)
				.setPushChnType(PushChnType.GCM)
				.setSendType(SendType.ONE2ONE)
				.setSeverId("01");;;
		MgsPush.Request.Builder apnsRequestBuilder = MgsPush.Request.newBuilder()
				.setAppKey(appInfo.getAppKey(App.IOS))
				.setAckMode(AckMode.ACK)
				.setPushChnType(PushChnType.APNS)
				.setSendType(SendType.ONE2ONE)
				.setSeverId("01");;;
		MgsPush.Request.Builder apnsFastRequestBuilder = MgsPush.Request.newBuilder()
				.setAppKey(appInfo.getAppKey(App.IOS))
				.setAckMode(AckMode.ACK)
				.setPushChnType(PushChnType.APNS)
				.setSendType(SendType.ONE2ONE)
				.setSeverId("01");;;
		
		MgsPush.PushPayload.Builder payloadBuilder = MgsPush.PushPayload.newBuilder();
		
		try {
			List<DirectQueue> rawData2 = (List<DirectQueue>) rawData;
			directPushResultService.batchUpdateDirectQueue(rawData2);
		} catch (Exception e) {
			log.warn("[DIRECT] TMS_QUEUE batch update fail {}", e.getMessage());
			for (DirectQueue item : rawData) {
				try {
					directPushResultService.updateDirectQueue(item);
				} catch (Exception e2) {
					log.error("[DIRECT] TMS_QUEUE update fail [{}] msg:{}", item.toString(), e.getMessage()); 
				}
			}
		}
		
		StringBuilder sb = new StringBuilder();
		List<DirectQueueResult> resultList = new ArrayList<DirectQueueResult>(); 
		
		for (DirectQueue item : rawData) {
			
			if ("N".equals(item.getNotiFlag())) {
				resultList.add(new DirectQueueResult(item, PushResponseConstants.DENY_PUSH));
				continue;
			}
			
			payloadBuilder.clear();
			sb.setLength(0);
			sb.append("PUSH_ID_VALUE").append("&&")
					.append( item.getDeviceId()).append("&&")
					.append( item.getReqUid()).append("&&")
					.append( item.getCustId()).toString();
			
			payloadBuilder.setId(sb.toString()).setToken(item.getPushToken());
			
			if (App.ANDROID.equals(item.getAppOs())) {
				payloadBuilder.setGcmMessage(this.makeGcmMqMessage(item));
				
				if (item.getPriority() > 0) {
					gcmFastRequestBuilder.addPayload(payloadBuilder.build());
				} else {
					gcmRequestBuilder.addPayload(payloadBuilder.build());
				}
			} else if (App.IOS.equals(item.getAppOs())) {
				payloadBuilder.setApnsMessage(this.makeApnsMqMessage(item));
				
				if (item.getPriority() > 0) {
					apnsFastRequestBuilder.addPayload(payloadBuilder.build());
				} else {
					apnsRequestBuilder.addPayload(payloadBuilder.build());
				}
			}
			
			resultList.add(new DirectQueueResult(item));
		}
		
		// APNS SEND priority 9
		if (!apnsFastRequestBuilder.getPayloadList().isEmpty()) {
			mqFastAutoReqProducer.send(apnsFastRequestBuilder.build());
		}
		// GCM SEND priority 9
		if (!gcmFastRequestBuilder.getPayloadList().isEmpty()) {
			mqFastAutoReqProducer.send(gcmFastRequestBuilder.build());
		}
		
		// APNS SEND
		if (!apnsRequestBuilder.getPayloadList().isEmpty()) {
			mqAutoReqProducer.send(apnsRequestBuilder.build());
		}
		// GCM SEND
		if (!gcmRequestBuilder.getPayloadList().isEmpty()) {
			mqAutoReqProducer.send(gcmRequestBuilder.build());
		}
		
		try {
			log.info("[DIRECT] result size :{}", resultList.size());
			directPushResultService.batchInsertDirectQueueResult(resultList);
		} catch (Exception e) {
			
			log.error("[DIRECT] TMS_QUEUE_RESULT batch update fail {}", e);
			for (DirectQueueResult item : resultList) {
				try {
					directPushResultService.insertDirectQueueResult(item);
				} catch (Exception e2) {
					log.error("[DIRECT] TMS_QUEUE_RESULT update fail [{}] msg:{}", item.toString(), e.getMessage()); 
				}
			}
		}
		
		
		log.info("ActiveMQ SEND [END] APNS:{},GCM:{},APNS_fast:{},GCM_fast:{}",
				apnsRequestBuilder.getPayloadList().size(),gcmRequestBuilder.getPayloadList().size(),
				apnsFastRequestBuilder.getPayloadList().size(),gcmFastRequestBuilder.getPayloadList().size());
		
		
	}
	
	private MgsPush.GcmMessage makeGcmMqMessage(DirectQueue item) {
		MgsPush.GcmMessage.Builder gcmMsgBuilder = MgsPush.GcmMessage.newBuilder();

		gcmMsgBuilder.setTimeToLive(1800)
		.setDelayWhileIdle(false)
		.setPushRichContent("")
		.setPushNotiTitle(item.getSubject())
		.setPushNotiMsg(item.getContent())
		.setPushNotiImg(item.getImgUrl())
		.setMsgType(PushMessage.TEXT)
		.setMsgId("DIRECT_SEQ"+item.getSeq());
		
		JSONObject json = new JSONObject();
		if (!StringUtils.isNull(item.getPushValue())) {
			json.put( "l", StringUtils.validString(item.getPushValue()));
		}
		gcmMsgBuilder.setPushData(json.toString());
		
		return gcmMsgBuilder.build();
	}
	
	private MgsPush.ApnsMessage makeApnsMqMessage(DirectQueue item) {
		MgsPush.ApnsMessage.Builder apnsMsgBuilder = MgsPush.ApnsMessage.newBuilder();
		apnsMsgBuilder
			.setMessage(item.getSubject()+"\n"+item.getContent())
			.setSound(StringUtils.validString(""))
			.setMsgType(PushMessage.TEXT)
			.setMsgId("DIRECT_SEQ"+item.getSeq());
		
		if (!StringUtils.isNull(item.getPushValue())) {
			apnsMsgBuilder.setPushKey("l")
				.setPushValue(StringUtils.validString(item.getPushValue()));
		}
		return apnsMsgBuilder.build();
	}
	
	

	@Override
	public ExitStatus afterStep(StepExecution arg0) {
		// TODO Auto-generated method stub
		return null;
	}

	
}
