package com.humuson.tms.batch.writer;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

import org.springframework.batch.core.ExitStatus;
import org.springframework.batch.core.StepExecution;
import org.springframework.batch.core.StepExecutionListener;
import org.springframework.batch.item.ItemWriter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;

import com.humuson.tms.batch.domain.BaseResult;
import com.humuson.tms.batch.domain.ChannelSendType;
import com.humuson.tms.batch.service.UpdateResultService;

/**
 * 
 * @author hyogun
 *
 * @param <T extends BaseResult >
 */
@Slf4j
public abstract class AbstractResultWriter<T extends BaseResult> implements ItemWriter<T>, StepExecutionListener {
	
	public static final String SEND = "S";
	public static final String READ = "R";
	public static final String CLICK = "C";
	public static final String DELIVER = "D";
	
	@Autowired protected JdbcTemplate jdbcTemplate;
	@Autowired protected NamedParameterJdbcTemplate namedParameterJdbcTemplate;
	
	@Autowired protected UpdateResultService updateResultServiceImpl;
	@Setter protected String updateCheckFlag;
	@Setter protected String updateAutoSendList;
	@Setter protected String updateAutoSendListByRowId;
	@Setter protected String updateCampSendList;
	@Setter protected String updateCampSendListByRowId;
	@Setter protected String updateAutoScheduleStat;
	@Setter protected String updateCampScheduleStat;
	@Setter protected String updateAutoOpenList;
	@Setter protected String updateCampOpenList;
	@Setter protected String updateAutoClickList;
	@Setter protected String updateCampClickList;
	@Setter protected String updateAutoOpenStat;
	@Setter protected String updateCampOpenStat;
	@Setter protected String updateAutoClickStat;
	@Setter protected String updateCampClickStat;
	@Setter protected String updateCampClickLinkStat;
	@Setter protected String updateAutoClickLinkStat;
	
	@Setter protected String updateAutoSendListDeliver;
	@Setter protected String updateAutoSendListDeliverByRowId;
	@Setter protected String updateCampSendListDeliver;
	@Setter protected String updateCampSendListDeliverByRowId;
	
	/**
	 * Process the supplied data element. Will not be called with any null items
	 * in normal operation.
	 *
	 * @param items items to be written
	 * @throws Exception if there are errors. The framework will catch the
	 * exception and convert or rethrow it as appropriate.
	 */
	@Override
	public void write(List<? extends T> rawData) throws Exception {
		
		if (rawData.isEmpty()) {
			return ;
		}
		
		this.updateCheckFlag(rawData);
		
		Map<String, List<T>> devidePostIdMap = new HashMap<String, List<T>>();
	
		String key = null;
		for (T baseResult : rawData) {
			if (baseResult.getType().startsWith("C")) {
				key = baseResult.getType() +"_"+baseResult.getPostId()+"_"+baseResult.getRtnType();
			} else if (baseResult.getType().startsWith("A")) {
				key = baseResult.getType() +"_"+baseResult.getWorkday()+"_"+baseResult.getSeqNo()+"_"+baseResult.getRtnType();
			}
			
			if (key == null) {
				log.error(baseResult.getType());
			}
			
			List<T> postIdList = devidePostIdMap.get(key);
			if (postIdList == null) {
				postIdList = new ArrayList<T>();
			}
			postIdList.add(baseResult);
			devidePostIdMap.put(key, postIdList);
		}
		
		Set<String> keySet = devidePostIdMap.keySet();
		
		String type = null;
		String postId = null;
		String rtnType = null;
		String[] keys = null;
		ChannelSendType sendType = null;
		
		for (String resultKey : keySet) {
			keys = resultKey.split("_");
			
			type = keys[0];
			if (type.startsWith("C")) {
				postId = keys[1];
				rtnType = keys[2];
				sendType = ChannelSendType.CAMP;
			} else if (type.startsWith("A")) {
				postId = keys[1]+"_"+keys[2];
				rtnType = keys[3];
				sendType = ChannelSendType.AUTO;
			}
			
			this.execute(sendType, postId, rtnType, devidePostIdMap.get(resultKey));
		}
		
	}
	
	/**
	 * 
	 * @param rawData
	 */
	protected abstract void updateCheckFlag(final List<? extends T> rawData);
	
	/**
	 * 
	 * @param channelSendType
	 * @param key
	 * @param rtnType
	 * @param logList
	 */
	public abstract void execute(ChannelSendType channelSendType, String key, String rtnType, List<T> logList);
	
	@Override
	public void beforeStep(StepExecution stepExecution) {
	}

	@Override
	public ExitStatus afterStep(StepExecution stepExecution) {
		return null;
	}
}
