package com.humuson.tms.batch.service.impl;

import static com.humuson.tms.constrants.PushResponseConstants.APNS_DUPLICATE_DEVICE_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_INVALID_DEVICE_TOKEN_FORMAT_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_NULL_DEVICE_TOKEN_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_NULL_ID_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_PAYLOAD_MAX_SIZE_EXCEEDED_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_UNKNOWN_DEVICE_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_UNKNOWN_ERROR;
import static com.humuson.tms.constrants.PushResponseConstants.DENY_PUSH;
import static com.humuson.tms.constrants.PushResponseConstants.DUPLICATE_DEVICE_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.INVALID_DEVICE_TOKEN_FORMAT_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.NO_SEND;
import static com.humuson.tms.constrants.PushResponseConstants.NULL_DEVICE_TOKEN_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.NULL_ID_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.PAYLOAD_MAX_SIZE_EXCEEDED_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.PRIVATE_INVALID_TOKEN;
import static com.humuson.tms.constrants.PushResponseConstants.PRIVATE_SERVER_ERROR;
import static com.humuson.tms.constrants.PushResponseConstants.PRIVATE_UNACTIVED_TOKEN;
import static com.humuson.tms.constrants.PushResponseConstants.SENDING;
import static com.humuson.tms.constrants.PushResponseConstants.SUCCESSFUL;
import static com.humuson.tms.constrants.PushResponseConstants.UNKNOWN_DEVICE_EXCEPTION;

import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import com.google.android.gcm.server.Message;
import com.humuson.rainboots.datastore.DataStore;
import com.humuson.rainboots.proto.messages.PushProtos.PushRequest;
import com.humuson.rainboots.proto.messages.PushProtos.PushRequest.PushType;
import com.humuson.rainboots.proto.messages.PushProtos.PushRequest.QosLevel;
import com.humuson.rainboots.proto.messages.PushProtos.PushResponse;
import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.domain.PushQueue;
import com.humuson.tms.batch.domain.PushResult;
import com.humuson.tms.batch.service.GcmHttpService;
import com.humuson.tms.batch.service.PrivatePushService;
import com.humuson.tms.batch.service.PushSendService;
import com.humuson.tms.common.security.HumusonDecryptor;
import com.humuson.tms.common.security.SSLChecker;
import com.humuson.tms.common.util.FileUtil;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.constrants.PushPayload;
import com.humuson.tms.constrants.PushResponseConstants;

import javapns.devices.exceptions.InvalidDeviceTokenFormatException;
import javapns.devices.implementations.basic.BasicDevice;
import javapns.notification.AppleNotificationServer;
import javapns.notification.AppleNotificationServerBasicImpl;
import javapns.notification.PushNotificationManager;
import javapns.notification.PushNotificationPayload;
import javapns.notification.PushedNotification;
import lombok.extern.slf4j.Slf4j;


/**
 * Rainboots / GCM / APNS 으로 푸시 메시지를 전송하기 위한 클래스
 * @author hyogun
 *
 */
@Slf4j
public class PushSendServiceImpl implements PushSendService<List<? extends PushQueue>,List<PushResult>> {
	
	static final String HTTP_URL = "http://";
	
	protected static final String DEFAULT_SERVER_NAME = "TMS-SEND-01";
	protected static final String RAINBOOTS_RUN 		= "Y";
	
	static final int DEFAULT_PUSH_LIVE_TIME = 1800;
	
	protected PushNotificationManager pushManager;
	
	@Autowired
	protected PrivatePushService<PushResponse, PushRequest> privateService;
	
	protected App appInfo;
	
	@Value("#{config['tms.db.enc.key']}")
	protected String encKey;
	
	protected final String serverName;
	
	@Value("#{config['tms.private.qos.level']}")
	protected int privateQosLevel;

	@Value("#{config['tms.apns.production.cn']}")
	protected String productionCN; // "Apple Push Services|Apple Production IOS Push Services"
	
	@Value("#{config['tms.apns.development.cn']}")
	protected String developmentCN; // "Apple Development IOS Push Services"
	
	@Value("#{config['send.gcm.type']}")
	protected String gcmType;
	
	@Autowired
	protected GcmCcsService gcmCcsService;
	
	@Autowired
	@Qualifier("gcmHttpServiceImpl")
	protected GcmHttpService<PushResult,PushQueue> gcmHttpServiceImpl;
	
	protected Map<String, Boolean> denyAppVersionMap = new ConcurrentHashMap<String, Boolean>();
	
	/**
	 * 기본 생성자
	 */
	public PushSendServiceImpl() {
		this.serverName = DEFAULT_SERVER_NAME;
		log.debug("PushSendServiceImpl :{} instance generate hashCode:{}", serverName, this.hashCode());
	}
	
	/**
	 * @param serverName
	 */
	public PushSendServiceImpl(String serverName) {
		this.serverName = serverName;
		log.debug("PushSendServiceImpl :{} instance generate hashCode:{}", serverName, this.hashCode());
	}
	
	@Override
	public synchronized void init(App appInfo) {
		try {
			this.appInfo = appInfo;
			
			String[] denyAppVersions = appInfo.getDenyAppVersion().split(",");
			
			for (String denyAppVer : denyAppVersions) {
				denyAppVersionMap.put(denyAppVer, true);
			}
			
			String apnsCert = appInfo.getApnsPushCert();
			String apnsPwd = appInfo.getApnsPushPwd();
			String gcmApiKey = appInfo.getGcmApiKey();
			long gcmProjectNum = appInfo.getGcmProjectNum();
			
			if (gcmApiKey != null && gcmType.equalsIgnoreCase("http")) {
				gcmHttpServiceImpl.init(appInfo);
			}else if(gcmApiKey != null && gcmType.equalsIgnoreCase("xmpp")){
				gcmCcsService.init(gcmProjectNum, gcmApiKey);
			}
			
			if (apnsCert != null && apnsPwd != null) {
				if (FileUtil.isValidFile(apnsCert)) {
					try {
						String decryptApnsPwd = HumusonDecryptor.decrypt(apnsPwd, encKey);
						this.openApnsConnection(apnsCert, decryptApnsPwd, SSLChecker.isProduction(new FileInputStream(apnsCert), decryptApnsPwd, productionCN, developmentCN, false));
					} catch (Exception e) {
						log.error("apns init exception [appInfo:{}, encKey:{}]", appInfo.toString(), encKey, e);
					}
				} else {
					log.error("APNS Cert File {} is not valid ", apnsCert);
				}
			} else {
				log.error("APNS Cert is Null or ApnsPwd is Null {}", appInfo.toString());
			}
		} catch (Exception e) {
			log.error("APNS & GCM init error", e);
		}
	}
	
	@Override
	public void close() {
		try {
			if (pushManager != null) {
				pushManager.stopConnection();
			}
		} catch (Exception e) {
			log.error("error : {}", e);
		}
	}
	
	protected void openApnsConnection (String apnsCertPath, String password, boolean isReal) {
		if (pushManager == null) {
			try {
				AppleNotificationServer server = new AppleNotificationServerBasicImpl(apnsCertPath, password, isReal);
				pushManager = new PushNotificationManager();
				
				//connection open
				pushManager.initializeConnection(server);
				
			} catch (Exception e) {
				log.error("error : {}", e);
				try {
					if (pushManager != null) {
						pushManager.stopConnection();
					}
				} catch (Exception e2) {
					log.error("stop connection error", e2);
				} finally {
					pushManager = null;
				}
			}
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see com.humuson.tms.batch.service.PushSendService#request(java.util.List, boolean, boolean)
	 */
	@Override
	public List<PushResult> request(List<? extends PushQueue> list,
			boolean isGcmReSend, boolean unActivePublish) throws Exception {
		return this.request(list, isGcmReSend, unActivePublish, false);
	}
	
	/*
	 * (non-Javadoc)
	 * @see com.humuson.tms.batch.service.PushSendService#request(java.util.List, boolean, boolean, boolean)
	 */
	@Override
	public List<PushResult> request(List<? extends PushQueue> list,
			boolean isGcmReSend, boolean unActivePublish, boolean useWakeupGcm) {
		
		Map<String, PushQueue> reqAndroidUserMap = new HashMap<String, PushQueue>();
		List<PushResult> pushResultList = new ArrayList<PushResult>();
		
		// pirvate push request protoBuf
		PushRequest.Builder builder = PushRequest.newBuilder();
		
		String androidAppKey = appInfo.getAppKey(App.ANDROID);
		if (privateService.useRainboots() 
				&& !StringUtils.isNull(androidAppKey)
				&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())) {
			builder.setAppkey(androidAppKey);
			builder.setAlias(serverName);
			builder.setType(PushType.ONE2ONE);
			builder.setQosLevel(QosLevel.valueOf(privateQosLevel));
			builder.setUnActivePublish(unActivePublish);
		}
		
		int denyAppVerCount = 0;
		for (PushQueue pushQueue : list) {
			PushResult pushResult = this.checkValidation(pushQueue, appInfo.getOs(pushQueue.getAppId()));
			if (pushResult != null) {
				pushResultList.add(pushResult);
				continue;
			}
			
			if (App.IOS.equals(appInfo.getOs(pushQueue.getAppId()))) { 	// iOS
				PushResult apnsResult = this.sendApnsMessage(pushQueue);
				pushResultList.add(apnsResult);
			} else {	// Android
				
				// make to Rainboots send list
				PushRequest.Payload.Builder payloadBuilder = PushRequest.Payload.newBuilder();
				
				if (privateService.useRainboots() 
						&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())) {
					
					payloadBuilder.setId(pushQueue.getPushId()+"_"+pushQueue.getDeviceId()+"_"+pushQueue.getReqUid()+"_"+pushQueue.getCustId());
					payloadBuilder.setToken(String.valueOf(pushQueue.getDeviceId()));
					payloadBuilder.setTimeToLive(pushQueue.getPushMessage().getPushTtl() > 0 ? pushQueue.getPushMessage().getPushTtl() :DEFAULT_PUSH_LIVE_TIME);
					// private server에서 msg Id를 발번하도록 기본 값으로 세팅함
					payloadBuilder.setMsgId(0);
					
					String realTimeRainbootsMsg = makeRainbootsMessage(pushQueue.getPushMessage());
					
					log.debug("realtime rainboots send message : {}", realTimeRainbootsMsg);
					payloadBuilder.setMessage(realTimeRainbootsMsg);
					
					builder.addPayload(payloadBuilder.build());
				} 
				
				reqAndroidUserMap.put(pushQueue.getPushId()+"_"+pushQueue.getDeviceId(), pushQueue);
			}		// end Android
		}		// end for loop
		
		if (denyAppVerCount > 0)
			log.info("deny app version filter count : {}", denyAppVerCount);
		
		if (reqAndroidUserMap.size() > 0) {	// android
			
			if (privateService.useRainboots()
					&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())
					&& builder.getPayloadCount() > 0) {		// send private message
				long startTime = System.currentTimeMillis();
				
				log.debug("rainboots reqeust : {}", builder.build());
				
				List<PushResult> rainbootsResult = sendRainboots(reqAndroidUserMap, builder.build(), isGcmReSend, false);
				
				pushResultList.addAll(rainbootsResult);
				
				log.debug("SEND PUSH RAINBOOTS REQUEST DURATION TIME :{}", System.currentTimeMillis() - startTime);
			} 
			
			// gcm resend or rainboots fail or no use rainboots
			if (reqAndroidUserMap.size() > 0 && isGcmReSend) {
				Collection<PushQueue> androidList = reqAndroidUserMap.values();
				
				if("http".equalsIgnoreCase(gcmType)){
					pushResultList.addAll(gcmHttpServiceImpl.sendGcmOne2OneList(androidList, appInfo.getGcmApiKey()));
				}else if ("xmpp".equalsIgnoreCase(gcmType)){
					StringBuilder sb = new StringBuilder();
					for (PushQueue que : androidList) {
						sb.setLength(0);
						String id = sb.append(que.getPushId()).append("&&")
								.append(que.getDeviceId()).append("&&")
								.append(que.getReqUid()).append("&&")
								.append(que.getCustId()).toString();
						String ccsMessage = gcmCcsService.makeCcsMessage(que,que.getPushMessage(),appInfo.getAppGrpId());
						try {
							//[upstream]
							log.info("ccsMessage: {}",ccsMessage);
							gcmCcsService.sendDownstreamMessage(ccsMessage);							
							
						} catch (Exception e) {
							// TODO Auto-generated catch block
							
							log.error("exception in resendGcm (XMPP) : {}", que.getPushId(), e);
							pushResultList.add(new PushResult(appInfo.getAppGrpId(), PushResponseConstants.ERROR_UNAVAILABLE,sb.toString(),
									App.ANDROID, que.getRowId()));
							gcmCcsService.setHitCount(0);
							gcmCcsService.reset();
							continue;
						}
						pushResultList.add(new PushResult(appInfo.getAppGrpId(), PushResponseConstants.SENDING_CCS,
								sb.toString(),App.ANDROID, que.getRowId()));
					}
				}
			} else if (reqAndroidUserMap.size() > 0){
				log.error("reqAndroidUserMap size :{} gcmReSendFlag:{}", reqAndroidUserMap.size(), isGcmReSend);
				Collection<PushQueue> androidList = reqAndroidUserMap.values();
				StringBuilder sb = new StringBuilder();
				
				for (PushQueue pushQueue : androidList) {
					sb.setLength(0);
					String id = sb.append(pushQueue.getPushId()).append("&&")
							.append(pushQueue.getDeviceId()).append("&&")
							.append(pushQueue.getReqUid()).append("&&")
							.append(pushQueue.getCustId()).toString();
					pushResultList.add(new PushResult(appInfo.getAppGrpId(), NO_SEND, 
							id, App.ANDROID,pushQueue.getRowId()));
				}
			}
		}

		return pushResultList;
	}
	
	@Override
	public List<PushResult> request(List<? extends PushQueue> list, PushMessage pushMessage,
			boolean isGcmReSend) {
		return this.request(list, pushMessage, isGcmReSend, false);
	}
	
	@Override
	public List<PushResult> request(List<? extends PushQueue> list, PushMessage pushMessage,
			boolean isGcmReSend, boolean unActivePublish) {
		return this.request(list, pushMessage, isGcmReSend, unActivePublish, false);
	}
	
	@Override
	public List<PushResult> request(List<? extends PushQueue> list, PushMessage pushMessage,
			boolean isGcmReSend, boolean unActivePublish, boolean useWakeupGcm) {
		
		Map<String, PushQueue> reqAndroidUserMap = new HashMap<String, PushQueue>();
		List<PushResult> pushResultList = new ArrayList<PushResult>();
		
		// pirvate push request protoBuf
		PushRequest.Builder builder = PushRequest.newBuilder();
		
		String rainbootsBulkMsg = null;
		
		if (privateService.useRainboots() && appInfo.getAppKey(App.ANDROID) != null
				&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())) {
			builder.setAppkey(appInfo.getAppKey(App.ANDROID));
			builder.setAlias(serverName);
			
			builder.setType(PushType.BROADCAST);
			// private server에서 임시적으로 저장하기 위한 msg id는 
			// 32767를 넘지 않로록 PMS의 msgId와 mod 연산한 결과 값을 세팅 함
			builder.setMsgId((int) (Long.parseLong(pushMessage.getMsgUid()) % DataStore.MAX_BULK_MSG_ID));
		
			builder.setQosLevel(QosLevel.valueOf(privateQosLevel));
			builder.setUnActivePublish(unActivePublish);
			if (pushMessage.getPushTtl() > 0){
				builder.setTimeToLive(pushMessage.getPushTtl());
			}else{
				builder.setTimeToLive(DEFAULT_PUSH_LIVE_TIME);
			}
			
			rainbootsBulkMsg = makeRainbootsMessage(pushMessage);
			
			builder.setMessage(rainbootsBulkMsg);
		}
		
		for (PushQueue pushQueue : list) {
			
			PushResult pushResult = this.checkValidation(pushQueue, appInfo.getOs(pushQueue.getAppId()));
			
			if (pushResult != null) {
				pushResultList.add(pushResult);
				continue;
			}
			
			if (App.IOS.equals(appInfo.getOs(pushQueue.getAppId()))) { 	// iOS
				
				PushResult apnsResult = this.sendApnsMessage(pushQueue);
				pushResultList.add(apnsResult);
				
			} else {	// Android
				// make to Rainboots send list
				reqAndroidUserMap.put(pushQueue.getPushId()+"_"+pushQueue.getDeviceId(), pushQueue);
				
				if (privateService.useRainboots()
						&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())) {
					PushRequest.Payload.Builder payloadBuilder = PushRequest.Payload.newBuilder();
					payloadBuilder.setId(pushQueue.getPushId()+"_"+pushQueue.getDeviceId()+"_"+pushQueue.getReqUid()+"_"+pushQueue.getCustId());
					payloadBuilder.setToken(String.valueOf(pushQueue.getDeviceId()));
					
					builder.addPayload(payloadBuilder.build());
				}
			}		// end Android
		}		// end for loop
		
		if (!reqAndroidUserMap.isEmpty()) {		// android
			
			if (privateService.useRainboots()
					&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())
					&& builder.getPayloadCount() > 0) {		// send private message
				long startTime = System.currentTimeMillis();
				List<PushResult> rainbootsResult = sendRainboots(reqAndroidUserMap, builder.build(), isGcmReSend, useWakeupGcm);
				pushResultList.addAll(rainbootsResult);
				log.info("send rainboots [size:{}, elapseTime:{}]", reqAndroidUserMap.size(), System.currentTimeMillis() - startTime);
			} 
			
			// gcm resend or rainboots fail or no use rainboots
			if (reqAndroidUserMap.size() > 0 && isGcmReSend) {
				Collection<PushQueue> androidList = reqAndroidUserMap.values();
				
				try {
					PushQueue[] pushQueues = new PushQueue[androidList.size()];
					androidList.toArray(pushQueues);
					
					long startTime = System.currentTimeMillis();
					
					List<PushResult> gcmSendResult;
					if("http".equalsIgnoreCase(gcmType)){
						Message gcmBulkMessage = gcmHttpServiceImpl.makeGcmMessage(pushQueues[0]);
						gcmSendResult = gcmHttpServiceImpl.sendGcmMulticastMessage(pushQueues, gcmBulkMessage,appInfo.getGcmApiKey());
						pushResultList.addAll(gcmSendResult);
					}else if ("xmpp".equalsIgnoreCase(gcmType)){
						StringBuilder sb = new StringBuilder();
						for (PushQueue que : pushQueues) {
							sb.setLength(0);
							String id = sb.append(que.getPushId()).append("&&")
									.append(que.getDeviceId()).append("&&")
									.append(que.getReqUid()).append("&&")
									.append(que.getCustId()).toString();
							String ccsMessage = gcmCcsService.makeCcsMessage(que,que.getPushMessage(),appInfo.getAppGrpId());
							try {
								//[upstream]
								log.info("ccsMessage: {}",ccsMessage);
								gcmCcsService.sendDownstreamMessage(ccsMessage);							
								
							} catch (Exception e) {
								// TODO Auto-generated catch block
								
								log.error("exception in resendGcm (XMPP) : {}", que.getPushId(), e);
								pushResultList.add(new PushResult(appInfo.getAppGrpId(), PushResponseConstants.ERROR_UNAVAILABLE,sb.toString(),
										App.ANDROID, que.getRowId()));
								gcmCcsService.setHitCount(0);
								gcmCcsService.reset();
								continue;
							}
							pushResultList.add(new PushResult(appInfo.getAppGrpId(), PushResponseConstants.SENDING_CCS,
									sb.toString(),App.ANDROID, que.getRowId()));
						}
					}
					
					log.info("send GCM [count:{}, elpasedTime:{}]", androidList.size(), System.currentTimeMillis() - startTime);
					
					
					
				} catch (Exception e) {
					log.error("GCM Send Error", e);
				}
			} else if (reqAndroidUserMap.size() > 0){
				log.error("reqAndroidUserMap size :{} reqAndroidUserMap.values():{}, gcmReSendFlag:{}", reqAndroidUserMap.size(), reqAndroidUserMap.values().size(), isGcmReSend);
				Collection<PushQueue> androidList = reqAndroidUserMap.values();
				StringBuilder sb = new StringBuilder();
				for (PushQueue pushQueue : androidList) {
					sb.setLength(0);
					String id = sb.append(pushQueue.getPushId()).append("&&")
							.append(pushQueue.getDeviceId()).append("&&")
							.append(pushQueue.getReqUid()).append("&&")
							.append(pushQueue.getCustId()).toString();
					pushResultList.add(new PushResult(appInfo.getAppGrpId(), NO_SEND, 
							id, 
							App.ANDROID, pushQueue.getRowId()));
				}
			}
		}

		return pushResultList;
	}
	
	protected List<PushResult> sendRainboots(Map<String, PushQueue> androidMap, 
			PushRequest pushRequest, boolean isGcmReSend, 
			boolean useWakeupGcm) {
		
		long startTime = System.currentTimeMillis();
		
		List<PushResult> pushResultList = new ArrayList<PushResult>();
		
		PushResponse response = privateService.request(pushRequest);
		
		if (response != null 
				&& response.getResult() == PushResponse.ResultType.SUCCESS) {
			log.debug("private server send result : {}", response.toString());
			
			int payloadSize = response.getPayloadCount();
			
			for (int i=payloadSize-1; i>=0; i--) {
				String ids[] = response.getPayload(i).getId().split("_");
				PushQueue pq = androidMap.get(ids[0]+"_"+ids[1]);
				if (response.getPayload(i).getResult() == PushResponse.ResultType.SUCCESS) {
					pushResultList.add(new PushResult(appInfo.getAppGrpId(), SENDING, response.getPayload(i).getId(), App.ANDROID, pq.getRowId()));
					androidMap.remove(ids[0]+"_"+ids[1]);
				} else if (!isGcmReSend) {
					if (response.getPayload(i).getResult() == PushResponse.ResultType.UNACTIVED_TOKEN) {
						pushResultList.add(new PushResult(appInfo.getAppGrpId(), PRIVATE_UNACTIVED_TOKEN, response.getPayload(i).getId(), App.ANDROID, pq.getRowId()));
					} else if (response.getPayload(i).getResult() == PushResponse.ResultType.INVALID_TOKEN) {
						pushResultList.add(new PushResult(appInfo.getAppGrpId(), PRIVATE_INVALID_TOKEN, response.getPayload(i).getId(), App.ANDROID, pq.getRowId()));
					}
					androidMap.remove(ids[0]+"_"+ids[1]);
				}
				
				//androidMap Key = PushId()+"_"+DeviceId() 
				//payload.getId() = PushId()+"_"+DeviceId()+"_"+ReqUid() + "_" + "cust_id"
				//androidMap.remove(ids[0]+"_"+ids[1]);
				
				if (log.isDebugEnabled()) {
					log.debug("return code : {}", response.getPayload(i).toString());
				}
			}
			
			long rainbootsElapsedTime = System.currentTimeMillis() - startTime;
			log.info("rainboots send [sendCount:{}, elapsedTime:{}]", pushRequest.getPayloadCount(), rainbootsElapsedTime);
			
			// Send GCM WakeUp Push Start
			if (useWakeupGcm) {
				sendWakeupGcm(pushRequest, androidMap);
			}
			// Send GCM WakeUp Push End
			
		} else {
			log.info("private server send fail result : {}", response == null ? "null" : response.getResult().toString());
			if (!isGcmReSend) {
				Set<String> sendRawIdSet = androidMap.keySet();
				for (String ids : sendRawIdSet) {
					PushQueue pq = androidMap.get(ids);
					pushResultList.add(new PushResult(appInfo.getAppGrpId(), PRIVATE_SERVER_ERROR, ids+"_"+pq.getReqUid()+"_"+pq.getCustId(), App.ANDROID, pq.getRowId()));
				}
				androidMap.clear();
			}
		}
		
		return pushResultList;
	}
	
	protected void sendWakeupGcm(PushRequest pushRequest, Map<String, PushQueue> androidMap) {
		long startTime = System.currentTimeMillis();
		Collection<PushQueue> wakeupGcmList = androidMap.values();
		PushQueue[] wakeupSendRaw = new PushQueue[wakeupGcmList.size()];
		wakeupGcmList.toArray(wakeupSendRaw);
		
		Message.Builder builder = new Message.Builder();
		builder.timeToLive(180);
		builder.delayWhileIdle(false);
		gcmHttpServiceImpl.sendGcmMulticastMessage(wakeupSendRaw, builder.build(),appInfo.getGcmApiKey());
		
		long gcmWakeUpPushElapsedTime = System.currentTimeMillis() - startTime;
		log.info("gcm wakeup push count : {} elapsedTime : {}", pushRequest.getPayloadCount(), gcmWakeUpPushElapsedTime);
	}

	/**
	 * APNS 전송 메소드
	 * @param pushQueue
	 * @return
	 */
	public synchronized PushResult sendApnsMessage(final PushQueue pushQueue) {
		
		StringBuilder sb = new StringBuilder();
		sb.setLength(0);
		String id = sb.append(pushQueue.getPushId()).append("&&")
				.append(pushQueue.getDeviceId()).append("&&")
				.append(pushQueue.getReqUid()).append("&&")
				.append(pushQueue.getCustId()).toString();
		if (pushManager == null) {
			return new PushResult(appInfo.getAppGrpId(), APNS_UNKNOWN_ERROR, id, App.IOS,pushQueue.getRowId());
		}
		
		try {
//			String message = pushQueue.getPushMessage().getPushTitle()+"\n"+pushQueue.getPushMessage().getPushMsg();
			
			// 17.04.12 iOS Push title 관련 수정
			String message = null;
			
			if (StringUtils.isNull(pushQueue.getPushMessage().getPushTitle())) {
				message = pushQueue.getPushMessage().getPushMsg();
			} else {
				message = pushQueue.getPushMessage().getPushTitle()+"\n"+pushQueue.getPushMessage().getPushMsg();
			}
			
			String pushKey = pushQueue.getPushMessage().getPushKey();
			String pushValue = pushQueue.getPushMessage().getPushValue();
			PushNotificationPayload payload = PushNotificationPayload.alert(message);
			
			if (appInfo.getPushSound() == null) {
				payload.addSound("default");
			} else {
				payload.addSound(appInfo.getPushSound());
			}

			if (appInfo.getIOsBadgeCount() > 0) {
				payload.addBadge(appInfo.getIOsBadgeCount());
			}
			
			payload.addCustomDictionary(PushPayload.MSG_TYPE, pushQueue.getPushMessage().getMsgType());
			payload.addCustomDictionary(PushPayload.MSG_ID, String.valueOf(pushQueue.getPushMessage().getMsgUid()));
			
			if (log.isDebugEnabled())
				log.debug("apns payload add before size:{} payload:{}", payload.getPayloadSize(), payload.toString());
			
			// payload size에 key/value를 삽입할 수 있는 경우에 customDictionary를 추가함
			if (!StringUtils.isNull(pushKey)
					&& payload.isEstimatedPayloadSizeAllowedAfterAdding(pushKey, pushValue)) {
				payload.addCustomDictionary(pushKey, pushValue);
				if (log.isDebugEnabled())
					log.debug("apns payload push key add after size:{} payload:{}", payload.getPayloadSize(), payload.toString());
			}
			
			BasicDevice device = new BasicDevice(pushQueue.getPushToken());
			
			PushedNotification notification = pushManager.sendNotification(device, payload, false);
			
//			log.info("====== apns payload size:{} payload:{} =====", payload.getPayloadSize(), payload.toString());
			
			String errorCode = null;
			if (notification.isSuccessful()) {
				errorCode = SUCCESSFUL;
			} else {
				errorCode = APNS_UNKNOWN_ERROR;
				
				if (notification.getException() != null) {
					String[] responseParts = notification.getException().toString().split(":");
						
					String err = responseParts[0];
					
					log.error("APNS error : {}", notification.getException());
					
			        if (err.contains(DUPLICATE_DEVICE_EXCEPTION)
			        		|| err.contains(INVALID_DEVICE_TOKEN_FORMAT_EXCEPTION)) {
			        	errorCode = APNS_DUPLICATE_DEVICE_EXCEPTION;
			        } else if (err.contains(NULL_DEVICE_TOKEN_EXCEPTION)) {
			        	errorCode = APNS_NULL_DEVICE_TOKEN_EXCEPTION;
			        } else if (err.contains(NULL_ID_EXCEPTION)) {
			        	errorCode = APNS_NULL_ID_EXCEPTION;
			        } else if (err.contains(UNKNOWN_DEVICE_EXCEPTION)) {
			        	errorCode = APNS_UNKNOWN_DEVICE_EXCEPTION;
			        } else if (err.contains(PAYLOAD_MAX_SIZE_EXCEEDED_EXCEPTION)) {
			        	errorCode = APNS_PAYLOAD_MAX_SIZE_EXCEEDED_EXCEPTION;
			        }
				}
			}
			 
			return new PushResult(appInfo.getAppGrpId(), errorCode, id, App.IOS,pushQueue.getRowId());
		} catch (Exception e) {
			log.error("APNS Send Error", e);
			if (e instanceof InvalidDeviceTokenFormatException) {
				return new PushResult(appInfo.getAppGrpId(), APNS_INVALID_DEVICE_TOKEN_FORMAT_EXCEPTION, id, App.IOS,pushQueue.getRowId());
			} else {
				return new PushResult(appInfo.getAppGrpId(), APNS_UNKNOWN_ERROR, id, App.IOS,pushQueue.getRowId());	
			}
		}
	}
	
	/**
	 * 
	 * @param message
	 * @param devices
	 * @param key
	 * @param value
	 * @param badge
	 * @throws Exception 
	 */
//	public PushResult sendGcmMessage(PushQueue pushQueue) {
//		
//		Message gcmMessage = makeGcmMessage(pushQueue.getPushMessage());
//		
//		Result result = null;
//		
//		try {
//			result = gcmSender.send(gcmMessage, pushQueue.getPushToken(), 3);
//		} catch (Exception e) {
//			log.error("exception in realtime send gcm : {}", e);
//		}
//		
//		return this.getGcmResult(result, pushQueue);
//	}
	
	@SuppressWarnings("unchecked")
	protected String makeRainbootsMessage(PushMessage pushMessage) {
		
		org.json.simple.JSONObject messageJson = new org.json.simple.JSONObject();
		
		try {
			messageJson.put(PushPayload.PUSH_NOTI_MSG, StringUtils.validString(pushMessage.getPushMsg()));
			messageJson.put(PushPayload.PUSH_NOTI_TITLE, StringUtils.validString(pushMessage.getPushTitle()));
			
			String pushImage = pushMessage.getPushImg();
			
			if (pushImage != null 
					&& (!pushImage.startsWith("http://"))
					&& "http://".length() >= pushImage.length()) {
				pushImage = "";
			}
			
			messageJson.put(PushPayload.PUSH_NOTI_IMG, pushImage == null ? "": pushImage);
			messageJson.put(PushPayload.PUSH_RICH_CONTENT, pushMessage.getPopupContent());
			messageJson.put(PushPayload.MSG_TYPE, pushMessage.getMsgType());
			messageJson.put(PushPayload.MSG_ID, String.valueOf(pushMessage.getMsgUid()));
			
			org.json.simple.JSONObject json = new org.json.simple.JSONObject();
			
			if (!StringUtils.isNull(pushMessage.getPushKey())) {
				json.put(pushMessage.getPushKey(), StringUtils.validString(pushMessage.getPushValue()));
			}
			
			if (!StringUtils.isNull(appInfo.getPushSound())) {
				messageJson.put(PushPayload.PUSH_SOUND, appInfo.getPushSound());
			}
			
			messageJson.put(PushPayload.PUSH_DATA, json.toString());
			
			log.info("rainboots message [{}]", messageJson.toString());
		} catch (Exception e) {
			log.error("exception in makeRainboots messaage", e);
		}
		
		return messageJson.toString();
	}

	protected PushResult checkValidation(PushQueue pushQueue, String os) {
		StringBuilder sb = new StringBuilder();
		String id = sb.append(pushQueue.getPushId()).append("&&")
				.append(pushQueue.getDeviceId()).append("&&")
				.append(pushQueue.getReqUid()).append("&&")
				.append(pushQueue.getCustId()).toString();
		
		if ("N".equals(pushQueue.getNotiFlag())) {
			return new PushResult(appInfo.getAppGrpId(), DENY_PUSH, id, os, pushQueue.getRowId());
		}
		
		if (App.IOS.equals(os)) {
			try {
				new BasicDevice(pushQueue.getPushToken());
			} catch (InvalidDeviceTokenFormatException e) {
				return new PushResult(appInfo.getAppGrpId(), APNS_INVALID_DEVICE_TOKEN_FORMAT_EXCEPTION, id, os, pushQueue.getRowId());
			}
		}
		
		return null;
	}
}


