package com.humuson.tms.batch.service.impl;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;

import com.humuson.rainboots.client.PushSendClientService;
import com.humuson.rainboots.client.PushSendClientServiceImpl;
import com.humuson.rainboots.proto.messages.PushProtos.PushRequest;
import com.humuson.rainboots.proto.messages.PushProtos.PushResponse;
import com.humuson.tms.batch.service.PrivatePushService;
import com.humuson.tms.common.util.StringUtils;

/**
 * rainboots server에게 푸시 메시지를 전송하기 위한 클래스
 * rainboots와 TCP 통신으로 PushRequest / PushResponse 송.수신 함
 * @author hyogun
 *
 */
public class PrivatePushServiceImpl implements PrivatePushService<PushResponse, PushRequest> {
	
	private static Logger logger = LoggerFactory.getLogger(PrivatePushServiceImpl.class);
	
	@Value("#{config['tms.private.server.list']}")
	private String serverList;
	@Value("#{config['use.tms.private.server']}")
	private boolean usePrivate;
	@Value("#{config['tms.private.timeout.mils']}")
	private long sendTimeoutMils;
	
	private PushSendClientService rainbootsClientService;
	
	@Override
	public void init() {
		if (!StringUtils.isNull(this.serverList) && usePrivate) {
			rainbootsClientService = new PushSendClientServiceImpl(serverList, sendTimeoutMils); 
			rainbootsClientService.init();
		}
	}
	
	@Override
	public void connect() {
	}

	@Override
	public void disconnect() {
		if (usePrivate) {
			rainbootsClientService.disconnect();
		}
	}
	
	@Override
	public boolean isActive() {
		if (!usePrivate || rainbootsClientService == null) return false;
		
		return rainbootsClientService.isActive();
	}
	
	@Override
	public PushResponse request(PushRequest requestProtoBuf) {
		
		if (!isActive()) {
			
			logger.error("rainboots send service is null.... try reconnect...");
			
			rainbootsClientService.connect();
			
			if (!isActive()) {
				logger.error("rainbootsClientService is all down!!!");
				return null;
			}
		}
		
		PushResponse pushResponse = null;
		
		try {
			
			synchronized (rainbootsClientService) {
				pushResponse = rainbootsClientService.request(requestProtoBuf);
			}
			
		} catch (Exception e) {
			logger.error("exception : {}", e);
		}
		
		return pushResponse;
	}

	@Override
	public boolean useRainboots() {
		return this.usePrivate;
	}

	@Override
	public void setRunning(boolean isRunning) {
	}

	@Override
	public boolean isRunning() {
		return false;
	}
}
