package com.humuson.tms.batch.service.impl;

import static com.humuson.tms.constrants.PushResponseConstants.APNS_DUPLICATE_DEVICE_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_NULL_DEVICE_TOKEN_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_NULL_ID_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_PAYLOAD_MAX_SIZE_EXCEEDED_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_UNKNOWN_DEVICE_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.APNS_UNKNOWN_ERROR;
import static com.humuson.tms.constrants.PushResponseConstants.DUPLICATE_DEVICE_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.INVALID_DEVICE_TOKEN_FORMAT_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.NULL_DEVICE_TOKEN_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.NULL_ID_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.PAYLOAD_MAX_SIZE_EXCEEDED_EXCEPTION;
import static com.humuson.tms.constrants.PushResponseConstants.SUCCESSFUL;
import static com.humuson.tms.constrants.PushResponseConstants.UNKNOWN_DEVICE_EXCEPTION;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.jivesoftware.smack.SmackException.NotConnectedException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import com.apple.ios.apns.ApnsConnectionManager;
import com.google.android.gcm.server.Message;
import com.humuson.rainboots.proto.messages.PushProtos.PushRequest;
import com.humuson.rainboots.proto.messages.PushProtos.PushRequest.PushType;
import com.humuson.rainboots.proto.messages.PushProtos.PushRequest.QosLevel;
import com.humuson.rainboots.proto.messages.PushProtos.PushResponse;
import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.service.GcmHttpService;
import com.humuson.tms.batch.service.PrivatePushService;
import com.humuson.tms.batch.service.PushResultService;
import com.humuson.tms.batch.service.PushSendService;
import com.humuson.tms.common.security.HumusonDecryptor;
import com.humuson.tms.common.util.FileUtil;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.mq.model.MgsPush;
import com.humuson.tms.mq.model.MgsPush.GcmMessage;
import com.humuson.tms.mq.model.MgsPush.PushChnType;
import com.humuson.tms.mq.model.MgsPush.PushPayload;
import com.humuson.tms.mq.model.MgsPush.Request;
import com.humuson.tms.mq.model.MgsPush.Response.ResponsePayload;
import com.humuson.tms.mq.model.MgsPush.Response.ResultCode;

import javapns.devices.exceptions.InvalidDeviceTokenFormatException;
import javapns.devices.implementations.basic.BasicDevice;
import javapns.notification.AppleNotificationServer;
import javapns.notification.AppleNotificationServerBasicImpl;
import javapns.notification.PushNotificationManager;
import javapns.notification.PushNotificationPayload;
import javapns.notification.PushedNotification;
import lombok.extern.slf4j.Slf4j;


/**
 * Rainboots / GCM / APNS 으로 푸시 메시지를 전송하기 위한 클래스
 * @author hyogun
 *
 */
@Slf4j
public class MqPushSendServiceImpl implements PushSendService <MgsPush.Request, MgsPush.Response> {
	
	protected static final String DEFAULT_SERVER_NAME = "TMS-SEND-01";
	protected static final String RAINBOOTS_RUN 		= "Y";
	
	@Autowired
	protected PrivatePushService<PushResponse, PushRequest> privateService;
	
	@Value("#{config['use.mgs.public.push']}")
	protected boolean useMgsPublicPush;
	
	@Value("#{config['tms.db.enc.key']}")
	protected String encKey;
	
	protected PushNotificationManager pushManager;
	
	@Autowired
	@Qualifier("gcmHttpMqServiceImpl")
	protected GcmHttpService<ResponsePayload,PushPayload> gcmHttpMqServiceImpl;
	
	@Autowired
	PushResultService pushResultServiceImpl;
	
	@Autowired
	protected GcmCcsService gcmCcsService;
	
	@Autowired
	ApnsConnectionManager apnsConnectionManager;
	
	@Value("#{config['send.gcm.type']}")
	protected String gcmType;

	protected App appInfo;
	
	protected Map<String, Boolean> denyAppVersionMap = new ConcurrentHashMap<String, Boolean>();
	
	@Value("#{config['tms.private.qos.level']}")
	protected int privateQosLevel;

	private String serverName;
	
	static final int DEFAULT_PUSH_LIVE_TIME = 1800;
	
	public MqPushSendServiceImpl() {
		this.serverName = DEFAULT_SERVER_NAME;
		log.debug("PushSendServiceImpl :{} instance generate hashCode:{}", serverName, this.hashCode());
	}
	
	/**
	 * @param serverName
	 */
	public MqPushSendServiceImpl(String serverName) {
		this.serverName = serverName;
		log.debug("PushSendServiceImpl :{} instance generate hashCode:{}", serverName, this.hashCode());
	}
	
	@Override
	public void init(App appInfo) {
		try {
			this.appInfo = appInfo;
			
			String[] denyAppVersions = appInfo.getDenyAppVersion().split(",");
			
			for (String denyAppVer : denyAppVersions) {
				denyAppVersionMap.put(denyAppVer, true);
			}
			
			String apnsCert = appInfo.getApnsPushCert();
			String apnsPwd = appInfo.getApnsPushPwd();
			String gcmApiKey = appInfo.getGcmApiKey();
			long gcmProjectNum = appInfo.getGcmProjectNum();
			
			if(PushChnType.GCM.equals(appInfo.getReqPushChnType())){
			
				if (gcmApiKey != null && gcmType.equalsIgnoreCase("http")) {
					gcmHttpMqServiceImpl.init(appInfo);
				}else if(gcmApiKey != null && gcmType.equalsIgnoreCase("xmpp")){
					gcmCcsService.init(gcmProjectNum, gcmApiKey);
				}
			}else if(PushChnType.APNS.equals(appInfo.getReqPushChnType())) {
				if (apnsCert != null && apnsPwd != null) {
					if (FileUtil.isValidFile(apnsCert)) {
						try {
							String decryptApnsPwd = HumusonDecryptor.decrypt(apnsPwd, encKey);
							pushManager = apnsConnectionManager.getPushNotificationManager(appInfo.getAppKey(App.IOS), apnsCert, decryptApnsPwd);
						} catch (Exception e) {
							log.error("apns init exception [appInfo:{}, encKey:{}] error : {}", appInfo.toString(), encKey, e);
						}
					} else {
						log.error("APNS Cert File {} is not valid ", apnsCert);
					}
				} else {
					log.error("APNS Cert is Null or ApnsPwd is Null {}", appInfo.toString());
				}
			}
		} catch (Exception e) {
			log.error("APNS & GCM init error", e);
		}
	}

	@Override
	public void close() {
//		try {
//			if (pushManager != null) {
//				pushManager.stopConnection();
//			}
//		} catch (Exception e) {
//			log.error("error : {}", e);
//		}
	}

	@Override
	public MgsPush.Response request(Request request, boolean isGcmReSend, boolean unActivePublish)
			throws Exception {
		return request(request,isGcmReSend,unActivePublish,false);
	}

	@Override
	public  MgsPush.Response request(Request request, boolean isGcmReSend, boolean unActivePublish,
			boolean useWakeupGcm) {
		
		Map<String, PushPayload> reqAndroidUserMap = new HashMap<String, PushPayload>();
		List<PushPayload> privateSendingList = new ArrayList<PushPayload>();
		
		MgsPush.Response.Builder mqResponseBuilder = MgsPush.Response.newBuilder();
		ResponsePayload.Builder  responsePayloadBuilder = ResponsePayload.newBuilder();
		
		boolean isGCM = PushChnType.GCM.equals(request.getPushChnType());
		boolean isAPNS = PushChnType.APNS.equals(request.getPushChnType());
		
		String serverId = request.getSeverId();
		
		
		// pirvate push request protoBuf
		PushRequest.Builder builder = PushRequest.newBuilder();
		
		String androidAppKey = appInfo.getAppKey(App.ANDROID);
		if (privateService.useRainboots() 
				&& !StringUtils.isNull(androidAppKey)
				&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())) {
			builder.setAppkey(androidAppKey);
			builder.setAlias(serverName);
			builder.setType(PushType.ONE2ONE);
			builder.setQosLevel(QosLevel.valueOf(privateQosLevel));
			builder.setUnActivePublish(unActivePublish);
		}
		
		
		

		int denyAppVerCount = 0;
		
		
		for(PushPayload payLoad : request.getPayloadList()){
			if(isAPNS){
				if(!checkIosValidation(mqResponseBuilder, responsePayloadBuilder, payLoad, serverId)){
					continue;
				}
				mqResponseBuilder.addResPayload(sendApnsMessage(responsePayloadBuilder,payLoad, serverId));
			}else if (isGCM){
				reqAndroidUserMap.put(payLoad.getId() +":"+serverId, payLoad);
				
				if (privateService.useRainboots()
						&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())) {
					PushRequest.Payload.Builder payloadBuilder = PushRequest.Payload.newBuilder();
//					payload : pushQueue.getPushId()+"&&"+pushQueue.getDeviceId()+"&&"+push Queue.getReqUid()+"&&"+pushQueue.getCustId();
					String id = payLoad.getId()+":"+serverId;
					payloadBuilder.setToken(id.split("&&")[1]);
					
					payloadBuilder.setId(id);
					
					payloadBuilder.setTimeToLive(payLoad.getGcmMessage().getTimeToLive() > 0 ? payLoad.getGcmMessage().getTimeToLive()  :DEFAULT_PUSH_LIVE_TIME);
					// private server에서 msg Id를 발번하도록 기본 값으로 세팅함
					payloadBuilder.setMsgId(0);
					
					String realTimeRainbootsMsg = makeRainbootsMessage(payLoad.getGcmMessage());
					
					log.debug("realtime rainboots send message : {}", realTimeRainbootsMsg);
					payloadBuilder.setMessage(realTimeRainbootsMsg);
					
					builder.addPayload(payloadBuilder.build());
					
				}
			}
		}
		
		if(isGCM){
			long t = System.currentTimeMillis();
			if (privateService.useRainboots()
					&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())
					&& builder.getPayloadCount() > 0) {		// send private message
				long startTime = System.currentTimeMillis();
				
				log.debug("rainboots reqeust : {}", builder.build());
				
				List<ResponsePayload> rainbootsResult = sendRainboots(responsePayloadBuilder,privateSendingList, reqAndroidUserMap, builder.build(), isGcmReSend, false);
				mqResponseBuilder.addAllResPayload(rainbootsResult);
				log.debug("SEND PUSH RAINBOOTS REQUEST DURATION TIME :{}", System.currentTimeMillis() - startTime);
			} 
			log.debug("rain t {}", System.currentTimeMillis() - t);
			if (reqAndroidUserMap.size() > 0 && isGcmReSend) {
				Collection<PushPayload> androidList = reqAndroidUserMap.values();
				try{
					long startTime = System.currentTimeMillis();
					
					if("http".equalsIgnoreCase(gcmType)){
						gcmHttpMqServiceImpl.setServerId(serverId);
						List<ResponsePayload> payLoadList = gcmHttpMqServiceImpl.sendGcmOne2OneList(androidList, appInfo.getGcmApiKey());
						for(ResponsePayload payloadData : payLoadList){
							mqResponseBuilder.addResPayload(payloadData);
						}
					}else if ("xmpp".equalsIgnoreCase(gcmType)){
						for (PushPayload payload : androidList) {
							String ccsMessage = gcmCcsService.makeCcsMessage(payload,appInfo.getAppGrpId(),request.getSeverId());
							try {
								gcmCcsService.sendDownstreamMessage(ccsMessage);							
								
							} catch (Exception e) {
								
								try{
									log.error("NotConnectedException in GcmSend (XMPP) retry 1 more: {}", payload.getId(), e);
									gcmCcsService.reset();
									gcmCcsService.sendDownstreamMessage(ccsMessage);
								}catch (NotConnectedException e2){
									log.error("NotConnectedException in GcmSend (XMPP) Retry Fail: {}", payload.getId(), e);
									responsePayloadBuilder.clear();
									responsePayloadBuilder.setId(payload.getId());
									responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.ERROR_UNAVAILABLE);
									mqResponseBuilder.addResPayload(responsePayloadBuilder.build());
									gcmCcsService.setHitCount(0);
									gcmCcsService.reset();
									continue;
								}
							} 
							responsePayloadBuilder.clear();
							responsePayloadBuilder.setId(payload.getId());
							responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.SENDING_CCS);
							mqResponseBuilder.addResPayload(responsePayloadBuilder.build());
						}
					}
					log.info("send {} [count:{}, elpasedTime:{} ms]", gcmType, androidList.size(), System.currentTimeMillis() - startTime);
				}catch (Exception e){
					log.error("GCM Send Error", e);
				}
			} else if (reqAndroidUserMap.size() > 0){
				log.error("reqAndroidUserMap size :{} gcmReSendFlag:{}", reqAndroidUserMap.size(), isGcmReSend);
				Collection<PushPayload> androidList = reqAndroidUserMap.values();
				StringBuilder sb = new StringBuilder();
				
				for (PushPayload payload : androidList) {
					responsePayloadBuilder.clear();
					responsePayloadBuilder.setId(payload.getId());
					responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.NO_SEND);
					mqResponseBuilder.addResPayload(responsePayloadBuilder.build());
				}
			}
		}
		if(isGCM){
			mqResponseBuilder.setAppKey(appInfo.getAppKey(App.ANDROID));
			mqResponseBuilder.setPushChnType(PushChnType.GCM);
		}else if(isAPNS){
			mqResponseBuilder.setAppKey(appInfo.getAppKey(App.IOS));
			mqResponseBuilder.setPushChnType(PushChnType.APNS);
		}
		
		if(!privateSendingList.isEmpty()){
			pushResultServiceImpl.batchInsertSendingStatus(privateSendingList, appInfo, serverId);
		}
		
		mqResponseBuilder.setResultCode(ResultCode.SUCCESS);
		
		try{
			return mqResponseBuilder.build();
		}catch( Exception e ){
			log.error("mqResponseBuilder make response Fail : {}",e);
			return null;
		}finally{
			mqResponseBuilder.clear();
		}
	}

	@Override
	public MgsPush.Response request(Request request, PushMessage pushMessage, boolean isGcmReSend) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public MgsPush.Response request(Request request, PushMessage pushMessage, boolean isGcmReSend,
			boolean unActivePublish) throws Exception {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public MgsPush.Response request(Request request, PushMessage pushMessage, boolean isGcmReSend,
			boolean unActivePublish, boolean useWakeupGcm) {
		// TODO Auto-generated method stub
		return null;

	}

	protected void openApnsConnection (String apnsCertPath, String password, boolean isReal) {
		if (pushManager == null) {
			try {
				AppleNotificationServer server = new AppleNotificationServerBasicImpl(apnsCertPath, password, isReal);
				pushManager = new PushNotificationManager();
				
				//connection open
				pushManager.initializeConnection(server);
				
			} catch (Exception e) {
				log.error("error : {}", e);
				try {
					if (pushManager != null) {
						pushManager.stopConnection();
					}
				} catch (Exception e2) {
					log.error("stop connection error", e2);
				} finally {
					pushManager = null;
				}
			}
		}
	}
	
	protected boolean checkIosValidation(MgsPush.Response.Builder mqResponseBuilder, ResponsePayload.Builder responsePayloadBuilder, PushPayload payLoad, String serverId) {
		boolean isValid = true;
		try {
			new BasicDevice(payLoad.getToken());
		} catch (InvalidDeviceTokenFormatException e) {
			isValid=false;
			responsePayloadBuilder.clear();
			responsePayloadBuilder.setId(payLoad.getId());
			responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.APNS_INVALID_DEVICE_TOKEN_FORMAT_EXCEPTION);
			responsePayloadBuilder.setServerId(serverId);
			mqResponseBuilder.addResPayload(responsePayloadBuilder.build());
		}
	
		return isValid;
	}
	
	/**
	 * APNS 전송 메소드
	 * @param pushQueue
	 * @return
	 */
	public synchronized MgsPush.Response.ResponsePayload sendApnsMessage(ResponsePayload.Builder responsePayloadBuilder, MgsPush.PushPayload pushPayload, String serverId) {
		
		StringBuilder sb = new StringBuilder();
		sb.setLength(0);
		if (pushManager == null) {
			responsePayloadBuilder.clear();
			responsePayloadBuilder.setId(pushPayload.getId());
			responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.APNS_UNKNOWN_ERROR);
			
			return responsePayloadBuilder.build();
		}
		
		try {
			String message =  pushPayload.getApnsMessage().getMessage();
			String pushKey = pushPayload.getApnsMessage().getPushKey();
			String pushValue = pushPayload.getApnsMessage().getPushValue();
			PushNotificationPayload payload = PushNotificationPayload.alert(message);
			
			if (appInfo.getPushSound() == null) {
				payload.addSound("default");
			} else {
				payload.addSound(appInfo.getPushSound());
			}

			if (appInfo.getIOsBadgeCount() > 0) {
				payload.addBadge(appInfo.getIOsBadgeCount());
			}
			
			payload.addCustomDictionary(com.humuson.tms.constrants.PushPayload.MSG_TYPE, pushPayload.getApnsMessage().getMsgType());
			payload.addCustomDictionary(com.humuson.tms.constrants.PushPayload.MSG_ID, String.valueOf(pushPayload.getApnsMessage().getMsgId()));
			
			if (log.isDebugEnabled())
				log.debug("apns payload add before size:{} payload:{}", payload.getPayloadSize(), payload.toString());
			
			// payload size에 key/value를 삽입할 수 있는 경우에 customDictionary를 추가함
			if (!StringUtils.isNull(pushKey)
					&& payload.isEstimatedPayloadSizeAllowedAfterAdding(pushKey, pushValue)) {
				payload.addCustomDictionary(pushKey, pushValue);
				if (log.isDebugEnabled())
					log.debug("apns payload push key add after size:{} payload:{}", payload.getPayloadSize(), payload.toString());
			}
			
			BasicDevice device = new BasicDevice(pushPayload.getToken());
			
			PushedNotification notification = pushManager.sendNotification(device, payload, false);
			
//			log.info("====== apns payload size:{} payload:{} =====", payload.getPayloadSize(), payload.toString());
			
			String errorCode = null;
			if (notification.isSuccessful()) {
				errorCode = SUCCESSFUL;
			} else {
				errorCode = APNS_UNKNOWN_ERROR;
				
				if (notification.getException() != null) {
					String[] responseParts = notification.getException().toString().split(":");
						
					String err = responseParts[0];
					
					log.error("APNS error : {}", notification.getException());
					
			        if (err.contains(DUPLICATE_DEVICE_EXCEPTION)
			        		|| err.contains(INVALID_DEVICE_TOKEN_FORMAT_EXCEPTION)) {
			        	errorCode = APNS_DUPLICATE_DEVICE_EXCEPTION;
			        } else if (err.contains(NULL_DEVICE_TOKEN_EXCEPTION)) {
			        	errorCode = APNS_NULL_DEVICE_TOKEN_EXCEPTION;
			        } else if (err.contains(NULL_ID_EXCEPTION)) {
			        	errorCode = APNS_NULL_ID_EXCEPTION;
			        } else if (err.contains(UNKNOWN_DEVICE_EXCEPTION)) {
			        	errorCode = APNS_UNKNOWN_DEVICE_EXCEPTION;
			        } else if (err.contains(PAYLOAD_MAX_SIZE_EXCEEDED_EXCEPTION)) {
			        	errorCode = APNS_PAYLOAD_MAX_SIZE_EXCEEDED_EXCEPTION;
			        }
				}
			}
			 
			responsePayloadBuilder.clear();
			responsePayloadBuilder.setId(pushPayload.getId());
			responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.valueOf(Integer.parseInt(errorCode)));
			responsePayloadBuilder.setServerId(serverId);
			return responsePayloadBuilder.build();
		} catch (Exception e) {
			log.error("APNS Send Error", e);
			if (e instanceof InvalidDeviceTokenFormatException) {
				responsePayloadBuilder.clear();
				responsePayloadBuilder.setId(pushPayload.getId());
				responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.APNS_INVALID_DEVICE_TOKEN_FORMAT_EXCEPTION);
				responsePayloadBuilder.setServerId(serverId);
				return responsePayloadBuilder.build();
			} else {
				responsePayloadBuilder.clear();
				responsePayloadBuilder.setId(pushPayload.getId());
				responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.APNS_UNKNOWN_ERROR);
				responsePayloadBuilder.setServerId(serverId);
				return responsePayloadBuilder.build();	
			}
		}
	}
	
	protected String makeRainbootsMessage(GcmMessage message) {
		
		org.json.simple.JSONObject messageJson = new org.json.simple.JSONObject();
		
		try {
			messageJson.put(com.humuson.tms.constrants.PushPayload.PUSH_NOTI_MSG, StringUtils.validString(message.getPushNotiMsg()));
			messageJson.put(com.humuson.tms.constrants.PushPayload.PUSH_NOTI_TITLE, StringUtils.validString(message.getPushNotiTitle()));
			
			String pushImage = message.getPushNotiImg();
			
			if (pushImage != null 
					&& (!pushImage.startsWith("http://"))
					&& "http://".length() >= pushImage.length()) {
				pushImage = "";
			}
			
			messageJson.put(com.humuson.tms.constrants.PushPayload.PUSH_NOTI_IMG, pushImage == null ? "": pushImage);
			messageJson.put(com.humuson.tms.constrants.PushPayload.PUSH_RICH_CONTENT, message.getPushRichContent());
			messageJson.put(com.humuson.tms.constrants.PushPayload.MSG_TYPE, message.getMsgType());
			messageJson.put(com.humuson.tms.constrants.PushPayload.MSG_ID, String.valueOf(message.getMsgId()));
			
			org.json.simple.JSONObject json = new org.json.simple.JSONObject();
			
//			if (!StringUtils.isNull(message.getPushKey())) {
//				json.put(pushMessage.getPushKey(), StringUtils.validString(pushMessage.getPushValue()));
//			}
//			
//			if (!StringUtils.isNull(appInfo.getPushSound())) {
//				messageJson.put(com.humuson.tms.constrants.PushPayload.PUSH_SOUND, appInfo.getPushSound());
//			}
//			
			messageJson.put(com.humuson.tms.constrants.PushPayload.PUSH_DATA, message.getPushData());
			
//			log.info("rainboots message [{}]", messageJson.toString());
		} catch (Exception e) {
			log.error("exception in makeRainboots messaage", e);
		}
		
		return messageJson.toString();
	}
	
	protected List<ResponsePayload> sendRainboots(ResponsePayload.Builder responsePayloadBuilder, List<PushPayload>privateSendingList, Map<String, PushPayload> androidMap, 
			PushRequest pushRequest, boolean isGcmReSend, 
			boolean useWakeupGcm) {
		
		long startTime = System.currentTimeMillis();
		
		PushResponse response = privateService.request(pushRequest);
		
		List<ResponsePayload> resultList = new ArrayList<ResponsePayload>();
		
		if (response != null 
				&& response.getResult() == PushResponse.ResultType.SUCCESS) {
			log.debug("private server send result : {}", response.toString());
			
			int payloadSize = response.getPayloadCount();
			
			for (int i=payloadSize-1; i>=0; i--) {
//				String ids[] = response.getPayload(i).getId().split("_");
				PushPayload pushPayload = androidMap.get(response.getPayload(i).getId());
				if (response.getPayload(i).getResult() == PushResponse.ResultType.SUCCESS) {
					responsePayloadBuilder.clear();
					responsePayloadBuilder.setId(response.getPayload(i).getId());
					responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.SENDING);
					resultList.add(responsePayloadBuilder.build());
					androidMap.remove(response.getPayload(i).getId());
					privateSendingList.add(pushPayload);
				} else if (!isGcmReSend) {
					responsePayloadBuilder.clear();
					responsePayloadBuilder.setId(response.getPayload(i).getId());
					
					if (response.getPayload(i).getResult() == PushResponse.ResultType.UNACTIVED_TOKEN) {
						responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.PRIVATE_UNACTIVED_TOKEN);
					} else if (response.getPayload(i).getResult() == PushResponse.ResultType.INVALID_TOKEN) {
						responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.PRIVATE_INVALID_TOKEN);
					}
					resultList.add(responsePayloadBuilder.build());
					androidMap.remove(response.getPayload(i).getId());
				}
				
				//androidMap Key = PushId()+"_"+DeviceId() 
				//payload.getId() = PushId()+"_"+DeviceId()+"_"+ReqUid() + "_" + "cust_id"
				//androidMap.remove(ids[0]+"_"+ids[1]);
				
				if (log.isDebugEnabled()) {
					log.debug("return code : {}", response.getPayload(i).toString());
				}
			}
			
			long rainbootsElapsedTime = System.currentTimeMillis() - startTime;
			log.info("rainboots send [sendCount:{}, elapsedTime:{}]", pushRequest.getPayloadCount(), rainbootsElapsedTime);
			
			// Send GCM WakeUp Push Start
			if (useWakeupGcm) {
				sendWakeupGcm(pushRequest, androidMap);
			}
			// Send GCM WakeUp Push End
			
		} else {
			log.info("private server send fail result : {}", response == null ? "null" : response.getResult().toString());
			if (!isGcmReSend) {
				Set<String> sendRawIdSet = androidMap.keySet();
				for (String ids : sendRawIdSet) {
					PushPayload payLoad = androidMap.get(ids);
					responsePayloadBuilder.clear();
					responsePayloadBuilder.setId(payLoad.getId());
					responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.PRIVATE_SERVER_ERROR);
					resultList.add(responsePayloadBuilder.build());
				}
				androidMap.clear();
			}
		}
		return resultList;
		
	}
	
	protected void sendWakeupGcm(PushRequest pushRequest, Map<String, PushPayload> androidMap) {
		long startTime = System.currentTimeMillis();
		Collection<PushPayload> wakeupGcmList = androidMap.values();
		PushPayload[] wakeupSendRaw = new PushPayload[wakeupGcmList.size()];
		wakeupGcmList.toArray(wakeupSendRaw);
		
		Message.Builder builder = new Message.Builder();
		builder.timeToLive(180);
		builder.delayWhileIdle(false);
		gcmHttpMqServiceImpl.sendGcmMulticastMessage(wakeupSendRaw, builder.build(),appInfo.getGcmApiKey());
		
		long gcmWakeUpPushElapsedTime = System.currentTimeMillis() - startTime;
		log.info("gcm wakeup push count : {} elapsedTime : {}", pushRequest.getPayloadCount(), gcmWakeUpPushElapsedTime);
	}
	
	
}


