package com.humuson.tms.batch.service.impl;

import static com.humuson.tms.constrants.PushResponseConstants.MQ_SENDING;
import static com.humuson.tms.constrants.PushResponseConstants.NO_SEND;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.json.simple.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import com.google.android.gcm.server.Message;
import com.humuson.rainboots.datastore.DataStore;
import com.humuson.rainboots.proto.messages.PushProtos.PushRequest;
import com.humuson.rainboots.proto.messages.PushProtos.PushRequest.PushType;
import com.humuson.rainboots.proto.messages.PushProtos.PushRequest.QosLevel;
import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.domain.PushQueue;
import com.humuson.tms.batch.domain.PushResult;
import com.humuson.tms.batch.service.MqProducer;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.constrants.PushResponseConstants;
import com.humuson.tms.mq.model.MgsPush;
import com.humuson.tms.mq.model.MgsPush.PushChnType;
import com.humuson.tms.mq.model.MgsPush.PushPayload;
import com.humuson.tms.mq.model.MgsPush.Request.AckMode;
import com.humuson.tms.mq.model.MgsPush.Request.SendType;

import lombok.extern.slf4j.Slf4j;


/**
 * Rainboots / GCM / APNS 으로 푸시 메시지를 전송하기 위한 클래스
 * @author hyogun
 *
 */
@Slf4j
public class MgsPushSendServiceImpl extends PushSendServiceImpl {
	
	@Autowired private MqProducer mgsProducer;
	
	@Value("#{config['use.mgs.public.push']}")
	protected boolean useMgsPublicPush;
	/**
	 * 기본 생성자
	 */
	public MgsPushSendServiceImpl() {
		log.debug("MQPushSendServiceImpl :{} instance generate hashCode:{}", serverName, this.hashCode());
	}
	
	/**
	 * @param serverName
	 */
	public MgsPushSendServiceImpl(String serverName) {
		super(serverName);
	}
	
	@Override
	public synchronized void init(App appInfo) {
		if (!useMgsPublicPush) {
			super.init(appInfo);
		} else {
			this.appInfo = appInfo;
			
			String[] denyAppVersions = appInfo.getDenyAppVersion().split(",");
			
			for (String denyAppVer : denyAppVersions) {
				denyAppVersionMap.put(denyAppVer, true);
			}
		}
	}
	
	@Override
	public void close() {
		if (!useMgsPublicPush) {
			super.close();
		}
	}
	
	/*
	 * (non-Javadoc)
	 * @see com.humuson.tms.batch.service.impl.PushSendServiceImpl#request(java.util.List, boolean, boolean, boolean)
	 */
	@Override
	public List<PushResult> request(List<? extends PushQueue> list,
			boolean isGcmReSend, boolean unActivePublish, boolean useWakeupGcm) {
		if (!useMgsPublicPush) {
			return super.request(list, isGcmReSend, unActivePublish, useWakeupGcm);
		}
		Map<String, PushQueue> reqAndroidUserMap = new HashMap<String, PushQueue>();
		List<PushResult> pushResultList = new ArrayList<PushResult>();
		
		// pirvate push request protoBuf
		PushRequest.Builder builder = PushRequest.newBuilder();
		
		String androidAppKey = appInfo.getAppKey(App.ANDROID);
		if (privateService.useRainboots() 
				&& !StringUtils.isNull(androidAppKey)
				&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())) {
			builder.setAppkey(androidAppKey);
			builder.setAlias(serverName);
			builder.setType(PushType.ONE2ONE);
			builder.setQosLevel(QosLevel.valueOf(privateQosLevel));
			builder.setUnActivePublish(unActivePublish);
		}
		
		int denyAppVerCount = 0;
		/*
		MgsPush.Request.Builder gcmRequestBuilder = MgsPush.Request.newBuilder()
				.setAppKey(appInfo.getAppKey(App.ANDROID))
				.setAckMode(AckMode.ACK)
				.setPushChnType(PushChnType.GCM)
				.setSendType(SendType.ONE2ONE);
		
		MgsPush.Request.Builder apnsRequestBuilder = MgsPush.Request.newBuilder()
				.setAppKey(appInfo.getAppKey(App.IOS))
				.setAckMode(AckMode.ACK)
				.setPushChnType(PushChnType.APNS)
				.setSendType(SendType.ONE2ONE);
		*/
		/* Android-iOS 중 한쪽만 등록된 앱일 경우 발생하는 에러의 버그 패치 */
		MgsPush.Request.Builder gcmRequestBuilder = null;
		MgsPush.Request.Builder apnsRequestBuilder = null;
		if (appInfo.getAppKey(App.ANDROID) != null) {
			gcmRequestBuilder = MgsPush.Request.newBuilder()
					.setAppKey(appInfo.getAppKey(App.ANDROID))
					.setAckMode(AckMode.ACK)
					.setPushChnType(PushChnType.GCM)
					.setSendType(SendType.ONE2ONE);
		}
		if (appInfo.getAppKey(App.IOS) != null) {
			apnsRequestBuilder = MgsPush.Request.newBuilder()
					.setAppKey(appInfo.getAppKey(App.IOS))
					.setAckMode(AckMode.ACK)
					.setPushChnType(PushChnType.APNS)
					.setSendType(SendType.ONE2ONE);
		}
		/* Android-iOS 중 한쪽만 등록된 앱일 경우 발생하는 에러의 버그 패치 */
		
		MgsPush.PushPayload.Builder pushPayloadBuilder = MgsPush.PushPayload.newBuilder();
			
		StringBuilder sb = new StringBuilder();
		String id = null;
		for (PushQueue pushQueue : list) {
			sb.setLength(0);
			id = sb.append(pushQueue.getPushId()).append("&&")
			.append(pushQueue.getDeviceId()).append("&&")
			.append(pushQueue.getReqUid()).append("&&")
			.append(pushQueue.getCustId()).toString();
			pushPayloadBuilder.clear();
			
			PushResult pushResult = this.checkValidation(pushQueue, appInfo.getOs(pushQueue.getAppId()));
			
			if (pushResult != null) {
				pushResultList.add(pushResult);
				continue;
			}
			
			if (App.IOS.equals(appInfo.getOs(pushQueue.getAppId()))) { 	// iOS
				
				pushPayloadBuilder.setToken(pushQueue.getPushToken())
					.setId(id)
					.setApnsMessage(makeApnsMqMessage(pushQueue.getPushMessage()));
				
				if(apnsRequestBuilder != null) apnsRequestBuilder.addPayload(pushPayloadBuilder.build());
				
				pushResultList.add(new PushResult(appInfo.getAppGrpId(), MQ_SENDING, 
						id, 
						App.IOS,pushQueue.getRowId()));
			} else {	// Android
				
				// make to Rainboots send list
				PushRequest.Payload.Builder payloadBuilder = PushRequest.Payload.newBuilder();
				
				if (privateService.useRainboots() 
						&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())) {
					
					payloadBuilder.setId(id);
					payloadBuilder.setToken(String.valueOf(pushQueue.getDeviceId()));
					payloadBuilder.setTimeToLive(pushQueue.getPushMessage().getPushTtl());
					// private server에서 msg Id를 발번하도록 기본 값으로 세팅함
					payloadBuilder.setMsgId(0);
					
					String realTimeRainbootsMsg = makeRainbootsMessage(pushQueue.getPushMessage());
					
					log.debug("realtime rainboots send message : {}", realTimeRainbootsMsg);
					payloadBuilder.setMessage(realTimeRainbootsMsg);
					
					builder.addPayload(payloadBuilder.build());
				} 
				
				reqAndroidUserMap.put(pushQueue.getPushId()+"_"+pushQueue.getDeviceId(), pushQueue);
			}		// end Android
		}		// end for loop
		
		if (denyAppVerCount > 0) {
			log.info("deny app version filter count : {}", denyAppVerCount);
		}
		
		if (reqAndroidUserMap.size() > 0) {		// android
			
			if (privateService.useRainboots()
					&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())
					&& builder.getPayloadCount() > 0) {		// send private message
				long startTime = System.currentTimeMillis();
				
				if (log.isDebugEnabled()) {
					log.debug("rainboots reqeust : {}", builder.build());
				}
				
				List<PushResult> rainbootsResult = sendRainboots(reqAndroidUserMap, builder.build(), isGcmReSend, false);
				
				pushResultList.addAll(rainbootsResult);
				
				if (log.isDebugEnabled()) {
					log.debug("SEND PUSH RAINBOOTS REQUEST DURATION TIME :{}", System.currentTimeMillis() - startTime);
				}
			} 
			
			// gcm resend or rainboots fail or no use rainboots
			if (!reqAndroidUserMap.isEmpty() && isGcmReSend) {
				Collection<PushQueue> androidList = reqAndroidUserMap.values();
				
				for (PushQueue gcmPushQueue : androidList) {
					sb.setLength(0);
					id = sb.append(gcmPushQueue.getPushId()).append("&&")
					.append(gcmPushQueue.getDeviceId()).append("&&")
					.append(gcmPushQueue.getReqUid()).append("&&")
					.append(gcmPushQueue.getCustId()).toString();
					pushPayloadBuilder.setId(id)
						.setToken(gcmPushQueue.getPushToken())
						.setGcmMessage(makeGcmMqMessage(gcmPushQueue.getPushMessage()));
					
					if(gcmRequestBuilder != null) gcmRequestBuilder.addPayload(pushPayloadBuilder.build());
					
					pushResultList.add(new PushResult(appInfo.getAppGrpId(), MQ_SENDING, id, App.ANDROID,gcmPushQueue.getRowId()));
				}
			} else if (reqAndroidUserMap.size() > 0){
				log.error("reqAndroidUserMap size :{} gcmReSendFlag:{}", reqAndroidUserMap.size(), isGcmReSend);
				Collection<PushQueue> androidList = reqAndroidUserMap.values();
				for (PushQueue pushQueue : androidList) {
					sb.setLength(0);
					id = sb.append(pushQueue.getPushId()).append("&&")
					.append(pushQueue.getDeviceId()).append("&&")
					.append(pushQueue.getReqUid()).append("&&")
					.append(pushQueue.getCustId()).toString();
					pushResultList.add(new PushResult(appInfo.getAppGrpId(), NO_SEND, 
							id, 
							App.ANDROID, pushQueue.getRowId()));
				}
			}
		}
		
		if (apnsRequestBuilder !=null && !apnsRequestBuilder.getPayloadList().isEmpty()) {
			log.info("mgs apns public push send [payloadSize:{}, payload:{}]", apnsRequestBuilder.getPayloadList().size(), apnsRequestBuilder.build().toString());
			mgsProducer.send(apnsRequestBuilder.build());
		}
		
		if (gcmRequestBuilder != null && !gcmRequestBuilder.getPayloadList().isEmpty()) {
			log.info("mgs gcm public push send [payloadSize:{}]", gcmRequestBuilder.getPayloadList().size());
			mgsProducer.send(gcmRequestBuilder.build());
		}

		return pushResultList;
	}
	
	private MgsPush.ApnsMessage makeApnsMqMessage(PushMessage pushMessage) {
		MgsPush.ApnsMessage.Builder apnsMsgBuilder = MgsPush.ApnsMessage.newBuilder();
		apnsMsgBuilder
			.setMessage(pushMessage.getPushTitle()+"\n"+pushMessage.getPushMsg())
			.setSound(StringUtils.validString(appInfo.getPushSound()))
			.setMsgType(pushMessage.getMsgType())
			.setMsgId(String.valueOf(pushMessage.getMsgUid()));
	
		if (!StringUtils.isNull(pushMessage.getPushKey())) {
			apnsMsgBuilder.setPushKey(pushMessage.getPushKey())
				.setPushValue(StringUtils.validString(pushMessage.getPushValue()));
		}
		
		return apnsMsgBuilder.build();
	}
	
	@SuppressWarnings("unchecked")
	private MgsPush.GcmMessage makeGcmMqMessage(PushMessage pushMessage) {
		MgsPush.GcmMessage.Builder gcmMsgBuilder = MgsPush.GcmMessage.newBuilder();

		gcmMsgBuilder.setTimeToLive(pushMessage.getPushTtl())
		.setDelayWhileIdle(false)
		.setPushNotiMsg(StringUtils.validString(pushMessage.getPushMsg()))
		.setPushRichContent(StringUtils.validString(pushMessage.getPopupContent()))
		.setPushNotiTitle(StringUtils.validString(pushMessage.getPushTitle()))
		.setPushNotiImg(StringUtils.validString(pushMessage.getPushImg()))
		.setMsgType(pushMessage.getMsgType())
		.setMsgId(pushMessage.getMsgUid());
		
		JSONObject json = new JSONObject();
		
		if (!StringUtils.isNull(pushMessage.getPushKey())) {
			json.put(pushMessage.getPushKey(), StringUtils.validString(pushMessage.getPushValue()));
		}
		gcmMsgBuilder.setPushData(json.toString());
		
		return gcmMsgBuilder.build();
	}
	
		
	/*
	 * (non-Javadoc)
	 * @see com.humuson.tms.batch.service.impl.PushSendServiceImpl#request(java.util.List, com.humuson.tms.batch.domain.PushMessage, boolean, boolean, boolean)
	 */
	@Override
	public List<PushResult> request(List<? extends PushQueue> list, PushMessage pushMessage,
			boolean isGcmReSend, boolean unActivePublish, boolean useWakeupGcm) {
		
		if (!useMgsPublicPush) {
			return super.request(list, pushMessage, isGcmReSend, unActivePublish, useWakeupGcm);
		}
		
		Map<String, PushQueue> reqAndroidUserMap = new HashMap<String, PushQueue>();
		List<PushResult> pushResultList = new ArrayList<PushResult>();
		
		// pirvate push request protoBuf
		PushRequest.Builder privateReqBuilder = PushRequest.newBuilder();
					
		if (privateService.useRainboots() && appInfo.getAppKey(App.ANDROID) != null
				&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())) {
			
			privateReqBuilder.setAppkey(appInfo.getAppKey(App.ANDROID));
			privateReqBuilder.setAlias(serverName);
			
			privateReqBuilder.setType(PushType.BROADCAST);
			// private server에서 임시적으로 저장하기 위한 msg id는 
			// 32767를 넘지 않로록 PMS의 msgId와 mod 연산한 결과 값을 세팅 함
			privateReqBuilder.setMsgId((int) (Long.parseLong(pushMessage.getMsgUid()) % DataStore.MAX_BULK_MSG_ID));
		
			privateReqBuilder.setQosLevel(QosLevel.valueOf(privateQosLevel));
			privateReqBuilder.setUnActivePublish(unActivePublish);
			
			if (pushMessage.getPushTtl() > 0) {
				privateReqBuilder.setTimeToLive(pushMessage.getPushTtl());
			}
			
			String rainbootsBulkMsg = makeRainbootsMessage(pushMessage);
			
			privateReqBuilder.setMessage(rainbootsBulkMsg);
		}
		/*
		MgsPush.Request.Builder mgsGcmRequestBuilder = MgsPush.Request.newBuilder()
				.setAppKey(appInfo.getAppKey(App.ANDROID))
				.setAckMode(AckMode.ACK)
				.setSendType(SendType.MULTICAST)
				.setGcmMessage(makeGcmMqMessage(pushMessage))
				.setPushChnType(PushChnType.GCM);
		
		MgsPush.Request.Builder apnsRequestBuilder = MgsPush.Request.newBuilder()
				.setAppKey(appInfo.getAppKey(App.IOS))
				.setAckMode(AckMode.ACK)
				.setSendType(SendType.MULTICAST)
				.setApnsMessage(makeApnsMqMessage(pushMessage))
				.setPushChnType(PushChnType.APNS);
		*/
		/* Android-iOS 중 한쪽만 등록된 앱일 경우 발생하는 에러의 버그 패치 */
		MgsPush.Request.Builder mgsGcmRequestBuilder = null;
		MgsPush.Request.Builder apnsRequestBuilder = null;
		if (appInfo.getAppKey(App.ANDROID) != null) {
			mgsGcmRequestBuilder = MgsPush.Request.newBuilder()
					.setAppKey(appInfo.getAppKey(App.ANDROID))
					.setAckMode(AckMode.ACK)
					.setSendType(SendType.MULTICAST)
					.setGcmMessage(makeGcmMqMessage(pushMessage))
					.setPushChnType(PushChnType.GCM);
		}
		if (appInfo.getAppKey(App.IOS) != null) {
			apnsRequestBuilder = MgsPush.Request.newBuilder()
					.setAppKey(appInfo.getAppKey(App.IOS))
					.setAckMode(AckMode.ACK)
					.setSendType(SendType.MULTICAST)
					.setApnsMessage(makeApnsMqMessage(pushMessage))
					.setPushChnType(PushChnType.APNS);
		}
		/* Android-iOS 중 한쪽만 등록된 앱일 경우 발생하는 에러의 버그 패치 */
		
		MgsPush.PushPayload.Builder mgsPublicPayloadBuilder = MgsPush.PushPayload.newBuilder();
		StringBuilder sb = new StringBuilder();
		String id = null;
		for (PushQueue pushQueue : list) {
			mgsPublicPayloadBuilder.clear();
			sb.setLength(0);
			id = sb.append(pushQueue.getPushId()).append("&&")
			.append(pushQueue.getDeviceId()).append("&&")
			.append(pushQueue.getReqUid()).append("&&")
			.append(pushQueue.getCustId()).toString();
			PushResult pushResult = this.checkValidation(pushQueue, appInfo.getOs(pushQueue.getAppId()));
			if (pushResult != null) {
				pushResultList.add(pushResult);
				continue;
			}
			
			mgsPublicPayloadBuilder.clear();
			if (App.IOS.equals(appInfo.getOs(pushQueue.getAppId()))) { 	// iOS
				if(apnsRequestBuilder != null) apnsRequestBuilder.addPayload(
						mgsPublicPayloadBuilder.setId(id).setToken(pushQueue.getPushToken()).build());
				pushResultList.add(new PushResult(appInfo.getAppGrpId(), MQ_SENDING, id, App.IOS,pushQueue.getRowId()));
			} else {	// Android
				// make to Rainboots send list
				reqAndroidUserMap.put(pushQueue.getPushId()+"_"+pushQueue.getDeviceId(), pushQueue);
				
				if (privateService.useRainboots()
						&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())) {
					PushRequest.Payload.Builder payloadBuilder = PushRequest.Payload.newBuilder();
					payloadBuilder.setId(id).setToken(String.valueOf(pushQueue.getDeviceId()));
					privateReqBuilder.addPayload(payloadBuilder.build());
				}
			}		// end Android
		}		// end for loop
		
		if (!reqAndroidUserMap.isEmpty()) {	// android
			
			if (privateService.useRainboots()
					&& RAINBOOTS_RUN.equals(appInfo.getPrivateFlag())
					&& privateReqBuilder.getPayloadCount() > 0) {		// send private message
				long startTime = System.currentTimeMillis();
				List<PushResult> rainbootsResult = sendRainboots(reqAndroidUserMap, privateReqBuilder.build(), isGcmReSend, useWakeupGcm);
				pushResultList.addAll(rainbootsResult);
				log.info("send rainboots [size:{}, elapseTime:{}]", reqAndroidUserMap.size(), System.currentTimeMillis() - startTime);
			} 
			
			// gcm resend or rainboots fail or no use rainboots
			if (isGcmReSend) {
				Collection<PushQueue> androidList = reqAndroidUserMap.values();

				for (PushQueue pushQueue : androidList) {
					sb.setLength(0);
					id = sb.append(pushQueue.getPushId()).append("&&")
					.append(pushQueue.getDeviceId()).append("&&")
					.append(pushQueue.getReqUid()).append("&&")
					.append(pushQueue.getCustId()).toString();
					mgsPublicPayloadBuilder.clear();
					mgsPublicPayloadBuilder.setId(id).setToken(pushQueue.getPushToken());
					
					if(mgsGcmRequestBuilder != null) mgsGcmRequestBuilder.addPayload(mgsPublicPayloadBuilder.build());
					pushResultList.add(new PushResult(appInfo.getAppGrpId(), MQ_SENDING, id, App.ANDROID,pushQueue.getRowId()));
				}
					
			} else if (!reqAndroidUserMap.isEmpty()) {
				log.error("reqAndroidUserMap size :{} reqAndroidUserMap.values():{}, gcmReSendFlag:{}", reqAndroidUserMap.size(), reqAndroidUserMap.values().size(), isGcmReSend);
				Collection<PushQueue> androidList = reqAndroidUserMap.values();
				
				for (PushQueue pushQueue : androidList) {
					sb.setLength(0);
					id = sb.append(pushQueue.getPushId()).append("&&")
					.append(pushQueue.getDeviceId()).append("&&")
					.append(pushQueue.getReqUid()).append("&&")
					.append(pushQueue.getCustId()).toString();
					pushResultList.add(new PushResult(appInfo.getAppGrpId(), NO_SEND,
							id,
							App.ANDROID,pushQueue.getRowId()));
				}
			}
		}
		
		if (apnsRequestBuilder != null && !apnsRequestBuilder.getPayloadList().isEmpty()) {
			log.info("mgs apns public push send [payloadSize:{}]", apnsRequestBuilder.getPayloadList().size());
			mgsProducer.send(apnsRequestBuilder.build());
		}
		
		if (mgsGcmRequestBuilder != null && !mgsGcmRequestBuilder.getPayloadList().isEmpty()) {
			log.info("mgs gcm public push send [payloadSize:{}]", mgsGcmRequestBuilder.getPayloadList().size());
			mgsProducer.send(mgsGcmRequestBuilder.build());
		}

		return pushResultList;
	}

	@Override
	protected void sendWakeupGcm(PushRequest pushRequest, Map<String, PushQueue> androidMap) {
		if (useMgsPublicPush) {
			super.sendWakeupGcm(pushRequest, androidMap);
		} else {
			long startTime = System.currentTimeMillis();
			Collection<PushQueue> wakeupGcmList = androidMap.values();
			MgsPush.Request.Builder wakeupRequestBuilder = MgsPush.Request.newBuilder()
					.setAppKey(appInfo.getAppKey(App.ANDROID))
					.setAckMode(AckMode.NOACK)
					.setSendType(SendType.MULTICAST)
					.setPushChnType(PushChnType.GCM);
			
			MgsPush.PushPayload.Builder payloadBuilder = MgsPush.PushPayload.newBuilder();
			
			for (PushQueue wakeUpQueue : wakeupGcmList) {
				payloadBuilder.clear();
				wakeupRequestBuilder.addPayload(payloadBuilder
						.setId(String.valueOf(wakeUpQueue.getPushId()))
						.setToken(wakeUpQueue.getPushToken()).build());
			}
			
			if (!wakeupRequestBuilder.getPayloadList().isEmpty()) {
				log.info("mgs gcm wakeup public push send [payloadSize:{}]", wakeupRequestBuilder.getPayloadList().size());
				mgsProducer.send(wakeupRequestBuilder.build());
			}
			
			long gcmWakeUpPushElapsedTime = System.currentTimeMillis() - startTime;
			log.info("gcm wakeup push count : {} elapsedTime : {}", pushRequest.getPayloadCount(), gcmWakeUpPushElapsedTime);
		}
	}
	
	
}


