package com.humuson.tms.batch.service.impl;

import static com.humuson.tms.constrants.PushResponseConstants.CHANGED_SUCCESSFUL;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_DEVICE_QUOTA_EXCEEDED;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_INVALID_REGISTRATION;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_MESSAGE_TOO_BIG;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_MISMATCH_SENDER_ID;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_MISSING_COLLAPSE_KEY;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_MISSING_REGISTRATION;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_NOT_REGISTERED;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_QUOTA_EXCEEDED;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_UNAVAILABLE;
import static com.humuson.tms.constrants.PushResponseConstants.SUCCESSFUL;
import static com.humuson.tms.constrants.PushResponseConstants.UNKNOWN_FAIL;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.json.simple.JSONObject;

import com.google.android.gcm.server.Constants;
import com.google.android.gcm.server.Message;
import com.google.android.gcm.server.MulticastResult;
import com.google.android.gcm.server.Result;
import com.google.android.gcm.server.Sender;
import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.domain.PushMessage;
import com.humuson.tms.batch.domain.PushQueue;
import com.humuson.tms.batch.domain.PushResult;
import com.humuson.tms.batch.service.GcmHttpService;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.constrants.PushPayload;

import lombok.extern.slf4j.Slf4j;

/**
 * @author sgeom
 *
 */
@Slf4j
public class GcmHttpServiceImpl implements GcmHttpService<PushResult, PushQueue>{
	
	ConcurrentHashMap<String, Sender> gcmSenderConcurrentMap = new ConcurrentHashMap<String,Sender>();
	
	public static Map<String, String> GCM_ERROR_CODE_MAP = new ConcurrentHashMap<String, String>();
	
	static {
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_NOT_REGISTERED, ERROR_NOT_REGISTERED);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_QUOTA_EXCEEDED, ERROR_QUOTA_EXCEEDED);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_DEVICE_QUOTA_EXCEEDED, ERROR_DEVICE_QUOTA_EXCEEDED);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_MISSING_REGISTRATION, ERROR_MISSING_REGISTRATION);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_INVALID_REGISTRATION, ERROR_INVALID_REGISTRATION);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_MISMATCH_SENDER_ID, ERROR_MISMATCH_SENDER_ID);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_MESSAGE_TOO_BIG, ERROR_MESSAGE_TOO_BIG);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_MISSING_COLLAPSE_KEY, ERROR_MISSING_COLLAPSE_KEY);
	}
	
	public App appInfo;
	
	@Override
	public PushResult sendGcmMessage(PushQueue pushQueue ,String gcmApiKey) {
		
		Message gcmMessage = makeGcmMessage(pushQueue);
		
		Result result = null;
		
		try {
			result =  gcmSenderConcurrentMap.get(gcmApiKey).send(gcmMessage, pushQueue.getPushToken(), 3);
		} catch (Exception e) {
			log.error("exception in realtime send gcm : {}", e);
		}
		
		return this.getGcmResult(result, pushQueue);
	}

	@Override
	public List<PushResult> sendGcmMulticastMessage(PushQueue[] rawUser, final Message msg, String gcmApiKey) {
		
		final Message message = (msg == null)? this.makeGcmMessage(rawUser[0]) : msg;
		
		MulticastResult multicastResult = null;
		
		final List<String> partialDevices = new ArrayList<String>();
		
		for (PushQueue user : rawUser) {
			
			if (StringUtils.isNull(user.getPushToken())) {
				log.error("GCM empty token is {}", user.getDeviceId());
			} else {
				partialDevices.add(user.getPushToken());
			}
		}
		
		try {
			
			multicastResult =  gcmSenderConcurrentMap.get(gcmApiKey).send(message, partialDevices, 1);
			
		} catch (Exception e) {
			log.error("multicate send gcm error", e);
			multicastResult =  null;
		}
		
		List<PushResult> resultList = new ArrayList<PushResult>();
		
		if (multicastResult != null) {
			
			List<Result> results = multicastResult.getResults();
			
			int size = results.size();
			
			for (int i=0; i<size; i++) {
				resultList.add(getGcmResult(results.get(i), rawUser[i]));
			}
			
		} else {
			StringBuilder sb = new StringBuilder();
			for (PushQueue user : rawUser) {
				sb.setLength(0);
				sb.append(user.getPushId()).append("&&")
						.append(user.getDeviceId()).append("&&")
						.append(user.getReqUid()).append("&&")
						.append(user.getCustId()).toString();
				resultList.add(new PushResult(appInfo.getAppGrpId(), UNKNOWN_FAIL, 
						sb.toString(),
						App.ANDROID, user.getRowId()));
			}
		}
		
		return resultList;
	}

	@Override
	public PushResult getGcmResult(Result result, PushQueue user) {
		
		String changedToken = null;
		
		String resultCode = SUCCESSFUL;
		
		StringBuilder sb = new StringBuilder();
		String id = sb.append(user.getPushId()).append("&&")
				.append(user.getDeviceId()).append("&&")
				.append(user.getReqUid()).append("&&")
				.append(user.getCustId()).toString();
		
		if (result == null) {
			return new PushResult(appInfo.getAppGrpId(), ERROR_UNAVAILABLE, 
					id, 
					App.ANDROID, user.getRowId()).setChangedToken(changedToken);
		}
		
		String messageId = result.getMessageId();
		
		
		if (messageId != null) {
			
			if (result.getCanonicalRegistrationId() != null) {
				// same device has more than on registration id: update it
				changedToken = result.getCanonicalRegistrationId();
				resultCode = CHANGED_SUCCESSFUL;
			}
			
		} else {
			
			String error = result.getErrorCodeName();
			if (GCM_ERROR_CODE_MAP.containsKey(result.getErrorCodeName())) {
				resultCode = GCM_ERROR_CODE_MAP.get(error);
			} else {
				resultCode = ERROR_UNAVAILABLE;
				log.error("gcm response is unavaliable : {}", result.toString());
			}
		}
		
		return new PushResult(appInfo.getAppGrpId(), resultCode, id, App.ANDROID,user.getRowId()).setChangedToken(changedToken);
	}

	@Override
	public void init(App appInfo) {
		this.appInfo = appInfo;
		this.appInfo = appInfo;
		if(gcmSenderConcurrentMap.get(appInfo.getGcmApiKey()) == null){
			Sender sender = new Sender(appInfo.getGcmApiKey());
			gcmSenderConcurrentMap.put(appInfo.getGcmApiKey(), sender); 
		}
		
	}
 
	@Override
	public Message makeGcmMessage(PushQueue que) {
		
		PushMessage pushMessage = que.getPushMessage();
		Message.Builder builder = new Message.Builder();
		builder.addData(PushPayload.PUSH_NOTI_MSG, pushMessage.getPushMsg());
		builder.addData(PushPayload.PUSH_NOTI_TITLE, StringUtils.validString(pushMessage.getPushTitle()));
		String pushImage = pushMessage.getPushImg();
		if (pushImage != null 
				&& (!pushImage.startsWith("http://"))
				&& "http://".length() >= pushImage.length()) {
			pushImage = "";
		}
		
		builder.addData(PushPayload.PUSH_NOTI_IMG, pushImage == null ? "": pushImage);
		builder.addData(PushPayload.PUSH_RICH_CONTENT, pushMessage.getPopupContent());
		builder.addData(PushPayload.MSG_TYPE, pushMessage.getMsgType());
		builder.addData(PushPayload.MSG_ID, String.valueOf(pushMessage.getMsgUid()));
		
		JSONObject json = new JSONObject();
		
		if (!StringUtils.isNull(pushMessage.getPushKey())) {
			json.put(pushMessage.getPushKey(), StringUtils.validString(pushMessage.getPushValue()));
		}
		
		builder.addData(PushPayload.PUSH_DATA, json.toString());
		builder.delayWhileIdle(false);
		
		log.info("gcm message [{}]", builder.build().toString());
		return builder.build();
	}

	@Override
	public List<PushResult> sendGcmOne2OneList(Collection<PushQueue> pushQueueCollection, String gcmApiKey) {
		
		List<PushResult> resList = new ArrayList<PushResult>();
		
		Sender sender = gcmSenderConcurrentMap.get(gcmApiKey);
		
		for( PushQueue pushQueue : pushQueueCollection){
		
			Message gcmMessage = makeGcmMessage(pushQueue);
			
			Result result = null;
			
			try {
				result =  sender.send(gcmMessage, pushQueue.getPushToken(), 3);
			} catch (Exception e) {
				log.error("exception in realtime send gcm : {}", e);
			}
			
			resList.add(this.getGcmResult(result, pushQueue));
		
		}
		return resList;
	}

	@Override
	public void setServerId(String serverId) {
		// MQ발송시 사용하기 위한 메소드
	}
}
