package com.humuson.tms.batch.service.impl;

import static com.humuson.tms.constrants.PushResponseConstants.ERROR_DEVICE_QUOTA_EXCEEDED;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_INVALID_REGISTRATION;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_MESSAGE_TOO_BIG;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_MISMATCH_SENDER_ID;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_MISSING_COLLAPSE_KEY;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_MISSING_REGISTRATION;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_NOT_REGISTERED;
import static com.humuson.tms.constrants.PushResponseConstants.ERROR_QUOTA_EXCEEDED;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.json.simple.JSONObject;

import com.google.android.gcm.server.Constants;
import com.google.android.gcm.server.Message;
import com.google.android.gcm.server.MulticastResult;
import com.google.android.gcm.server.Result;
import com.google.android.gcm.server.Sender;
import com.humuson.tms.batch.domain.App;
import com.humuson.tms.batch.service.GcmHttpService;
import com.humuson.tms.common.util.StringUtils;
import com.humuson.tms.mq.model.MgsPush;
import com.humuson.tms.mq.model.MgsPush.GcmMessage;
import com.humuson.tms.mq.model.MgsPush.PushPayload;
import com.humuson.tms.mq.model.MgsPush.Response;
import com.humuson.tms.mq.model.MgsPush.Response.ResponsePayload;

import lombok.extern.slf4j.Slf4j;

/**
 * @author sgeom
 *
 */
@Slf4j
public class GcmHttpMqServiceImpl implements GcmHttpService<ResponsePayload, MgsPush.PushPayload>{
	
	ConcurrentHashMap<String, Sender> gcmSenderConcurrentMap = new ConcurrentHashMap<String,Sender>();
	
	public static Map<String, String> GCM_ERROR_CODE_MAP = new ConcurrentHashMap<String, String>();
	
	String serverId;
	
	static {
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_NOT_REGISTERED, ERROR_NOT_REGISTERED);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_QUOTA_EXCEEDED, ERROR_QUOTA_EXCEEDED);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_DEVICE_QUOTA_EXCEEDED, ERROR_DEVICE_QUOTA_EXCEEDED);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_MISSING_REGISTRATION, ERROR_MISSING_REGISTRATION);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_INVALID_REGISTRATION, ERROR_INVALID_REGISTRATION);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_MISMATCH_SENDER_ID, ERROR_MISMATCH_SENDER_ID);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_MESSAGE_TOO_BIG, ERROR_MESSAGE_TOO_BIG);
		GCM_ERROR_CODE_MAP.put(Constants.ERROR_MISSING_COLLAPSE_KEY, ERROR_MISSING_COLLAPSE_KEY);
	}
	
//	public App appInfo;
	
	/* (non-Javadoc)
	 * @see com.humuson.tms.batch.service.GcmHttpService#sendGcmMessage(java.lang.Object, java.lang.String)
	 */
	@Override
	public MgsPush.Response.ResponsePayload sendGcmMessage(MgsPush.PushPayload payLoad, String gcmApiKey) {
		
		Message gcmMessage = makeGcmMessage(payLoad);
		
		Result result = null;
		try {
			result  = gcmSenderConcurrentMap.get(gcmApiKey).send(gcmMessage, payLoad.getToken(), 3);
		} catch (Exception e) {
			log.error("exception in realtime send gcm : {}", e);
		}
		
		return this.getGcmResult(result, payLoad);
		
	}

	@Override
	public List<ResponsePayload> sendGcmMulticastMessage(MgsPush.PushPayload[] rawUser, Message msg, String gcmApiKey) {
		
		final Message message = (msg == null)? this.makeGcmMessage(rawUser[0]) : msg;
		
		MulticastResult multicastResult = null;
		
		ResponsePayload.Builder tempPayloadBuilder = ResponsePayload.newBuilder();
		
		final List<String> partialDevices = new ArrayList<String>();
		
		for (MgsPush.PushPayload payLoad : rawUser) {
			
			if (StringUtils.isNull(payLoad.getToken())) {
				log.error("GCM empty token is {}", payLoad.getId());
			} else {
				partialDevices.add(payLoad.getToken());
			}
		}
		
		try {
			multicastResult =  gcmSenderConcurrentMap.get(gcmApiKey).send(message, partialDevices, 1);
			
		} catch (Exception e) {
			log.error("multicate send gcm error", e);
			multicastResult =  null;
		}
		
		List<ResponsePayload> resultList = new ArrayList<ResponsePayload>();
		
		if (multicastResult != null) {
			
			List<Result> results = multicastResult.getResults();
			
			int size = results.size();
			
			for (int i=0; i<size; i++) {
				resultList.add(getGcmResult(results.get(i), rawUser[i]));
			}
			
		} else {
			StringBuilder sb = new StringBuilder();
			for (MgsPush.PushPayload user : rawUser) {
				tempPayloadBuilder.clear();
				tempPayloadBuilder.setId(user.getId());
				tempPayloadBuilder.setReturnCode(Response.ReturnCode.UNKNOWN_FAIL);
				resultList.add(tempPayloadBuilder.build());
			}
		}
		
		
		
		return resultList;
	}

	@Override
	public MgsPush.Response.ResponsePayload getGcmResult(Result result, MgsPush.PushPayload payLoad) {
		
		ResponsePayload.Builder  responsePayloadBuilder = ResponsePayload.newBuilder();
		
		String changedToken = null;
		
		responsePayloadBuilder.setId(payLoad.getId());
		responsePayloadBuilder.setServerId(serverId);
		responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.SUCCESSFUL);
		
		if (result == null) {
			responsePayloadBuilder.clear();
			responsePayloadBuilder.setId(payLoad.getId());
			responsePayloadBuilder.setServerId(serverId);
			responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.ERROR_UNAVAILABLE);
			return responsePayloadBuilder.build();
		}
		
		String messageId = result.getMessageId();
		
		
		if (messageId != null) {
			
			if (result.getCanonicalRegistrationId() != null) {
				// same device has more than on registration id: update it
				changedToken = result.getCanonicalRegistrationId();
				responsePayloadBuilder.clear();
				responsePayloadBuilder.setId(payLoad.getId());
				responsePayloadBuilder.setServerId(serverId);
				responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.CHANGED_SUCCESSFUL);
			}
			
		} else {
			
			String error = result.getErrorCodeName();
			if (GCM_ERROR_CODE_MAP.containsKey(result.getErrorCodeName())) {
				String resultCode = GCM_ERROR_CODE_MAP.get(error);
				responsePayloadBuilder.clear();
				responsePayloadBuilder.setId(payLoad.getId());
				responsePayloadBuilder.setServerId(serverId);
				responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.valueOf(Integer.parseInt(resultCode)));
			} else {
				responsePayloadBuilder.clear();
				responsePayloadBuilder.setId(payLoad.getId());
				responsePayloadBuilder.setServerId(serverId);
				responsePayloadBuilder.setReturnCode(MgsPush.Response.ReturnCode.ERROR_UNAVAILABLE);
				log.error("gcm response is unavaliable : {}", result.toString());
			}
		}
		
		return responsePayloadBuilder.build();
	}

	@Override
	public void init(App appInfo) {
//		this.appInfo = appInfo;
		if(gcmSenderConcurrentMap.get(appInfo.getGcmApiKey()) == null){
			Sender sender = new Sender(appInfo.getGcmApiKey());
			gcmSenderConcurrentMap.put(appInfo.getGcmApiKey(), sender); 
		}
	}
 
	@Override
	public Message makeGcmMessage(MgsPush.PushPayload payload) {
		GcmMessage gcmMessage = payload.getGcmMessage();
		Message.Builder builder = new Message.Builder();
		builder.addData(com.humuson.tms.constrants.PushPayload.PUSH_NOTI_MSG, gcmMessage.getPushNotiMsg());
		builder.addData(com.humuson.tms.constrants.PushPayload.PUSH_NOTI_TITLE, StringUtils.validString(gcmMessage.getPushNotiTitle()));
		String pushImage = gcmMessage.getPushNotiImg();
		if (pushImage != null 
				&& (!pushImage.startsWith("http://"))
				&& "http://".length() >= pushImage.length()) {
			pushImage = "";
		}
		
		builder.addData(com.humuson.tms.constrants.PushPayload.PUSH_NOTI_IMG, pushImage == null ? "": pushImage);
		builder.addData(com.humuson.tms.constrants.PushPayload.PUSH_RICH_CONTENT, gcmMessage.getPushRichContent());
		builder.addData(com.humuson.tms.constrants.PushPayload.MSG_TYPE, gcmMessage.getMsgType());
		builder.addData(com.humuson.tms.constrants.PushPayload.MSG_ID, String.valueOf(gcmMessage.getMsgId()));
		
		JSONObject json = new JSONObject();
		
//		if (!StringUtils.isNull(pushMessage.getPushKey())) {
//			json.put(pushMessage.getPushKey(), StringUtils.validString(pushMessage.getPushValue()));
//		}
		
		builder.addData(com.humuson.tms.constrants.PushPayload.PUSH_DATA, gcmMessage.getPushData());
		builder.delayWhileIdle(false);
		
		//log.info("gcm message [{}]", builder.build().toString());
		return builder.build();
	}

	@Override
	public List<MgsPush.Response.ResponsePayload> sendGcmOne2OneList(Collection<PushPayload> payloadList, String gcmApiKey) {
		
		List<MgsPush.Response.ResponsePayload> resList = new ArrayList<MgsPush.Response.ResponsePayload>();
		
		
		Sender sender = gcmSenderConcurrentMap.get(gcmApiKey);
		
		for(PushPayload payload : payloadList ){
		
			Message gcmMessage = makeGcmMessage(payload);
			
			Result result = null;
			try {
				result  = sender.send(gcmMessage, payload.getToken(), 3);
				if (log.isDebugEnabled()) {
					log.debug("result : {}, payload:{}", result.getMessageId(), payload.getToken());
				}
				
			} catch (Exception e) {
				log.error("exception in realtime send gcm : {}", e);
			}
			
			resList.add(this.getGcmResult(result, payload));
			
		}
		
		return resList;
	}
	
	@Override
	public void setServerId(String serverId) {
		// MQ발송시 사용하기 위한 메소드
		this.serverId=serverId;
	}
}
